/********************************************************************** 
 * Copyright (c) 2008, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: DeleteRefactoring.java,v 1.15 2010/03/29 17:33:55 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyNodeCache;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.CompositeReferencerProxyNode;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceTypeRegistry;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.internal.util.TestUIUtilities;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IReferencerProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeDeleter;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeUpdater;
import org.eclipse.hyades.test.ui.navigator.refactoring.DeleteRefactoringEvent;
import org.eclipse.hyades.test.ui.navigator.refactoring.RefactoringEvent;
import org.eclipse.hyades.test.ui.navigator.refactoring.UpdateRefactoringEvent;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.Refactoring;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Delete refactoring.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul E. Slauenwhite
 * @version March 29, 2010
 * @since   March 25, 2008
 */
public class DeleteRefactoring extends Refactoring {
	private List containers; 
	private List proxies; 
	private List targetProxies; 
	private List seenElements;
	private Change change;
	private RefactoringContext context;
	private boolean deepRemove;
	private boolean update; 
	private Collection nodesToUpdateLater;
	private boolean projectRemove;
	private boolean projectClosed;
	private boolean physicalDelete;
	private HashMap associationMap; // map associating proxy change and their container, used to enable/disable container delete if proxy delete is selected or not

	public HashMap getAssociationMap() {
		return associationMap;
	}
	
	private class NodeToUpdateLater {
		private CompositeChange composite;
		private String refType;
		private CompositeReferencerProxyNode node;
		
		public NodeToUpdateLater(CompositeChange composite, String refType, CompositeReferencerProxyNode node) {
			this.composite = composite;
			this.refType = refType;
			this.node = node;
		}

		public CompositeChange getComposite() {
			return composite;
		}

		public CompositeReferencerProxyNode getNode() {
			return node;
		}

		public String getRefType() {
			return refType;
		}
	}
	
	public boolean isProjectRemove() {
		return projectRemove;
	}
	
	public boolean isProjectClosed() {
		return projectClosed;
	}
	
	public List getContainers() {
		return containers;
	}

	public List getProxies() {
		return proxies;
	}

	public DeleteRefactoring(List containers, List proxies) {
		this.containers = containers;
		this.proxies = proxies;
		this.context = new RefactoringContext();
		this.deepRemove = false;
		this.projectRemove = false;
		this.projectClosed = false;
		for (Iterator it = containers.iterator(); (it.hasNext()) && (!this.projectRemove); ) {
			IContainer cont = (IContainer)it.next();
			if (cont instanceof IProject) {
				this.projectRemove = true;
				if (!cont.isAccessible()) {
					this.projectClosed = true;
				}
			}
		}
		this.physicalDelete = false;
		initializeChangeComputation();
	}
	
	private void initializeChangeComputation() {
		this.change = null; // else all explodes if "back" => "preview" (reuse of the first change instead of re-calculate it)
		this.targetProxies = new ArrayList();
		this.seenElements = new LinkedList();
		this.nodesToUpdateLater = new HashSet();		
		this.associationMap = new HashMap();
		DeleteFileChange.setDeletetefactoring(this);
		DeleteContainerChange.setDeletetefactoring(this);
	}
	
	public RefactoringStatus checkFinalConditions(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		pm.beginTask("", 3); //$NON-NLS-1$
		initializeChangeComputation();
		try {
			try {
				TestUIUtilities.createProxyNodes(ResourcesPlugin.getWorkspace().getRoot(), new SubProgressMonitor(pm, 1));
			} catch (OperationCanceledException e) {
				return RefactoringStatus.createFatalErrorStatus(RefactoringMessages.ReorgRefactoring_CANCELED_OPERATION_MSG);
			}			
			change = createChange(new SubProgressMonitor(pm, 1));
			RefactoringStatus valid = change.isValid(new SubProgressMonitor(pm, 1));
			if(valid.hasError()) { // bugzilla 262595 : in case of closed project, a warning status is raised, but we have to continue the delete if user ask for it
				change = null;
			}
			return valid;
		} finally {
			pm.done();
		}
	}

	public RefactoringStatus checkInitialConditions(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		pm.beginTask("", 1); //$NON-NLS-1$
		try {
			if (containers != null) {
				boolean oneProject = false;
				boolean oneNotProject = ((proxies != null) && (!proxies.isEmpty()));
				for (Iterator it = containers.iterator(); (it.hasNext()) && (!(oneProject && oneNotProject));) {
					IContainer cont = (IContainer)it.next();
					if (cont instanceof IProject) {
						oneProject = true;
					} else {
						oneNotProject = true;
					}
				}
				if (oneProject && oneNotProject) {
					return RefactoringStatus.createFatalErrorStatus(RefactoringMessages.DeleteResources_delete_error_mixed_types);
				}
			}
			return new RefactoringStatus();
		} finally {
			pm.done();
		}
	}
	
	public Change createChange(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		if (change != null) {
			return change;
		}
		pm.beginTask("", containers.size()+proxies.size()+1); //$NON-NLS-1$
		try {
			searchAllResourceToDelete(targetProxies);
			RefactoringTransactionRootChange c = new RefactoringTransactionRootChange(context);
			TrashBox trash = null;
			try {
				trash = TrashBox.instance();
			} catch (Exception e) {
			}		
			c.add(trash.createRefreshOnUndoChange());
			CustomRefactoringManager.addAllStartingChange(c,ICustomRefactoring.DELETE);
			List proxyReferencers = new LinkedList();
			for (Iterator it = targetProxies.iterator(); it.hasNext();) {
				proxyReferencers.add(new CompositeReferencerProxyNode((IProxyNode) it.next()));
			}	
			for (Iterator it = proxyReferencers.iterator(); it.hasNext();) {
				CompositeReferencerProxyNode ref = (CompositeReferencerProxyNode) it.next();
				if (!seenElements.contains(ref)) {
					Change refChange = createDeleteReferencerChange(ref);
					if (refChange != null) {
						c.add(refChange);						
					}
				}	
				if (proxies.contains(ref.getProxy())) {
					pm.worked(1);					
				}
			}				
			for (Iterator it = containers.iterator(); it.hasNext();) {
				IContainer elem = (IContainer)(it.next());
				Change containerChange = createDeleteContainerChange(elem);
				if (containerChange != null) {
					c.add(containerChange);
				}
				if (containers.contains(elem)) {
					pm.worked(1);					
				}
			}
			processNodesToUpdateLater();
			CustomRefactoringManager.addAllEndingChange(c,ICustomRefactoring.DELETE);
			c.markAsSynthetic();
			if (trash != null) {
				c.add(trash.createCleanTrashChange());
			}			
			return c;
		} finally {
			pm.done();
		}
	}
	
	private void searchAllResourceToDelete(List proxyToDelete) {
		for (Iterator it = proxies.iterator(); it.hasNext();) {
			Object elem = it.next();
			if (!proxyToDelete.contains(elem)) {
				proxyToDelete.add(elem);
			}
		}
		for (Iterator it = containers.iterator(); it.hasNext();) {
			recursiveSearchAllResourceToDelete((IContainer)(it.next()),proxyToDelete);
		}		
	}
	
	
	private void recursiveSearchAllResourceToDelete(IResource resource,List proxyToDelete) {
		if (resource == null) {
			return;
		}
		if ((!resource.exists()) || (!resource.isAccessible())) {
			return;
		}
		if (resource instanceof IContainer) {
			IResource childs[];
			try {
				childs = ((IContainer)resource).members();
			} catch (CoreException e) {
				// should not happend thanks to previous check
				return;
			}
			for (int i=0; i<childs.length; i++) {
				recursiveSearchAllResourceToDelete(childs[i], proxyToDelete);				
			}
			return;
		}
		if (resource instanceof IFile) {
			IFile file = (IFile)resource;
			IProxyNode proxy = FileProxyNodeCache.getInstance().getProxy(file);
			if (proxy != null) {
				if (!proxyToDelete.contains(proxy)) {
					proxyToDelete.add(proxy);
				}
			} 
		}
	}
	
	/**
	 * Adds the given change as the first children of the given composite change.
	 * @param composite
	 * @param c
	 */
	private static void insertFirst(CompositeChange composite, Change c) {
		composite.add(c);
		Change[] children = composite.getChildren();
		for (int i = 0; i < children.length-1; i++) {
			composite.remove(children[i]);
			composite.add(children[i]);
		}
	}
	
	/**
	 * Completes delete proxy composite change by adding update changes only 
	 * for cases where the element to update still exists
	 */
	private void processNodesToUpdateLater() {
		for (Iterator it = nodesToUpdateLater.iterator(); it.hasNext();) {
			NodeToUpdateLater element = (NodeToUpdateLater) it.next();
			Change updateTypeChange = createUpdateReferenceTypeChange(element.getRefType(), element.getNode());
			if (updateTypeChange != null) {
				updateTypeChange.setEnabled(update);
				insertFirst(element.getComposite(), updateTypeChange);
			}			
		}
	}

	private Change createDeleteReferencerChange(CompositeReferencerProxyNode node) {
		IProxyNodeDeleter deleter = (IProxyNodeDeleter) node.getAdapter(IProxyNodeDeleter.class);
		if(deleter != null) {
			CompositeChange composite = new CompositeChange(NLS.bind(RefactoringMessages.DELETE_PROXY, node.getText()));
			seenElements.add(node);
			//- we need to add as children all referenced elements from ref type with owns flag set to true.
			for (Iterator it = node.getReferenceTypes().iterator(); it.hasNext();) {
				String refType = (String) it.next();
				String oppType = ReferenceTypeRegistry.getInstance().getOppositeReferenceType(refType);
				if (ReferenceTypeRegistry.getInstance().owns(refType)) {
					Change referenceTypeChange = createDeleteReferenceTypeChange(refType, node);
					if(referenceTypeChange != null) {
						referenceTypeChange.setEnabled(deepRemove);
						composite.add(referenceTypeChange);
						continue;
					}
				}
				//- opposite explicit references need to be updated according to the deletion of node
				if (ReferenceTypeRegistry.getInstance().isExplicit(oppType)) {
					nodesToUpdateLater.add(new NodeToUpdateLater(composite, refType, node));
				}
			}			
			RefactoringEvent event = new DeleteRefactoringEvent(node);
			CompositeListenerChange compositeListener = new CompositeListenerChange(event);	
			Change deleteChange = deleter.createDeleteChange(context);
			deleteChange.setEnabled(physicalDelete);
			compositeListener.add(deleteChange);
			composite.add(compositeListener);
			return composite;
		}
		return null;
	}

	private Change createUpdateReferenceTypeChange(String refType, CompositeReferencerProxyNode node) {
		CompositeChange composite = new CompositeChange(NLS.bind(RefactoringMessages.UPDATE_PROXY, ReferenceTypeRegistry.getInstance().getName(refType)));
		Collection references = node.getReferences(refType);
		for (Iterator it = references.iterator(); it.hasNext();) {
			IProxyNode proxy = (IProxyNode) it.next();
			if (!seenElements.contains(new CompositeReferencerProxyNode(proxy))) {
				IProxyNodeUpdater updater = (IProxyNodeUpdater) proxy.getAdapter(IProxyNodeUpdater.class);
				if (updater != null) {
					RefactoringEvent event = new UpdateRefactoringEvent(proxy,node,refType,null);
					CompositeListenerChange compositeListener = new CompositeListenerChange(event);	
					compositeListener.add(updater.createUpdateChange(context, node, refType, null));
					composite.add(compositeListener);
				}
			}						
		}		
		return composite.getChildren().length > 0 ? composite : null;
	}
	
	private Change createDeleteReferenceTypeChange(String refType, CompositeReferencerProxyNode node) {
		CompositeChange composite = new CompositeChange(NLS.bind(RefactoringMessages.DELETE_PROXY, ReferenceTypeRegistry.getInstance().getName(refType)));
		Collection references = node.getReferences(refType);
		for (Iterator it = references.iterator(); it.hasNext();) {
			IProxyNode proxy = (IProxyNode) it.next();
			
			//Note: If a deep remove, only delete referenced proxies that not target proxies:
			if ((proxy.getAdapter(IFile.class) != null) && (!targetProxies.contains(proxy))) {

				//- if the referenced proxy is a referencer itself we need to get its references as well.
				if (proxy instanceof IReferencerProxyNode) {
					CompositeReferencerProxyNode referenced = new CompositeReferencerProxyNode((IReferencerProxyNode) proxy);
					//- we need to check that the referenced element was not previously processed.
					//- this check is needed because of bidirectional/circular references.
					if (!seenElements.contains(referenced)) {						
						Change refChange = createDeleteReferencerChange(referenced);
						if (refChange != null) {
							composite.add(refChange);
						}						
					}
				} else {
					// - the referenced element is not a referencer
					if (!seenElements.contains(proxy)) {
						RefactoringEvent event = new DeleteRefactoringEvent(proxy);
						CompositeListenerChange compositeListener = new CompositeListenerChange(event);	
						compositeListener.add(new DeleteFileChange((IFile) proxy.getUnderlyingResource()));
						composite.add(compositeListener);
					}
				}
			}			
		}		
		return composite.getChildren().length > 0 ? composite : null;
	}

	private Change createDeleteContainerChange(IContainer container) {
		RefactoringEvent event = new DeleteRefactoringEvent(container);
		CompositeListenerChange composite = new CompositeListenerChange(event);
		String baseDisplayName;
		if (container instanceof IProject) {
			baseDisplayName = RefactoringMessages.DELETE_PROJECT;
		} else {
			baseDisplayName = RefactoringMessages.DELETE_FOLDER;
		}
		DeleteContainerCompositeChange contChange = new DeleteContainerCompositeChange(container,physicalDelete,baseDisplayName,(List)associationMap.get(container));
		if(container.isAccessible()) {
			IResource[] resources;
			try {
				resources = container.members();
			} catch (CoreException e) {
				UiPlugin.logWarning("Unable to get members of container: "+container.getFullPath().toOSString(), e); //$NON-NLS-1$
				return null;
			}
			for (int i = 0; i < resources.length; i++) {
				if (!resources[i].isDerived()) {
					if (resources[i] instanceof IFile) {
						IFile file = (IFile)resources[i];
						IProxyNode proxy = FileProxyNodeCache.getInstance().getProxy(file);
						if (proxy == null) {
							//- plain file that are not displayed by the navigator,
							//- those files should be shown in the refactoring since they are potentially deleted						
							RefactoringEvent subevent = new DeleteRefactoringEvent(file);
							CompositeListenerChange compositeListener = new CompositeListenerChange(subevent);
							Change deleteChange = new DeleteFileChange(file);
							deleteChange.setEnabled(physicalDelete);
							compositeListener.add(deleteChange);
							contChange.add(compositeListener);
						}
					} else {
						contChange.add(createDeleteContainerChange((IContainer)resources[i]));
					}
				}
			}
		}
		if (container instanceof IProject) {
			contChange.add(new DeleteLogicalProjectChange((IProject)container));
		}
		if (contChange.getChildren().length > 0) {
			composite.add(contChange);
		} else {
			composite.add(contChange.getDeleteContainerChange());
		}
		return composite;
	}

	public String getName() {
		return "";  //$NON-NLS-1$
	}

	public void setDeepRemove(boolean deepRemove) {
		this.deepRemove = deepRemove;
	}

	public boolean getDeepRemove() {
		return deepRemove;
	}
	
	public void setUpdate(boolean updateValue) {
		this.update = updateValue;
	}
	
	public boolean getUpdate() {
		return update;
	}
	
	public void setPhysicalDelete(boolean physicalDelete) {
		this.physicalDelete = physicalDelete;
	}
	
	public boolean getPhysicalDelete() {
		return physicalDelete;
	}
}
