/********************************************************************** 
 * Copyright (c) 2008, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: MoveModelChange.java,v 1.7 2010/04/12 12:38:49 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.util.Date;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.navigator.EObjectProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IRefactoringContext;
import org.eclipse.hyades.test.ui.navigator.refactoring.EMFRefactoringTransaction;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Move model change.</p>
 *
 *
 * @author  Jerome Bozier
 * @author  Jerome Gout
 * @author  Paul Slauenwhite
 * @version April 12, 2010
 * @since   March 20, 2008
 */
public class MoveModelChange extends Change {
	private EMFRefactoringTransaction context;
	protected IPath destination;
	protected EObjectProxyNode proxy;
	private boolean shouldBeRenamed;
	private RefactoringStatus status;
	private IRefactoringContext globalContext;

	public MoveModelChange(EObjectProxyNode node, IRefactoringContext context, IPath destinationPath) {
		this.proxy = node;
		this.globalContext = context;
		this.destination = destinationPath;
		this.context = (EMFRefactoringTransaction) context.getRefactoringTransaction(EMFRefactoringTransaction.EMF_REFACTORING_TRANSACTION_ID);
		Assert.isNotNull(this.context);
		this.status = new RefactoringStatus();
		IFile destFile = ResourcesPlugin.getWorkspace().getRoot().getFile(destination);
		if (destFile.exists()) {
			handleFileAlreadyExists(this.status);
		}
	}

	public Object getModifiedElement() {
		return proxy;
	}

	public String getName() {
		String fileName = proxy.getUnderlyingResource().getFullPath().toString();
		return NLS.bind(RefactoringMessages.MOVE_FILE, fileName, destination.removeLastSegments(1));
	}

	public void initializeValidationData(IProgressMonitor pm) {
	}

	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		return this.status;
	}

	protected void handleFileAlreadyExists(RefactoringStatus status) {
		status.addInfo(NLS.bind(RefactoringMessages.ALREADY_EXISTS_PROXY_DURING_MOVE, destination.toPortableString()));
		shouldBeRenamed = true;
	}
	
	public Change perform(IProgressMonitor pm) throws CoreException {
		Change undo = null;
		//- load the referencer
		Resource res = context.getResourceSet().getResource(((EObjectProxyNode)proxy).getOriginatorURI().trimFragment(), true);
		context.addSavedResource(res);
		context.addChangedResourceURI(res, getNewURI());		
		IPath originalPath = new Path(proxy.getOriginatorURI().devicePath());
		undo = new MoveModelChange(proxy, globalContext, originalPath);
		return undo;
	}

	public URI getNewURI() {
		return URI.createPlatformResourceURI(getNewName(), false);
	}

	private String getNewName() {
		if(shouldBeRenamed) {
			String fullFileName = destination.lastSegment();
			int dotIndex = fullFileName.lastIndexOf('.');
			long number = new Date().getTime();
			String newName = fullFileName.substring(0, dotIndex-1)+number+fullFileName.substring(dotIndex);
			return destination.removeLastSegments(1).append(newName).toPortableString();
		} else {
			return destination.toPortableString();
		}
	}	
}
