/********************************************************************** 
 * Copyright (c) 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: MoveRefactoring.java,v 1.4 2009/11/27 14:44:22 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.CompositeReferencerProxyNode;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceTypeRegistry;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IReferencerProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeMover;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeUpdater;
import org.eclipse.hyades.test.ui.navigator.refactoring.MoveRefactoringEvent;
import org.eclipse.hyades.test.ui.navigator.refactoring.RefactoringEvent;
import org.eclipse.hyades.test.ui.navigator.refactoring.UpdateRefactoringEvent;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.NullChange;
import org.eclipse.osgi.util.NLS;

/**
 * @author Jerome Gout
 * @author Jerome Bozier
 * @version November 25, 2009
 */
public class MoveRefactoring extends ReorgRefactoring {
	private List folders; 
	private List proxies; 
	private IContainer destination;
	
	public MoveRefactoring(List folders, List proxies) {
		super();
		this.folders = folders;
		this.proxies = proxies;
	}

	public List getContainers() {
		return folders;
	}

	public List getProxies() {
		return proxies;
	}
	
	public Change createChange(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		if (change != null) {
			return change;
		}
		pm.beginTask("", folders.size() + proxies.size()); //$NON-NLS-1$
		try {
			RefactoringTransactionRootChange c = new RefactoringTransactionRootChange(context);
			CustomRefactoringManager.addAllStartingChange(c,ICustomRefactoring.MOVE);
			for (Iterator it = folders.iterator(); it.hasNext();) {
				IFolder folder = (IFolder)it.next();
				Change containerChange = createMoveContainerChange(folder, destination);
				if (containerChange != null) {
					c.add(containerChange);
				}
				pm.worked(1);
			}
			for (Iterator it = proxies.iterator(); it.hasNext();) {
				IProxyNode proxy = (IProxyNode) it.next();
				Change refChange = createMoveProxyChange(proxy, destination);
				if (refChange != null) {					
					c.add(refChange);
				}
				pm.worked(1);
			}
			CustomRefactoringManager.addAllEndingChange(c,ICustomRefactoring.MOVE);
			c.markAsSynthetic();
			return c;
		} finally {
			pm.done();
		}
	}

	public String getName() {
		return "";  //$NON-NLS-1$
	}
	
	public void setDestination(IContainer container) {
		this.destination = container;
		change = null;
		seenElements.clear();
	}

	protected Change createMoveProxyChange(IProxyNode node, IContainer dest) {
		if (!seenElements.contains(node)) {
			seenElements.add(node);
			if (node instanceof IReferencerProxyNode) {
				return createMoveReferencerChange(node, dest);
			} else {
				return createShallowMoveProxyChange(node, dest);
			}
		}		
		return null;
	}

	private Change createShallowMoveProxyChange(IProxyNode node, IContainer dest) {
		IFile file = (IFile)node.getAdapter(IFile.class);
		IProxyNodeMover mover = (IProxyNodeMover)node.getAdapter(IProxyNodeMover.class);		
		Change mainChange;
		if (mover != null) {
			String fileName;
			if (file == null) {
				fileName = node.getUnderlyingResource().getName();
			} else {
				fileName = file.getName();
			}
			mainChange = mover.createMoveChange(context, dest.getFullPath().append(fileName));
		} else {
			if(file != null) {
				mainChange = new MoveFileChange(file, dest);
			} else {
				return new NullChange(NLS.bind(RefactoringMessages.UPDATE_PROXY, node.getUnderlyingResource().getFullPath().toPortableString()));
			}
		}
		RefactoringEvent event = new MoveRefactoringEvent(node,destination);
		CompositeListenerChange compositeListener = new CompositeListenerChange(event);
		compositeListener.add(mainChange);
		return compositeListener;
	}

	private Change createMoveReferencerChange(final IProxyNode node, final IContainer dest) {
		CompositeChange composite = new CompositeChange(NLS.bind(RefactoringMessages.MOVE_PROXY, node.getText(), dest.getName()));
		Change moveNode = createShallowMoveProxyChange(node, dest);
		CompositeReferencerProxyNode ref = new CompositeReferencerProxyNode(node);
		for (Iterator it = ref.getReferenceTypes().iterator(); it.hasNext();) {
			String type = (String)it.next();
			String oppType = ReferenceTypeRegistry.getInstance().getOppositeReferenceType(type);
			if(ReferenceTypeRegistry.getInstance().isExplicit(oppType)) {
				composite.add(createUpdateReferenceTypeChange(ref, type, dest));
			}
		}
		if(moveNode != null) {
			composite.add(moveNode);
		}
		return composite;
	}

	private Change createUpdateReferenceTypeChange(CompositeReferencerProxyNode ref, String type, IContainer dest) {
		CompositeChange change = new CompositeChange(ReferenceTypeRegistry.getInstance().getName(type));
		for (Iterator it = ref.getReferences(type).iterator(); it.hasNext();) {
			IProxyNode proxy = (IProxyNode)it.next();
			IProxyNodeUpdater updater = (IProxyNodeUpdater) proxy.getAdapter(IProxyNodeUpdater.class);
			if (updater != null) {
				IPath destPath = dest.getFullPath().append(ref.getUnderlyingResource().getName());				
				RefactoringEvent event = new UpdateRefactoringEvent(proxy,ref,type,destPath);
				CompositeListenerChange composite = new CompositeListenerChange(event);	
				composite.add(updater.createUpdateChange(context, ref, type, destPath));
				change.add(composite);
			}
		}
		return change;
	}

	private Change createMoveContainerChange(IFolder folder, IContainer dest) {
		RefactoringEvent event = new MoveRefactoringEvent(folder,destination);
		CompositeListenerChange composite = new CompositeListenerChange(event);		
		MoveFolderCompositeChange change = new MoveFolderCompositeChange(folder, dest.getFullPath());
		performContainerChildren(change, (IContainer)folder, dest.getFullPath().append(folder.getName()));
		change.freeze();
		composite.add(change);
		return composite;
	}
}
