/********************************************************************** 
 * Copyright (c) 2008, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: MoveRefactoringWizard.java,v 1.9 2010/05/05 20:42:48 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.ltk.core.refactoring.Refactoring;
import org.eclipse.ltk.ui.refactoring.RefactoringWizard;
import org.eclipse.ltk.ui.refactoring.UserInputWizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.model.BaseWorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;

/**
 * <p>MoveRefactoringWizard.java</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @version May 5, 2010
 * @since   March 20, 2008
 */
public class MoveRefactoringWizard extends RefactoringWizard {
	
	private static class MoveRefactoringInputPage extends UserInputWizardPage {
		private MoveRefactoring refactoring;
		private TreeViewer viewer;
		private List allParents = null;
		
		public MoveRefactoringInputPage(Refactoring ref) {
			super(""); //$NON-NLS-1$
			this.refactoring = (MoveRefactoring) ref;
		}

		public void createControl(Composite parent) {
			initializeDialogUnits(parent);
			Composite result= new Composite(parent, SWT.NONE);
			setControl(result);
			result.setLayout(new GridLayout());
			
			addLabel(result);
			viewer = createViewer(result);
			expandSelectionContainer();
			viewer.addSelectionChangedListener(new ISelectionChangedListener(){
				public void selectionChanged(SelectionChangedEvent event) {
					MoveRefactoringInputPage.this.viewerSelectionChanged(event);
				}
			});
			Dialog.applyDialogFont(result);
		}
		
		private void expandSelectionContainer() {
			Collection containers = refactoring.getContainers();
			Collection proxies = refactoring.getProxies();
			Collection parents = new HashSet();
			for (Iterator it = containers.iterator(); it.hasNext();) {
				parents.add(((IContainer) it.next()).getParent());
			}
			for (Iterator it = proxies.iterator(); it.hasNext();) {
				parents.add(((IProxyNode) it.next()).getUnderlyingResource().getParent());
			}
			for (Iterator it = parents.iterator(); it.hasNext();) {
				viewer.expandToLevel(it.next(), 0);
			}
		}

		protected Control addLabel(Composite parent) {
			Label label= new Label(parent, SWT.NONE);
			String text;
			int containers= refactoring.getContainers().size();
			int proxies= refactoring.getProxies().size();

			if (containers == 0 && proxies == 1) {
				text= NLS.bind(RefactoringMessages.MOVE_CHOOSE_DESTINATION_SINGLE, ((IProxyNode)refactoring.getProxies().get(0)).getText());
			} else if (containers == 1 && proxies == 0) {
				text= NLS.bind(RefactoringMessages.MOVE_CHOOSE_DESTINATION_SINGLE, ((IResource)refactoring.getContainers().get(0)).getName());
			} else {
				text= NLS.bind(RefactoringMessages.MOVE_CHOOSE_DESTINATION_MULTI, String.valueOf(containers + proxies));
			}
			label.setText(text);
			label.setLayoutData(new GridData(SWT.BEGINNING, SWT.END, false, false));
			buildParentList();
			return label;
		}
		
		private void buildParentList() {
			allParents = new ArrayList();
			for (Iterator it = refactoring.getContainers().iterator(); it.hasNext(); ) {
				allParents.add(((IResource)it.next()).getParent());
			}
			for (Iterator it = refactoring.getProxies().iterator(); it.hasNext(); ) {
				Object parent = ((IProxyNode)it.next()).getUnderlyingResource().getParent();
				if (parent instanceof IContainer) {
					allParents.add(parent);
				}
			}
		}
		
		private TreeViewer createViewer(Composite parent) {
			TreeViewer treeViewer= new TreeViewer(parent, SWT.SINGLE | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
			GridData gd= new GridData(GridData.FILL_BOTH);
			gd.widthHint= convertWidthInCharsToPixels(40);
			gd.heightHint= convertHeightInCharsToPixels(15);
			treeViewer.getTree().setLayoutData(gd);
			BaseWorkbenchContentProvider provider = new BaseWorkbenchContentProvider() {
				public Object[] getChildren(Object element) {
					Object[] children = super.getChildren(element);
					List resChildren = new LinkedList();
					for (int i = 0; i < children.length; i++) {
						if ((children[i] instanceof IContainer) && (((IContainer)children[i]).isAccessible()) && (!TestNavigator.getFiltersManager().filter(children[i]))) {
							resChildren.add(children[i]);
						}
					}
					return resChildren.toArray();
				}
			};
			treeViewer.setLabelProvider(new WorkbenchLabelProvider());
			treeViewer.setContentProvider(provider);
			treeViewer.setSorter(null);
			treeViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
			return treeViewer;
		}

		private void viewerSelectionChanged(SelectionChangedEvent event) {
			
			if(isPageComplete()){

				//Assumption: isPageComplete() assures the selection is a IStructuredSelection containing a container as the first element:
				refactoring.setDestination(((IContainer)(((IStructuredSelection)(event.getSelection())).getFirstElement())));
				
				setPageComplete(true);
			}
			else{
				setPageComplete(false);
			}
		}

		public boolean isPageComplete() {
			
			//Case 1: The selection is missing (should never occur since the dialog does not allow de-selection):			
			ISelection selection = viewer.getSelection();
			
			if ((selection == null) || (selection.isEmpty()) || (!(selection instanceof IStructuredSelection))) {

				setMessage(null);
				setErrorMessage(RefactoringMessages.MISSING_CONTAINER);
				
				return false;
			}

			//Case 2: The selection is not a container (should never occur since the dialog only displays containers):			
			Object selectedContainer = ((IStructuredSelection)(selection)).getFirstElement();
			
			if ((selectedContainer == null) || (!(selectedContainer instanceof IContainer))) {

				setMessage(null);
				setErrorMessage(RefactoringMessages.MISSING_CONTAINER);
				
				return false;
			}

			//Case 3: The selection is the same as an existing parent:			
			IContainer container = ((IContainer)(selectedContainer));

			if (allParents != null) {
				
				Iterator allParentsIterator = allParents.iterator();
				
				while(allParentsIterator.hasNext()){
				
					IContainer parent = ((IContainer)(allParentsIterator.next()));
					
					if (container.equals(parent)) {
						
						setMessage(null);
						setErrorMessage(RefactoringMessages.DUPLICATE_NAME);
						
						return false;	
					}
				}
			}		

			//Case 4: The selection is the same as an existing container/proxy:	
			Iterator refactoringContainersIterator = refactoring.getContainers().iterator();

			while(refactoringContainersIterator.hasNext()){

				String refactoringContainerName = ((IContainer)(refactoringContainersIterator.next())).getName();
				
				if (!TestUIUtil.isResourcePathLengthValid(refactoringContainerName, container)) {

					setMessage(null);
					setErrorMessage(RefactoringMessages.LONG_NAME);

					return false;		
				}
				else if (!TestUIUtil.isResourcePathAvailable(refactoringContainerName, container)) {

					setMessage(null);
					setErrorMessage(RefactoringMessages.DUPLICATE_NAME);

					return false;		
				}
			}

			Iterator refactoringProxiesIterator = refactoring.getProxies().iterator();

			while(refactoringProxiesIterator.hasNext()){

				String refactoringProxiesName = ((IProxyNode)(refactoringProxiesIterator.next())).getUnderlyingResource().getName();
				
				if (!TestUIUtil.isResourcePathLengthValid(refactoringProxiesName, container)) {

					setMessage(null);
					setErrorMessage(RefactoringMessages.LONG_NAME);

					return false;		
				}
				else if (!TestUIUtil.isResourcePathAvailable(refactoringProxiesName, container)) {

					setMessage(null);
					setErrorMessage(RefactoringMessages.DUPLICATE_NAME);

					return false;		
				}
			}

			//Case 5 (default): Valid new container:
			setMessage(null);
			setErrorMessage(null);
			
			return true;	
		}
	}
	
	public MoveRefactoringWizard(Refactoring refactoring) {
		super(refactoring, DIALOG_BASED_USER_INTERFACE | NONE);
	}
	
	protected void addUserInputPages() {
		MoveRefactoringInputPage page = new MoveRefactoringInputPage(getRefactoring());
		addPage(page);
	}
}
