/********************************************************************** 
 * Copyright (c) 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: PasteContainerChange.java,v 1.1 2010/10/26 09:21:36 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.navigator.refactoring.AbstractPasteChange;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;

/**
 * <p>Change to apply to perform a folder external paste.</p>
 * 
 * 
 * @author  Jerome Bozier
 * @version August 27, 2010
 * @version Since 27, 2010
 */
public class PasteContainerChange extends AbstractPasteChange {

	private String externalFileName;
	private IContainer destination;
	private String externalBaseName;
	private String destName;
	
	public PasteContainerChange(String externalFileName, IContainer destination) {
		this.externalFileName = externalFileName;
		this.destination = destination;
		this.externalBaseName = new Path(externalFileName).lastSegment();
		this.destName = PasteFileChange.buildNewName(destination.getFullPath().append(externalBaseName));
	}
	
	@Override
	public Object getModifiedElement() {
		return externalFileName;
	}

	@Override
	public String getName() {
		return NLS.bind(RefactoringMessages.PASTE_FOLDER, externalBaseName, destName);
	}

	@Override
	public void initializeValidationData(IProgressMonitor pm) {
	}

	@Override
	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		RefactoringStatus status = new RefactoringStatus();
		File tmp = new File(externalFileName);
		if (!tmp.exists()) {
			status.addError(RefactoringMessages.NO_FOLDER);
		}	
		return status;
	}

	@Override
	public Change perform(IProgressMonitor pm) throws CoreException {
		Change undo = null;
		pm.beginTask("", 1);  //$NON-NLS-1$
		File file = new File(externalFileName);
		try {
			File destFile = ResourcesPlugin.getWorkspace().getRoot().getLocation().append(destName).toFile();
			if (file.exists()) {
				recursiveCopy(file,destFile,destination);
				undo = new DeleteContainerChange(destination.getFolder(new Path(new Path(destName).lastSegment())),true,null);
			}
			pm.worked(1);
			return undo;
		} catch (IOException e) {
			UiPlugin.logError(RefactoringMessages.IO_ERROR_PASTING_FILE); // TODO : change "file" into folder when we can externalize strings
			new ErrorDialog(Display.getDefault().getActiveShell(), 
					RefactoringMessages.ERROR,
					NLS.bind(RefactoringMessages.IO_ERROR_PASTING_FILE, file.getName()),// TODO : change "file" into folder when we can externalize strings
					new Status(IStatus.ERROR,
							UiPlugin.PLUGIN_ID,
							IStatus.OK,e.toString(),e),IStatus.ERROR).open();
			return undo;
		} finally {
			pm.done();
		}
	}
	
	private void recursiveCopy(File source,File dest,IContainer destfolder) throws IOException, CoreException {
		if (source.isDirectory()) {
			IFolder folder = destfolder.getFolder(new Path(dest.getName()));
			if (!dest.exists()) {
				dest.mkdirs();
				folder.create(true, true, new NullProgressMonitor());
			}
			if (!folder.exists()) {
				folder.create(true, true, new NullProgressMonitor());
			}
			String files[] = source.list();
			for(int i = 0; i < files.length; i++){
				recursiveCopy(new File(source, files[i]), new File(dest, files[i]),folder);
			 
			}
		} else {
			if (!source.exists()) {
				// error, deal it cleanly
				return;
			} else {
				IFile file = destfolder.getFile(new Path(dest.getName()));
				InputStream in = new FileInputStream(source);
				if( file.exists() ) { // should not occur
					file.setContents( in, true, false, null );
				} else {
					file.create(in, true, null );
				}
				in.close();
			}
		}
	}

	@Override
	public String getNewName() {
		return destName;
	}

}
