/********************************************************************** 
 * Copyright (c) 2008, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: PasteFileChange.java,v 1.7 2010/10/26 09:21:36 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigatorMessages;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.navigator.refactoring.AbstractPasteChange;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;

/**
 * <p>Change to apply to perform a file paste.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @version August 27, 2010
 * @since   March 20, 2008
 */
public class PasteFileChange extends AbstractPasteChange {

	private IFile file = null;	
	private IContainer destFolder;
	private String destName = null;
	private String externalBaseName = null;
	private String externalFileName = null;

	// constructor for workspace files
	public PasteFileChange(IFile file, IContainer destination) {
		super();
		this.file = file;
		this.destFolder = destination;
		this.destName = buildNewName(destFolder.getFullPath().append(file.getName()));
	}

	// constructor for external file
	public PasteFileChange(String externalFileName, IContainer destination) {
		super();
		this.file = null;
		this.destFolder = destination;
		this.externalFileName = externalFileName;
		this.externalBaseName = new Path(externalFileName).lastSegment();
		this.destName = buildNewName(destFolder.getFullPath().append(externalBaseName));
	}
	
	public Object getModifiedElement() {
		return file;
	}

	public String getName() {
		if (file != null) {
			IPath path = file.getFullPath().removeFirstSegments(ResourcesPlugin.getWorkspace().getRoot().getFullPath().segmentCount());
			return NLS.bind(RefactoringMessages.PASTE_FILE, path.toString(), destName);
		} else {
			return NLS.bind(RefactoringMessages.PASTE_FILE, externalBaseName, destName);
		}
	}

	public void initializeValidationData(IProgressMonitor pm) {
	}

	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		RefactoringStatus status = new RefactoringStatus();
		if (file != null) {
			if (!file.exists()) {
				status.addError(RefactoringMessages.NO_FILE);
			}
		} else {
			File tmp = new File(externalFileName);
			if (!tmp.exists()) {
				status.addError(RefactoringMessages.NO_FILE);
			}
		}
		return status;
	}
	
	public String getNewName() {
		return destName;
	}

	public static String buildNewName(IPath destination) {
		IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
		if ( (root.getFolder(destination).exists()) || (root.getFile(destination).exists())) {
			String newBaseName = null;
			boolean done = false;		
			String fullFileName = destination.lastSegment();
			int dotIndex = fullFileName.lastIndexOf('.');
			String basePart = fullFileName.substring(0, dotIndex);
			String endPart = fullFileName.substring(dotIndex);
			String newFullName = null;
			String ret = null;
			for (int i=0; (!done); i++) {
				if (i != 0) {
					newBaseName = NLS.bind(TestNavigatorMessages.FileFolderInContainerPaste_NameTwoArg,new Integer(i),basePart);
				} else {
					newBaseName = NLS.bind(TestNavigatorMessages.FileFolderInContainerPaste_NameOneArg,basePart);					 
				}
				newFullName = newBaseName+endPart;
				ret = destination.removeLastSegments(1).append(newFullName).toPortableString();
				IPath newpath = new Path(ret);
				done = (!root.getFolder(newpath).exists()) && (!root.getFile(newpath).exists());
			}
			return ret;
		} else {
			return destination.toPortableString();
		}
	}
		
	public Change perform(IProgressMonitor pm) throws CoreException {
		Change undo = null;
		pm.beginTask("", 1);  //$NON-NLS-1$
		try {
			if (file != null) { // eclipse copy
				IContainer parent = file.getParent();
				file.copy(new Path(destName), false, pm);
				if (parent != null) {
					IFile destFile = destFolder.getFile(new Path(new Path(destName).lastSegment()));
					undo = new DeleteFileChange(destFile);
				}
				file.refreshLocal(0, pm);
			} else {
				IFile file = destFolder.getFile(new Path(new Path(destName).lastSegment()));								
				try {
					InputStream source = new FileInputStream(externalFileName);
					if( file.exists() ) { // should not occur
						file.setContents( source, true, false, null );
					} else {
						file.create( source, true, null );
					}
					source.close();
				} catch (FileNotFoundException e) {
					UiPlugin.logError(RefactoringMessages.NO_FILE); // should not occur, tested in "isValid()"
				} catch (IOException e) {
					UiPlugin.logError(RefactoringMessages.IO_ERROR_PASTING_FILE);
					new ErrorDialog(Display.getDefault().getActiveShell(), 
							RefactoringMessages.ERROR,
							NLS.bind(RefactoringMessages.IO_ERROR_PASTING_FILE, file.getName()),
							new Status(IStatus.ERROR,
									UiPlugin.PLUGIN_ID,
									IStatus.OK,
									e.toString(),
									e),IStatus.ERROR).open();
				}
				undo = new DeleteFileChange(destFolder.getFile(new Path(new Path(destName).lastSegment())));
			}
			pm.worked(1);
			return undo;
		} finally {
			pm.done();
		}
	}

}
