/********************************************************************** 
 * Copyright (c) 2008, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: PasteModelChange.java,v 1.10 2010/05/27 14:20:52 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.loaders.util.Guid;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigatorMessages;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.navigator.EObjectProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IRefactoringContext;
import org.eclipse.hyades.test.ui.navigator.refactoring.EMFRefactoringTransaction;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Paste model change.</p>
 *
 *
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version May 27, 2010
 * @since   March 20, 2008
 */
public class PasteModelChange extends Change {
	private EMFRefactoringTransaction context;
	protected IPath destination;
	protected IProxyNode proxy;
	private boolean shouldBeRenamed;
	private RefactoringStatus status;
	private String newName = null;
	private String newBaseName = null;
	
	
	public PasteModelChange(IRefactoringContext context,IProxyNode node, IPath destinationPath,String [] targetName) {
		this.proxy = node;
		this.destination = destinationPath;
		this.context = (EMFRefactoringTransaction) context.getRefactoringTransaction(EMFRefactoringTransaction.EMF_REFACTORING_TRANSACTION_ID);
		Assert.isNotNull(this.context);
		this.status = new RefactoringStatus();
		IFile destFile = ResourcesPlugin.getWorkspace().getRoot().getFile(destination);
		if (destFile.exists()) {
			handleFileAlreadyExists(this.status);
		}
		this.newName = getNewName(); 
		if ((targetName != null) && (targetName.length > 0)) {
			targetName[0] = newName;
		}
	}

	public Object getModifiedElement() {
		return proxy;
	}

	public String getName() {
		String fileName = proxy.getUnderlyingResource().getFullPath().toString();
		return NLS.bind(RefactoringMessages.PASTE_PROXY, fileName, newName /*destination.removeLastSegments(1)*/);
	}

	public void initializeValidationData(IProgressMonitor pm) {
	}

	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		RefactoringStatus localStatus =  this.status;
		if (!TestUIUtil.isResourcePathLengthValid(destination.toOSString())) {
			localStatus.addFatalError(RefactoringMessages.LONG_NAME);
		}
		return localStatus;
	}

	protected void handleFileAlreadyExists(RefactoringStatus status) {
		shouldBeRenamed = true;
	}
	
	private class UndoPasteModelChange extends Change {

		public Object getModifiedElement() {
			return null;
		}

		public String getName() {
			return null;
		}

		public void initializeValidationData(IProgressMonitor pm) {
		}

		public RefactoringStatus isValid(IProgressMonitor pm)
				throws CoreException, OperationCanceledException {
			return new RefactoringStatus();
		}

		public Change perform(IProgressMonitor pm) throws CoreException {
			// does nothing : updated element are deleted on undo, so no action
			// but we need a not null undo change to not disable it
			return null;
		}
		
	}
	
	public Change perform(IProgressMonitor pm) throws CoreException {
		if (proxy instanceof EObjectProxyNode) {
			//- load the referencer
			Resource res = context.getResourceSet().getResource(((EObjectProxyNode)proxy).getOriginatorURI().trimFragment(), true);
			res.setURI(getNewURI());
			try {
				EMFUtil.save(res);
				EMFUtil.getWorkspaceFile(res).refreshLocal(0, pm);
				List objects = res.getContents();
				if (!objects.isEmpty()) {
					EObject object = (EObject)objects.get(0);
					if (object instanceof CMNNamedElement) {

						CMNNamedElement cmNamedElement = ((CMNNamedElement)(object));

						if (shouldBeRenamed) {
							cmNamedElement.setName(newBaseName);
						}

						cmNamedElement.setId(new Guid().toString());

						EMFUtil.save(res);

						EMFUtil.getWorkspaceFile(res).refreshLocal(0, pm);
					}	
				}
			} catch (Exception e) {
				UiPlugin.logError(e);
			}
		}
		return new UndoPasteModelChange();
	}
	
	public URI getNewURI() {
		return URI.createPlatformResourceURI(newName, false);
	}

	// call it one time only, because can be sometime a bit slow
	private String getNewName() {
		if(shouldBeRenamed) {
			boolean done = false;		
			String fullFileName = destination.lastSegment();
			int dotIndex = fullFileName.lastIndexOf('.');
			String basePart = fullFileName.substring(0, dotIndex);
			String endPart = fullFileName.substring(dotIndex);
			String newFullName = null;
			String ret = null;
			IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
			for (int i=0; (!done); i++) {
				 if (i != 0) {
					 newBaseName = NLS.bind(TestNavigatorMessages.FileFolderInContainerPaste_NameTwoArg,new Integer(i),basePart);
				 } else {
					 newBaseName = NLS.bind(TestNavigatorMessages.FileFolderInContainerPaste_NameOneArg,basePart);					 
				 }
				newFullName = newBaseName+endPart;
				ret = destination.removeLastSegments(1).append(newFullName).toPortableString();
				IPath newpath = new Path(ret);
				done = (!root.getFolder(newpath).exists()) && (!root.getFile(newpath).exists());
			}
			return ret;
		} else {
			return destination.toPortableString();
		}
	}
}
