/********************************************************************** 
 * Copyright (c) 2009, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: PasteRefactoring.java,v 1.8 2010/10/26 09:21:36 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.io.File;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyNodeCache;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.internal.util.TestUIUtilities;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IReferencerProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodePaster;
import org.eclipse.hyades.test.ui.navigator.refactoring.PasteRefactoringEvent;
import org.eclipse.hyades.test.ui.navigator.refactoring.RefactoringEvent;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.NullChange;
import org.eclipse.ltk.core.refactoring.Refactoring;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Paste refactoring.</p>
 * 
 * 
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version August 27, 2010
 * @since   March 20, 2008
 */
public class PasteRefactoring extends Refactoring {
	private Change change;
	private IContainer destination;
	private RefactoringContext context;
	protected List seenElements;
	private List folders; 
	private List proxies; 
	private List externalFiles; 	

	public PasteRefactoring(List folders, List proxies,List externalFiles) {
		super();
		this.seenElements = new LinkedList();
		this.context = new RefactoringContext();
		this.folders = folders;
		this.proxies = proxies;
		this.externalFiles = externalFiles;
	}
	
	public List getContainers() {
		return folders;
	}

	public List getProxies() {
		return proxies;
	}
	
	public String getName() {
		return RefactoringMessages.UPDATE_OWNED_RESSOURCES_REFERENCES;  
	}
	
	public void setDestination(IContainer container) {
		this.destination = container;
		change = null;
	}

	public RefactoringStatus checkFinalConditions(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		pm.beginTask("", 3); //$NON-NLS-1$
		try {
			try {
				TestUIUtilities.createProxyNodes(ResourcesPlugin.getWorkspace().getRoot(), new SubProgressMonitor(pm, 1));
			} catch (OperationCanceledException e) {
				return RefactoringStatus.createFatalErrorStatus(RefactoringMessages.ReorgRefactoring_CANCELED_OPERATION_MSG);
			}			
			change = createChange(new SubProgressMonitor(pm, 1));
			RefactoringStatus valid = change.isValid(new SubProgressMonitor(pm, 1));
			if(!valid.isOK()) {
				change = null;
			}
			return valid;
		} finally {
			pm.done();
		}
	}

	public RefactoringStatus checkInitialConditions(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		pm.beginTask("", 1); //$NON-NLS-1$
		try {
			return new RefactoringStatus();
		} finally {
			pm.done();
		}
	}

	public Change createChange(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		if (change != null) {
			return change;
		}
		pm.beginTask("", folders.size() + proxies.size()+externalFiles.size()); //$NON-NLS-1$ 
		try {
			RefactoringTransactionRootChange c = new RefactoringTransactionRootChange(context);
			CustomRefactoringManager.addAllStartingChange(c,ICustomRefactoring.PASTE);
			for (Iterator it = folders.iterator(); it.hasNext();) {
				Change containerChange = createPasteContainerChange((IFolder) it.next(), destination);
				if (containerChange != null) {
					c.add(containerChange);
				}
				pm.worked(1);
			}
			for (Iterator it = proxies.iterator(); it.hasNext();) {
				Change refChange = createPasteProxyChange((IProxyNode) it.next(), destination);
				if (refChange != null) {
					c.add(refChange);
				}
				pm.worked(1);
			}
			for (Iterator it = externalFiles.iterator(); it.hasNext();) {
				String externalFileName = (String) it.next(); // can't use IFile because it can be something outside any workspace
				File theFile = new File(externalFileName);
				Change refChange = null;
				if (theFile != null) {
					if (theFile.isFile()) {
						refChange = new PasteFileChange(externalFileName, destination);
					} else {
						if (theFile.isDirectory()) {
							refChange = new PasteContainerChange(externalFileName, destination);
						}
					}
				}
				if (refChange != null) {
					RefactoringEvent event = new PasteRefactoringEvent(externalFileName,destination);
					CompositeListenerChange composite = new CompositeListenerChange(event);		
					composite.add(refChange);
					c.add(composite);
				}
				pm.worked(1);
			}
			CustomRefactoringManager.addAllEndingChange(c,ICustomRefactoring.PASTE);
			c.markAsSynthetic();
			return c;
		} finally {
			pm.done();
		}
	}
	
	private Change createPasteContainerChange(IFolder folder, IContainer dest) {
		RefactoringEvent event = new PasteRefactoringEvent(folder,dest);
		CompositeListenerChange composite = new CompositeListenerChange(event);	
		IPath [] finalPath = new IPath[1]; 
		PasteFolderCompositeChange change = new PasteFolderCompositeChange(folder, dest.getFullPath(),finalPath);
		performContainerChildren(change, (IContainer)folder, finalPath[0]);
		change.freeze();
		composite.add(change);
		return composite;
	}
	
	private Change performContainerChildren(CompositeChange change, IContainer source, IPath newPath) {
		if(source.isAccessible()) {
			IResource[] resources;
			try {
				resources = source.members();
			} catch (CoreException e) {
				UiPlugin.logError("Unable to get members of container: "+source.getFullPath().toOSString(), e); //$NON-NLS-1$
				return null;
			}
			for (int i = 0; i < resources.length; i++) {
				if (resources[i] instanceof IFile) {
					IFile file = (IFile)resources[i];
					IProxyNode proxy = FileProxyNodeCache.getInstance().getProxy(file);
					if (proxy != null) {
						// add here stuff to memorize changed children
					}
				} else {
					performContainerChildren(change, (IContainer)resources[i], newPath.append(resources[i].getName()));
				}
			}
		}
		return change;
	}
	
	// physical paste in case there is no link
	// create a new proxy on dest location + copy proxy link if any
	private Change createShallowPasteProxyChange(IProxyNode node, IContainer dest,String [] targetName) {
		IFile file = (IFile)node.getAdapter(IFile.class);
		IProxyNodePaster paster = (IProxyNodePaster)node.getAdapter(IProxyNodePaster.class);
		Change mainChange;
		if (paster != null) {
			String fileName;
			if (file == null) {
				fileName = node.getUnderlyingResource().getName();
			} else {
				fileName = file.getName();
			}
			IPath newPath = dest.getFullPath().append(fileName);
			mainChange = paster.createPasteChange(context, newPath,targetName);
		} else {		
			if(file != null) {
				mainChange = new PasteFileChange(file, dest);
			} else {
				return new NullChange(NLS.bind(RefactoringMessages.UPDATE_PROXY, node.getUnderlyingResource().getFullPath().toPortableString()));
			}
		}
		RefactoringEvent event = new PasteRefactoringEvent(node,dest);
		CompositeListenerChange composite = new CompositeListenerChange(event);
		composite.add(mainChange);
		return composite;
	}
	
	protected Change createPasteProxyChange(IProxyNode node, IContainer dest) {
		if (!seenElements.contains(node)) {
			seenElements.add(node);
			if (node instanceof IReferencerProxyNode) {
				Change change = createPasteReferencerChange((IReferencerProxyNode)node, dest,new String[1]);
				return change;
			} else {
				return createShallowPasteProxyChange(node, dest,new String[1]);
			}
		}		
		return null;
	}
	
	// logical paste in case there is links
	private Change createPasteReferencerChange(IReferencerProxyNode nodeA,IContainer dest,String [] targetName) {
		CompositeChange composite = new CompositeChange(NLS.bind(RefactoringMessages.PASTE_PROXY, nodeA.getText(), dest.getName()));
		Change pasteNode = createShallowPasteProxyChange(nodeA, dest,targetName); 
		if(pasteNode != null) {
			composite.add(pasteNode);
		}				
		return composite;
	}
}
