/********************************************************************** 
 * Copyright (c) 2008, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: RefactoringContext.java,v 1.2 2010/09/10 18:33:00 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.navigator.IRefactoringTransaction;
import org.eclipse.hyades.test.ui.navigator.actions.IRefactoringContext;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;

/**
 * <p>Refactoring context.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Paul Slauenwhite
 * @version September 10, 2010
 * @since   March 20, 2008
 */
public class RefactoringContext implements IRefactoringTransaction,	IRefactoringContext {
	//- internal storage (domainId, IConfigurationElement)
	private static Map knownTransactions;
	//- loaded transaction for this instance (domainId, IRefactoringTransaction)
	private Map transactions;
	
	static {
		knownTransactions = new HashMap();
	}
	
	public RefactoringContext() {
		
		//Note: Transactions MUST be maintained in a predictable iteration order for processing (e.g. commit).
		transactions = new LinkedHashMap();
	}

	public void beginChanges(IProgressMonitor pm) {
		pm.beginTask("", transactions.values().size()); //$NON-NLS-1$
		try {
			for (Iterator it = transactions.values().iterator(); it.hasNext();) {
				IRefactoringTransaction transaction = (IRefactoringTransaction) it.next();
				transaction.beginChanges(new SubProgressMonitor(pm, 1));
			}
		} finally {
			pm.done();
		}		
	}

	public void cancelChanges(IProgressMonitor pm) {
		pm.beginTask("", transactions.values().size()); //$NON-NLS-1$
		try {
			for (Iterator it = transactions.values().iterator(); it.hasNext();) {
				IRefactoringTransaction transaction = (IRefactoringTransaction) it.next();
				transaction.cancelChanges(new SubProgressMonitor(pm, 1));
			}
		} finally {
			pm.done();
		}		
	}

	public Change commitChanges(IProgressMonitor pm) throws CoreException {
		CompositeChange c = new CompositeChange(""); //$NON-NLS-1$
		pm.beginTask("", transactions.values().size()); //$NON-NLS-1$
		try {
			for (Iterator it = transactions.values().iterator(); it.hasNext();) {
				IRefactoringTransaction transaction = (IRefactoringTransaction) it.next();
				c.add(transaction.commitChanges(new SubProgressMonitor(pm, 1)));
			}
			return c;
		} finally {
			pm.done();
		}		
	}

	public IRefactoringTransaction getRefactoringTransaction(String domainId) {
		IRefactoringTransaction t = (IRefactoringTransaction) transactions.get(domainId);
		if (t != null) {
			return t;
		} else {
			//- need to load from internal storage
			try {
				t = internalGetRefactoringTransaction(domainId);
				if (t != null) {
					//- save this transaction in this instance.
					transactions.put(domainId, t);
					return t;
				}
			} catch (CoreException e) {
				UiPlugin.logError(e);
			}
			return null;
		}
	}

	private static IRefactoringTransaction internalGetRefactoringTransaction(String domainId) throws CoreException {
		IConfigurationElement e = (IConfigurationElement)knownTransactions.get(domainId);
		if(e == null) {
			//- need to load from extension points.
			IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + ".testNavigatorRefactoringTransactions"); //$NON-NLS-1$
			if (extPoint != null) {
				IConfigurationElement[] members = extPoint.getConfigurationElements();
				for (int i = 0; i < members.length; i++) {
					IConfigurationElement element = members[i];
					if ("transaction".equals(element.getName())) { //$NON-NLS-1$
						String id = element.getAttribute("domainId"); //$NON-NLS-1$
						if(id != null) {
							knownTransactions.put(id, element);
						}
					}
				}
				//- once extensions have been loaded try to retrieve the asked refactoring transaction
				e = (IConfigurationElement)knownTransactions.get(domainId);
			}
		}
		if(e == null) {
			//- this domainId is unknown
			UiPlugin.logError("Unable to find a registered refactoring transaction with the domain Id: '"+domainId+'\''); //$NON-NLS-1$
			return null;
		}
		return (IRefactoringTransaction) e.createExecutableExtension("class"); //$NON-NLS-1$
	}

}
