/********************************************************************** 
 * Copyright (c) 2008, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: RenameContainerChange.java,v 1.4 2009/11/27 14:44:21 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourceAttributes;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.internal.util.TestUIUtilities;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.NullChange;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * RenameContainerChange.java
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @version November 20, 2009
 * @since   March 20, 2008
 */
public class RenameContainerChange extends CompositeChange {
	private IContainer container;
	private boolean isFolder;
	private String newName;

	public RenameContainerChange(IContainer container, String newName) {
		super(""); //$NON-NLS-1$
		this.container = container;
		this.newName = newName;
		this.isFolder = container.getFullPath().segmentCount() > 1;
	}
	
	public Object getModifiedElement() {
		return container;
	}

	public String getName() {
		IPath path = container.getFullPath().removeFirstSegments(ResourcesPlugin.getWorkspace().getRoot().getFullPath().segmentCount());
		return NLS.bind(RefactoringMessages.RENAME_CONTAINER, path.toString(), path.removeLastSegments(1).append(newName));
	}

	public void initializeValidationData(IProgressMonitor pm) {
	}

	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		RefactoringStatus status = new RefactoringStatus();
		IContainer newContainer = getNewContainer();
		if (!container.isAccessible()) {
			if(isFolder) {
				status.addError(RefactoringMessages.NO_FOLDER);
			} else {
				status.addError(RefactoringMessages.NO_PROJECT);
			}
		}
		ResourceAttributes att = container.getResourceAttributes();
		if(att != null && att.isReadOnly()) {
			if(isFolder) {
				status.addError(RefactoringMessages.READ_ONLY_FOLDER);
			} else {
				status.addError(RefactoringMessages.READ_ONLY_PROJECT);
			}
		}
		if(newContainer.exists()) {
			if(isFolder) {
				status.addError(NLS.bind(RefactoringMessages.ALREADY_EXISTS_FOLDER, newContainer.getName()));
			} else {
				status.addError(NLS.bind(RefactoringMessages.ALREADY_EXISTS_PROJECT, newContainer.getName()));
			}
		}

		return status;
	}

	private IContainer getNewContainer() {
		if(isFolder) {
			return container.getParent().getFolder(new Path(newName));
		} else {
			return ResourcesPlugin.getWorkspace().getRoot().getProject(newName);
		}
	}

	public Change perform(IProgressMonitor pm) throws CoreException {
		Change ret;
		if(isFolder) {
			ret = performFolder(pm);
		} else {
			ret = performProject(pm);
		}
		TestUIUtilities.recursiveRefreshProxy(getNewContainer());
		return ret;
	}

	private Change performProject(IProgressMonitor pm) {
		pm.beginTask("", 2); //$NON-NLS-1$
		try {
			IProject project = (IProject) container;
			String originalName = project.getName();
			try {
				//- first process all children
				super.perform(new SubProgressMonitor(pm, 1));
				IProjectDescription description = project.getDescription();
				IPath newPath = container.getFullPath().removeLastSegments(1).append(newName);
				description.setName(newPath.segment(0));
		        project.move(description, IResource.FORCE | IResource.SHALLOW, new SubProgressMonitor(pm, 1));
			} catch (CoreException e) {
				UiPlugin.logError(e);
			}
			IProject renamedProject = ResourcesPlugin.getWorkspace().getRoot().getProject(newName);
			return new RenameContainerChange(renamedProject,originalName);
		} finally {
			pm.done();
		}		
	}
	
	private Change performFolder(IProgressMonitor pm) {
		pm.beginTask("", 2); //$NON-NLS-1$
		try {
			IFolder folder = (IFolder) container;
			String originalName = folder.getName();
			IPath newPath = folder.getFullPath().removeLastSegments(1).append(newName);
			try {
				//- first process all children
				super.perform(new SubProgressMonitor(pm, 1));
				folder.move(newPath, IResource.KEEP_HISTORY, new SubProgressMonitor(pm, 1));
			} catch (CoreException e) {
				UiPlugin.logError(e);
			}
			IFolder renamedFolder = ResourcesPlugin.getWorkspace().getRoot().getFolder(newPath);
			return new RenameContainerChange(renamedFolder,originalName);
		} finally {
			pm.done();
		}		
	}
	
	public void freeze() {
		if(getChildren().length == 0) {
			markAsSynthetic();
			//- add a dummy child in order to keep the composite alive
			add(new NullChange(getName()));
		}
	}
}
