/********************************************************************** 
 * Copyright (c) 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: RenameRefactoring.java,v 1.7 2009/12/11 09:16:52 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.util.Iterator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.CompositeReferencerProxyNode;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceTypeRegistry;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IReferencerProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer2;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeUpdater;
import org.eclipse.hyades.test.ui.navigator.refactoring.RefactoringEvent;
import org.eclipse.hyades.test.ui.navigator.refactoring.RenameRefactoringEvent;
import org.eclipse.hyades.test.ui.navigator.refactoring.UpdateRefactoringEvent;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.NullChange;
import org.eclipse.osgi.util.NLS;

/**
 * @author Jerome Gout
 * @author Jerome Bozier
 * @version December 10, 2009
 */
public class RenameRefactoring extends ReorgRefactoring {
	private IProxyNode proxy;
	private IContainer container;
	private String name;

	public RenameRefactoring(IProxyNode node) {
		super();
		this.proxy = node;
	}

	public RenameRefactoring(IContainer cont) {
		super();
		this.container = cont;
	}

	public Change createChange(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		if (change != null) {
			return change;
		}
		try {
			pm.beginTask("", 1);  //$NON-NLS-1$
			RefactoringTransactionRootChange c = new RefactoringTransactionRootChange(context);
			CustomRefactoringManager.addAllStartingChange(c,ICustomRefactoring.RENAME);
			if(container != null) {
				Change containerChange = createRenameContainerChange(new SubProgressMonitor(pm, 1));
				if (containerChange != null) {
					c.add(containerChange);
				}
			} else if(proxy != null) {
				Change proxyChange = createRenameProxyNodeChange();
				if (proxyChange != null) {
					c.add(proxyChange);
				}
				pm.worked(1);
			}
			CustomRefactoringManager.addAllEndingChange(c,ICustomRefactoring.RENAME);
			c.markAsSynthetic();
			return c;
		} finally {
			pm.done();
		}
	}

	private Change createRenameProxyNodeChange() {
		if (!seenElements.contains(proxy)) {
			seenElements.add(proxy);
			if (proxy instanceof IReferencerProxyNode) {
				return createRenameModelChange();
			} else {
				return createShallowRenameProxyChange();
			}
		}		
		return null;
	}
	
	private Change createShallowRenameProxyChange() {
		IProxyNodeRenamer2 renamer = (IProxyNodeRenamer2)proxy.getAdapter(IProxyNodeRenamer2.class);
		Change mainChange;
		if (renamer != null) {
			mainChange = renamer.createRenameChange(context, name);
		} else {
			IFile file = (IFile)proxy.getAdapter(IFile.class);
			if (file == null) {
				IResource res = proxy.getUnderlyingResource();
				if (res instanceof IFile) {
					file = (IFile)res;
				}
			}
			if (file != null) {
				mainChange = new RenameFileChange(file, name);
			} else {
				return new NullChange(NLS.bind(RefactoringMessages.UPDATE_PROXY, proxy.getUnderlyingResource().getFullPath().toPortableString()));
			}
		}
		RefactoringEvent event = new RenameRefactoringEvent(proxy,name);
		CompositeListenerChange composite = new CompositeListenerChange(event);	
		composite.add(mainChange);
		return composite;
	}

	private Change createRenameModelChange() {
		CompositeChange composite = new CompositeChange(""); //$NON-NLS-1$
		Change renameNode = createShallowRenameProxyChange();
		if(renameNode != null) {
			composite.add(renameNode);
		}
		CompositeReferencerProxyNode ref = new CompositeReferencerProxyNode(proxy);
		for (Iterator it = ref.getReferenceTypes().iterator(); it.hasNext();) {
			String type = (String)it.next();
			String oppType = ReferenceTypeRegistry.getInstance().getOppositeReferenceType(type);
			if(ReferenceTypeRegistry.getInstance().isExplicit(oppType)) {
				composite.add(createUpdateReferenceTypeChange(ref, type));
			}
		}
		composite.markAsSynthetic();
		return composite;
	}

	private Change createUpdateReferenceTypeChange(CompositeReferencerProxyNode ref, String type) {
		CompositeChange change = new CompositeChange(ReferenceTypeRegistry.getInstance().getName(type));
		for (Iterator it = ref.getReferences(type).iterator(); it.hasNext();) {
			IProxyNode localproxy = (IProxyNode)it.next();
			IProxyNodeUpdater updater = (IProxyNodeUpdater) localproxy.getAdapter(IProxyNodeUpdater.class);
			if (updater != null) {
				IPath oldPath = ref.getUnderlyingResource().getFullPath();
				IPath destPath = oldPath.removeLastSegments(1).append(new Path(name+'.'+oldPath.getFileExtension()));				
				RefactoringEvent event = new UpdateRefactoringEvent(proxy,ref,type,destPath);
				CompositeListenerChange composite = new CompositeListenerChange(event);	
				composite.add(updater.createUpdateChange(context, ref, type, destPath));				
				change.add(composite);
			}
		}
		return change;
	}
	
	private Change createRenameContainerChange(IProgressMonitor pm) {
		RefactoringEvent event = new RenameRefactoringEvent(container,name);
		CompositeListenerChange composite = new CompositeListenerChange(event);	
		RenameContainerChange change = new RenameContainerChange(container, name);
		performContainerChildren(change, container, container.getFullPath().removeLastSegments(1).append(name));
		change.freeze();
		composite.add(change);
		return composite;
	}

	public void setNewName(String text) {
		name = text;
		change = null;
		seenElements.clear();
	}
}
