/********************************************************************** 
 * Copyright (c) 2008, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ReorgContainerChange.java,v 1.4 2009/11/27 14:44:21 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourceAttributes;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;

/**
 * Abstract class for handling the change for deletion or move of a container.
 * This change has no children. 
 * <p/>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @version November 20, 2009
 * @since   March 20, 2008
 */
public abstract class ReorgContainerChange extends PhysicalDeleteChange { // in fact, only for delete container
	private IPath path;
	
	public ReorgContainerChange(IContainer cont) {
		path = getPath(cont);
	}
	
	public static IPath getPath(IContainer container){
		return container.getFullPath().removeFirstSegments(ResourcesPlugin.getWorkspace().getRoot().getFullPath().segmentCount());
	}
	
	private IContainer getContainer(IPath path) {
		if(path.segmentCount() == 1) {
			//- this the path of a project
			return ResourcesPlugin.getWorkspace().getRoot().getProject(path.lastSegment());
		} else {
			//- this the path of a folder
			return ResourcesPlugin.getWorkspace().getRoot().getFolder(path);
		}
	}

	public IPath getPath() {
		return path;
	}
	
	public Object getModifiedElement() {
		return getContainer(path);
	}

	public abstract String getName();

	public void initializeValidationData(IProgressMonitor pm) {
	}

	public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException,	OperationCanceledException {
		RefactoringStatus status = new RefactoringStatus();
		Object modified = getModifiedElement();
		ResourceAttributes resAtt = null;
		if (modified instanceof IResource) { // project or folder
			IResource resource = (IResource)modified;
			if (!resource.isAccessible()) {
				if (resource instanceof IFolder) {
					status.addFatalError(RefactoringMessages.NO_FOLDER);
				}
				if (resource instanceof IProject) {
					if (resource.exists()) { // project is closed
						status.addWarning(RefactoringMessages.NOT_OPEN_PROJECT);
					} else {
						status.addFatalError(RefactoringMessages.NOT_EXIST_PROJECT);
					}
				}
			} else {
				resAtt = resource.getResourceAttributes();
				if (resAtt == null) {
					status.addError(RefactoringMessages.ERROR);
				} else {			
					if (resAtt.isReadOnly()) {
						if (resource instanceof IFolder) {
							status.addWarning(RefactoringMessages.READ_ONLY_FOLDER);
						}
						if (resource instanceof IProject) {
							status.addWarning(RefactoringMessages.READ_ONLY_PROJECT);
						}
					}
				}
			}
		} else {
			status.addError(RefactoringMessages.ERROR);
		}
		return status;
	}

	public Change perform(IProgressMonitor pm) throws CoreException {
		IContainer container = getContainer(path);
		pm.beginTask("", 1); //$NON-NLS-1$
		try {
			return reorgFolder(container, new SubProgressMonitor(pm, 1));
		} finally {
			pm.done();
		}
	}

	abstract public Change reorgFolder(IContainer container, SubProgressMonitor monitor);

}
