/********************************************************************** 
 * Copyright (c) 2009, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ReorgRefactoring.java,v 1.7 2010/03/29 17:33:55 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyNodeCache;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.CompositeReferencerProxyNode;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceTypeRegistry;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.internal.util.TestUIUtilities;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IReferencerProxyNode;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeUpdater;
import org.eclipse.hyades.test.ui.navigator.refactoring.RefactoringEvent;
import org.eclipse.hyades.test.ui.navigator.refactoring.UpdateRefactoringEvent;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.Refactoring;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;

/**
 * <p>Move and rename refactoring.<p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version March 29, 2010
 * @since   March 20, 2008
 */
public abstract class ReorgRefactoring extends Refactoring {
	protected Change change;
	protected List seenElements;
	protected RefactoringContext context;
	
	public ReorgRefactoring() {
		initializeChanges();
	}

	private void initializeChanges() {
		this.change = null;
		this.seenElements = new LinkedList();
		this.context = new RefactoringContext();
	}

	public RefactoringStatus checkFinalConditions(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		initializeChanges();
		pm.beginTask("", 3); //$NON-NLS-1$
		try {
			try {
				TestUIUtilities.createProxyNodes(ResourcesPlugin.getWorkspace().getRoot(), new SubProgressMonitor(pm, 1));
			} catch (OperationCanceledException e) {
				return RefactoringStatus.createFatalErrorStatus(RefactoringMessages.ReorgRefactoring_CANCELED_OPERATION_MSG);
			}			
			change = createChange(new SubProgressMonitor(pm, 1));
			pm.worked(1);
			return change.isValid(new SubProgressMonitor(pm, 1));
		} finally {
			pm.done();
		}
	}

	public RefactoringStatus checkInitialConditions(IProgressMonitor pm) throws CoreException, OperationCanceledException {
		pm.beginTask("", 1); //$NON-NLS-1$
		try {
			pm.worked(1);
			return new RefactoringStatus();
		} finally {
			pm.done();
		}
	}

	public String getName() {
		return "";  //$NON-NLS-1$
	}

	protected Change performContainerChildren(CompositeChange change, IContainer source, IPath newPath) {
		if(source.isAccessible()) {
			IResource[] resources;
			try {
				resources = source.members();
			} catch (CoreException e) {
				UiPlugin.logError("Unable to get members of container: "+source.getFullPath().toOSString(), e); //$NON-NLS-1$
				return null;
			}
			for (int i = 0; i < resources.length; i++) {
				if (resources[i] instanceof IFile) {
					IFile file = (IFile)resources[i];
					IProxyNode proxy = FileProxyNodeCache.getInstance().getProxy(file);
					if (proxy != null) {
						if(!seenElements.contains(proxy)) {
							seenElements.add(proxy);
							//- is this proxy adaptable to updater ?
							Change updateChange = null;
							IProxyNodeUpdater updater = (IProxyNodeUpdater)proxy.getAdapter(IProxyNodeUpdater.class);
							IPath dest = newPath.append(file.getName());
							if(updater != null) {
								updateChange = updater.createUpdateChange(context, source, dest);
							}
							CompositeChange composite = new CompositeChange(NLS.bind(RefactoringMessages.CHANGES_FOR_UPDATING, proxy.getText()));
							if(updateChange != null) {								
								RefactoringEvent event = new UpdateRefactoringEvent(proxy,source,dest);
								CompositeListenerChange compositeListener = new CompositeListenerChange(event);	
								compositeListener.add(updateChange);
								composite.add(compositeListener);
							}
							//- process all references
							if (proxy instanceof IReferencerProxyNode) {
								CompositeReferencerProxyNode ref = new CompositeReferencerProxyNode(proxy);
								for (Iterator it = ref.getReferenceTypes().iterator(); it.hasNext();) {
									createRenameReferencerTypeChildren(composite, ref, (String)it.next(), dest);
								}
							}
							change.add(composite);
						}
					}
				} else {
					performContainerChildren(change, (IContainer)resources[i], newPath.append(resources[i].getName()));
				}
			}
		}
		return change;
	}

	private void createRenameReferencerTypeChildren(CompositeChange change, CompositeReferencerProxyNode ref, String type, IPath destinationPath) {
		String oppType = ReferenceTypeRegistry.getInstance().getOppositeReferenceType(type);
		if(ReferenceTypeRegistry.getInstance().isExplicit(oppType)) {
			CompositeChange c = new CompositeChange(ReferenceTypeRegistry.getInstance().getName(type));
			for (Iterator it = ref.getReferences(type).iterator(); it.hasNext();) {
				IProxyNode proxy = (IProxyNode)it.next();
				IProxyNodeUpdater updater = (IProxyNodeUpdater) proxy.getAdapter(IProxyNodeUpdater.class);
				if (updater != null) {
					Change updateChange = updater.createUpdateChange(context, ref, type, destinationPath);
					if (updateChange != null) {						
						RefactoringEvent event = new UpdateRefactoringEvent(proxy, ref, type, destinationPath);
						CompositeListenerChange compositeListener = new CompositeListenerChange(event);	
						compositeListener.add(updateChange);						
						c.add(compositeListener);
					}						
				}
			}
			if(c.getChildren().length > 0) {
				change.add(c);
			}
		}
	}

}
