/********************************************************************** 
 * Copyright (c) 2009, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: TrashBox.java,v 1.3 2010/06/28 12:19:40 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.navigator.refactoring;

import java.io.File;
import java.util.HashMap;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.util.ZipUtils;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;

/**
 * this class deals with delete undo
 * goal is to store deleted elements inside a folder on disk to be able to retrieve them in case of undo
 * deleted resource is stored as a temporary zip file (for container, it contains the whole container)
 * we keep a map of deleted resource that associate the original resource name with the temporary zip name
 * 
 * project that are physically deleted are deleted as container for physical resource storing
 * 
 * deleting a project also remove them from workspace, so we keep a map of deleted project 
 * with their description (if open) or their name (if not)
 * 
 * zip file are considered as temporary, so should be deleted automatically when JVM exit cleanly
 * trash folder is stored inside .metadata\.plugins\org.eclipse.hyades.test.ui and is also deleted on jvm exit
 * 
 * @author  Jerome Bozier
 * @version June 4, 2010
 * @since   March 2, 2010
 */
public class TrashBox {
	private static final String TRASH_BOX_NAME = ".trashbox"; //$NON-NLS-1
	private static TrashBox instance = null;
	private static HashMap<String,String> deletedResourceMap = null;
	private static HashMap<String,Object> deletedProjects = null;
	private static File trashDir = null; 
	
	/**
	 * - initialize memorization maps (one for zip names/resource name, one for project physical location/description or name if closed
	 * - create trashbox directory if it does not exists. empty it if it already exists
	 */
	private TrashBox() throws Exception {
		deletedResourceMap = new HashMap<String,String>();
		deletedProjects = new HashMap<String,Object>();
		trashDir = new File(Platform.getStateLocation(Platform.getBundle(UiPlugin.getID())).append(TRASH_BOX_NAME).toOSString());
		if (!trashDir.exists()) {
			if (!trashDir.mkdirs()) {
				throw new Exception("can not create trashbox");
			}
		} else {
			if (!trashDir.isDirectory()) {
				throw new Exception("trashbox exists and is not a directory");
			}
			File [] contents = trashDir.listFiles();
			if (contents != null) {
				for (int i=0; i<contents.length; i++) {
					contents[i].delete();
				}
			}
		}
		trashDir.deleteOnExit();
	}

	/**
	 * create (if none) or return the trashbox
	 */
	public static TrashBox instance() throws Exception {
		if (instance == null) {
			instance = new TrashBox();
		}
		return instance;		
	}
	
	/**
	 * save deleted project name (if closed) or description
	 * if physical delete, also save the physical project as a zip
	 */
	public void trashProject(IProject project,boolean physicalDelete) throws Exception {
		String deletedResourceName = project.getLocation().toOSString();
		if (project.isOpen()) {
			deletedProjects.put(deletedResourceName,project.getDescription());
		} else {
			deletedProjects.put(deletedResourceName,project.getName()); // store name instead of descriptor if closed
		}
		if (physicalDelete) {
			trashResource(project);
		}
	}
	
	/**
	 * save resource in a temporary zipped file
	 * store the original resource name with the corresponding zip name  
	 */
	public void trashResource(IResource resource) throws Exception {		
		if (resource.exists()) {
			File target = null;
			target = File.createTempFile("deleted",".zip",trashDir);
			String deletedResourceName = resource.getLocation().toOSString();
			ZipUtils zip = new ZipUtils(deletedResourceName,target.getAbsolutePath());
			zip.zip();
			target.deleteOnExit();
			deletedResourceMap.put(deletedResourceName,target.getAbsolutePath());
		}
	}
	
	/**
	 * physically retrieve the project contents if needed
	 * add project to workspace
	 */
	public void untrashProject(String projectName) throws Exception {
		untrashResource(projectName); // retrieve project contents if physically deleted
		Object info = deletedProjects.get(projectName);
		if (info != null) {
			if (info instanceof IProjectDescription) { // project was open
				IProjectDescription desc = (IProjectDescription)info;
				IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(desc.getName());
				project.create(new NullProgressMonitor());
				project.open(new NullProgressMonitor());
			} else { // project was closed
				IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject((String)info);
				project.create(new NullProgressMonitor());
			}
			deletedProjects.remove(projectName);
		}
	}
	
	/**
	 * retrieve zip name associated to resource name
	 * unzip it and return used zip file name
	 */
	public void untrashResource(String resourceName) throws Exception {
		String zipName = deletedResourceMap.get(resourceName);
		if (zipName == null) {
			return;
		}
		ZipUtils zip = new ZipUtils(resourceName,zipName);
		zip.unzip(); 
		zip.deleteteZip();
		deletedResourceMap.remove(resourceName);
	}
	
	/**
	 * delete all files inside trash box
	 * reset storing maps
	 */
	public void cleanTrash() {
		File [] contents = trashDir.listFiles();
		if (contents != null) {
			for (int i=0; i<contents.length; i++) {
				contents[i].delete();
			}
		}
		deletedResourceMap.clear();
		deletedProjects.clear();
	}
	
	/**
	 * a change to retrieve a project stored in trashbox
	 *
	 */
	private class UntrashProjectChange extends Change {
		private String projectName = null;
		
		public UntrashProjectChange(IProject project) {
			this.projectName = project.getLocation().toOSString();
		}
		
		public Object getModifiedElement() {
			return null;
		}

		public String getName() {
			return "retrieving project from trash"; //$NON-NLS-1  // undo change so not visible inside UI
		}

		public void initializeValidationData(IProgressMonitor pm) {
		}

		public RefactoringStatus isValid(IProgressMonitor pm)
				throws CoreException, OperationCanceledException {
			return new RefactoringStatus();
		}

		public Change perform(IProgressMonitor pm) throws CoreException {
			Object val = deletedProjects.get(projectName);
			if (val == null) {
				return null; // we do nothing, so no undo
			}
			try {
				untrashProject(projectName);
			} catch (Exception e) {
				UiPlugin.logError("error while retrieving project : "+projectName);
				UiPlugin.logError(e);
				return null;
			} finally {
				pm.worked(1);
			}
			return null;
		}		
	}

	/**
	 * a change to retrieve a file or a container stored in trashbox
	 *
	 */
	private class UntrashResourceChange extends Change {
		private String resourceName = null;
		
		public UntrashResourceChange(IResource resource) {
			this.resourceName = resource.getLocation().toOSString();
		}
		
		public Object getModifiedElement() {
			return null;
		}

		public String getName() {
			return "retrieving resource from trash"; //$NON-NLS-1  // undo change so not visible inside UI
		}

		public void initializeValidationData(IProgressMonitor pm) {
		}

		public RefactoringStatus isValid(IProgressMonitor pm)
				throws CoreException, OperationCanceledException {
			return new RefactoringStatus();
		}

		public Change perform(IProgressMonitor pm) throws CoreException {
			try {
				untrashResource(resourceName);
			} catch (Exception e) {
				UiPlugin.logError("error while retrieving resource : "+resourceName);
				UiPlugin.logError(e);
				return null;
			} finally {
				pm.worked(1);
			}
			return null;
		}		
	}
	
	/**
	 * create a change for the given resource, depending if it's a project or not
	 */
	public Change createUntrashChange(IResource resource) {
		if (resource instanceof IProject) {
			return new UntrashProjectChange((IProject)resource);
		} else {
			return new UntrashResourceChange(resource);
		}
	}
	
	/**
	 * a change to retrieve a file or a container stored in trashbox
	 *
	 */
	private class CleanTrashChange extends CompositeChange {
		
		public CleanTrashChange() {
			super("");
			super.markAsSynthetic();
		}

		public Change perform(IProgressMonitor pm) throws CoreException {
			Change undo = super.perform(pm);
			cleanTrash();
			return undo;
		}	
		
		public boolean isEnabled() {
			boolean needToPerform = true;
			Change[] children = getChildren();
			for (int i = 0; i < children.length; i++) {
				needToPerform &= ((Change)children[i]).isEnabled();
			}
			return needToPerform;
		}
	}
	
	public Change createCleanTrashChange() {
		return new CleanTrashChange();
	}
	
	/**
	 * a change to refresh workspace
	 *
	 */
	private class RefreshWorkspaceChange extends Change {
		
		public Object getModifiedElement() {
			return null;
		}

		public String getName() {
			return "refreshing workspace"; //$NON-NLS-1  // undo change so not visible inside UI
		}

		public void initializeValidationData(IProgressMonitor pm) {
		}

		public RefactoringStatus isValid(IProgressMonitor pm)
				throws CoreException, OperationCanceledException {
			return new RefactoringStatus();
		}

		public Change perform(IProgressMonitor pm) throws CoreException {
			ResourcesPlugin.getWorkspace().getRoot().refreshLocal(IResource.DEPTH_INFINITE, new NullProgressMonitor());
			return null;
		}		
	}
	
	/**
	 * invisible change that does nothing, but does a refresh on workspace when undo is trigger
	 *
	 */
	private class RefreshOnUndoChange extends CompositeChange {
		
		public RefreshOnUndoChange() {
			super("");
			super.markAsSynthetic();
		}

		public Change perform(IProgressMonitor pm) throws CoreException {
			Change undo = super.perform(pm);
			CompositeChange c = new CompositeChange("");
			c.add(new RefreshWorkspaceChange());
			c.add(undo);
			return c;
		}	
		
		public boolean isEnabled() {
			boolean needToPerform = true;
			Change[] children = getChildren();
			for (int i = 0; i < children.length; i++) {
				needToPerform &= ((Change)children[i]).isEnabled();
			}
			return needToPerform;
		}
	}
	/**
	 * create an invisible change that does nothing, but does a refresh on workspace when undo is trigger
	 * @return
	 */
	public Change createRefreshOnUndoChange() {
		return new RefreshOnUndoChange();
	}
}
 