/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestPreferencePage.java,v 1.20 2009/05/19 02:26:22 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.preference;

import org.eclipse.core.runtime.Preferences;
import org.eclipse.hyades.test.core.TestCorePlugin;
import org.eclipse.hyades.test.core.TestCorePreferences;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.test.ui.internal.util.TestUIUtilities;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Link;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

/**
 * <p>Test tools preferences.</p>
 * 
 * 
 * @author      Paul E. Slauenwhite
 * @author      Jerome Gout
 * @version     May 18, 2009
 * @since       February 1, 2005
 */
public class TestPreferencePage extends PreferencePage
	implements IWorkbenchPreferencePage, ModifyListener
{
	private Button alwaysOpenEditorButton;
	private Button neverOpenEditorButton;
	private Button promptOpenEditorButton;
	private Text saveManagerIntervalText;
	private Text defaultHostName;
	private Text defaultPort;	
	
	/**
	 * Constructor for PluginPreferencePage.
	 */
	public TestPreferencePage()
	{
		super();
		setPreferenceStore(UiPlugin.getInstance().getPreferenceStore());
		
	}

	/**
	 * @see PreferencePage#createContents(Composite)
	 */
	protected Control createContents(Composite parent){
		
		//Composite pane:
		GridLayout compositePaneGroupGridLayout = new GridLayout(1, true);		
		compositePaneGroupGridLayout.marginWidth = compositePaneGroupGridLayout.marginHeight = 0;
		compositePaneGroupGridLayout.horizontalSpacing = compositePaneGroupGridLayout.verticalSpacing = 5;

		Composite compositePane= new Composite(parent, SWT.NONE);	
		compositePane.setLayout(compositePaneGroupGridLayout);
		compositePane.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		//Editors group:						
		Group editorGroup = new Group(compositePane, SWT.SHADOW_ETCHED_IN);
		editorGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
		editorGroup.setLayout(new GridLayout(1, true));
		editorGroup.setText(UiPluginResourceBundle.EDITORS_GROUP_LABEL); 
		
		alwaysOpenEditorButton = new Button(editorGroup, SWT.RADIO);
		alwaysOpenEditorButton.setText(UiPluginResourceBundle.EDITORS_GROUP_ALWAYS_LABEL);

		neverOpenEditorButton = new Button(editorGroup, SWT.RADIO);
		neverOpenEditorButton.setText(UiPluginResourceBundle.EDITORS_GROUP_NEVER_LABEL);

		promptOpenEditorButton = new Button(editorGroup, SWT.RADIO);
		promptOpenEditorButton.setText(UiPluginResourceBundle.EDITORS_GROUP_PROMPT_LABEL);

		//Location group:
		Group locationGroup = new Group(compositePane, SWT.SHADOW_ETCHED_IN);
		locationGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
		locationGroup.setLayout(new GridLayout(2, false));
		locationGroup.setText(UiPluginResourceBundle.LOCATION_GROUP_LABEL); 

		Label defaultHostNameLabel = new Label(locationGroup, SWT.NULL);
		defaultHostNameLabel.setText(UiPluginResourceBundle.DEFAULT_HOST_NAME); 

		defaultHostName = new Text(locationGroup, SWT.BORDER);
		defaultHostName.setLayoutData(GridDataUtil.createHorizontalFill());
		defaultHostName.addModifyListener(this);

		Label defaultPortLabel = new Label(locationGroup, SWT.NULL);
		defaultPortLabel.setText(UiPluginResourceBundle.DEFAULT_PORT); 
		
		defaultPort = new Text(locationGroup, SWT.BORDER);
		defaultPort.setLayoutData(GridDataUtil.createHorizontalFill());
		defaultPort.addModifyListener(this);

		Link testConnectionLink = new Link(locationGroup, SWT.NULL);
		testConnectionLink.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 2, 1));
		testConnectionLink.setText("<a>" + UiPluginResourceBundle.TEST_CONNECTION_NAME + "</a>"); //$NON-NLS-1$ //$NON-NLS-2$
		testConnectionLink.addSelectionListener(new SelectionAdapter(){
			
			public void widgetSelected(SelectionEvent selectionEvent) {
				TestUIUtilities.testConnection(defaultHostName.getText(), defaultPort.getText());
			}
		});
		
		//Test execution group:
		Group testExecutionGroup = new Group(compositePane, SWT.SHADOW_ETCHED_IN);
		testExecutionGroup.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));
		testExecutionGroup.setLayout(new GridLayout(2, false));
		testExecutionGroup.setText(UiPluginResourceBundle.EXECUTION_GROUP_LABEL); 
		
		Label saveManagerIntervalLabel = new Label(testExecutionGroup, SWT.NONE);
		saveManagerIntervalLabel.setText(UiPluginResourceBundle.SAVE_INT_PREF); 
		
		saveManagerIntervalText = new Text(testExecutionGroup, SWT.BORDER);
		saveManagerIntervalText.setLayoutData(GridDataUtil.createHorizontalFill());

        initializeValues();
        
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(parent, UiPlugin.getID() + ContextIds.TEST_PREF_CONT);										
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(defaultHostName, UiPlugin.getID() + ContextIds.TEST_PREF_HOST_CONT);										
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(defaultPort, UiPlugin.getID() + ContextIds.TEST_PREF_PORT_CONT);										
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(alwaysOpenEditorButton, UiPlugin.getID() + ContextIds.TEST_PREF_EDIT_CONT);										
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(neverOpenEditorButton, UiPlugin.getID() + ContextIds.TEST_PREF_EDIT_CONT);										
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(promptOpenEditorButton, UiPlugin.getID() + ContextIds.TEST_PREF_EDIT_CONT);										
		UiPlugin.getDefault().getWorkbench().getHelpSystem().setHelp(saveManagerIntervalText, UiPlugin.getID() + ContextIds.TEST_PREF_SAVE_CONT);										

		Dialog.applyDialogFont( compositePane );
			
		return compositePane;
	}

	/**
	 * Initializes states of the controls from the preference store.
	 */
	private void initializeValues()
	{
		IPreferenceStore testuiStore = getPreferenceStore();
		TestCorePlugin.getDefault().getPluginPreferences();
		
		setOpenEditorButtonSelection(testuiStore.getString(TestUI.OPEN_EDITOR));
		saveManagerIntervalText.setText(String.valueOf(testuiStore.getLong(TestCorePreferences.SAVE_INTERVAL)));
		defaultHostName.setText(testuiStore.getString(TestCorePreferences.DEFAULT_HOST_NAME));
		defaultPort.setText(testuiStore.getString(TestCorePreferences.DEFAULT_PORT));
	}

	/*
	 * @see PreferencePage#performDefaults()
	 */
	protected void performDefaults()
	{
		super.performDefaults();		
		initializeDefaults();

	}
	
	protected IPreferenceStore doGetPreferenceStore() {
		return UiPlugin.getInstance().getPreferenceStore();
	}
	
	
	/**
	 * Initializes states of the controls using default values
	 * in the preference store.
	 */
	private void initializeDefaults()
	{
		IPreferenceStore testuiStore = getPreferenceStore();
		Preferences testcorePrefs = TestCorePlugin.getDefault().getPluginPreferences();

		setOpenEditorButtonSelection(testuiStore.getDefaultString(TestUI.OPEN_EDITOR));
		saveManagerIntervalText.setText(String.valueOf(testcorePrefs.getDefaultLong(TestCorePreferences.SAVE_INTERVAL)));
		defaultHostName.setText(testcorePrefs.getDefaultString(TestCorePreferences.DEFAULT_HOST_NAME));
		defaultPort.setText(testcorePrefs.getDefaultString(TestCorePreferences.DEFAULT_PORT));
	}
	
	/**
	 * Do anything necessary because the OK button has been pressed.
	 *
	 * @return whether it is okay to close the preference page
	 */
	public boolean performOk() {
		storeValues();
		UiPlugin.getInstance().savePluginPreferences();
		return true;
	}

	/**
	 * Stores the values of the controls back to the preference store.
	 */
	private void storeValues()
	{
		IPreferenceStore testuiStore = getPreferenceStore();
		Preferences testcorePrefs = TestCorePlugin.getDefault().getPluginPreferences();

		testuiStore.setValue(TestUI.OPEN_EDITOR, getOpenEditorButtonSelection());
		
		try{
			testcorePrefs.setValue(TestCorePreferences.SAVE_INTERVAL, Long.parseLong(saveManagerIntervalText.getText().trim()));			
			testuiStore.setValue(TestCorePreferences.SAVE_INTERVAL, Long.parseLong(saveManagerIntervalText.getText().trim()));
		}
		catch(NumberFormatException n){
			UiPlugin.logError(n);
		}
		
		testcorePrefs.setValue(TestCorePreferences.DEFAULT_PORT, defaultPort.getText().trim());
		testcorePrefs.setValue(TestCorePreferences.DEFAULT_HOST_NAME, defaultHostName.getText().trim());
		
		testuiStore.setValue(TestCorePreferences.DEFAULT_PORT, defaultPort.getText().trim());
		testuiStore.setValue(TestCorePreferences.DEFAULT_HOST_NAME, defaultHostName.getText().trim());
	}
	
	/**
	 * @see IWorkbenchPreferencePage#init(IWorkbench)
	 */
	public void init(IWorkbench workbench) {
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e)
	{
		if (e.widget == defaultPort){
			
			if(!TestUIUtilities.isValidPort(defaultPort.getText())){

				setErrorMessage(UiPluginResourceBundle._ERROR_INVALID_PORT_RANGE); 
				defaultPort.setFocus();					
			}
			else{						
				setErrorMessage(null);
			}				
		}
		else if (e.widget == defaultHostName){
			
			if(!TestUIUtilities.isValidHostName(defaultHostName.getText())){

				setErrorMessage(UiPluginResourceBundle._ERROR_INVALID_HOST_NAME); 
				defaultHostName.setFocus();					
			}
			else{						
				setErrorMessage(null);
			}				
		}
	}

	private void setOpenEditorButtonSelection(String selection){

		if(selection.equals(MessageDialogWithToggle.ALWAYS)){

			alwaysOpenEditorButton.setSelection(true);
			neverOpenEditorButton.setSelection(false);
			promptOpenEditorButton.setSelection(false);
		}
		else if(selection.equals(MessageDialogWithToggle.NEVER)){
			
			alwaysOpenEditorButton.setSelection(false);
			neverOpenEditorButton.setSelection(true);
			promptOpenEditorButton.setSelection(false);
		}
		else{
			
			alwaysOpenEditorButton.setSelection(false);
			neverOpenEditorButton.setSelection(false);
			promptOpenEditorButton.setSelection(true);
		}
	}

	private String getOpenEditorButtonSelection() {
		
		if (alwaysOpenEditorButton.getSelection()) {
			return (MessageDialogWithToggle.ALWAYS);
		}
		else if (neverOpenEditorButton.getSelection()) {
			return (MessageDialogWithToggle.NEVER);
		}
		else {
			return (MessageDialogWithToggle.PROMPT);
		}
	}
}
