/*******************************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogResultTreeContentProvider.java,v 1.1 2006/03/15 20:25:09 bjiang Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.search;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;

public class LogResultTreeContentProvider implements ITreeContentProvider, ILogResultContentProvider {
	
	private TreeViewer viewer;
	private TestLogSearchResult result;
	private HashMap childrenMap;
	
	public LogResultTreeContentProvider(TreeViewer treeViewer)
	{
		this.viewer = treeViewer;
	}
	
	public synchronized void init(TestLogSearchResult sResult) {
		this.result = sResult;
		childrenMap= new HashMap();
		if (result != null) {
			Object[] elements= result.getElements();
			for (int i= 0; i < elements.length; i++) {
				insert(elements[i], false);
			}
		}
	}

	public Object[] getChildren(Object parentElement) {
		Set children= (Set)childrenMap.get(parentElement);
		if (children == null)
			return new Object[0];
		return children.toArray();
	}

	public Object getParent(Object element) {
		if(element instanceof TPFExecutionEvent)
		{
			TPFExecutionEvent executionEvent = (TPFExecutionEvent)element;
            if (executionEvent.eContainer() instanceof TPFExecutionHistory)
            {
				//- added to return the TPFExecutionResult in case the eContainer is a TPFExecutionHistory
				TPFExecutionHistory exhis = (TPFExecutionHistory)executionEvent.eContainer();
				TPFExecutionResult exres = exhis.getExecutionResult();
				if (exres.eContainer() != null) 
					return exhis.getExecutionResult();
            }
            else 
                return executionEvent.eContainer();
		}
		else if(element instanceof TPFExecutionHistory)
			return ((TPFExecutionHistory)element).getExecutionResult();
		else if(element instanceof TPFExecutionResult)
        {
            TPFExecutionResult executionResult = (TPFExecutionResult)element;
            if(executionResult.getInvocationExecutionEvent() != null)
                return executionResult.getInvocationExecutionEvent();
        }
		return null;
	}

	public boolean hasChildren(Object element) {
		return (getChildren(element).length > 0) ? true : false;
	}

	public Object[] getElements(Object inputElement) {
		return getChildren(inputElement);
	}

	public void dispose() {
		if(childrenMap != null)
		{
			childrenMap.clear();
			childrenMap = null;
		}
		result = null;
		viewer = null;
	}

	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		if(newInput instanceof TestLogSearchResult)
			init((TestLogSearchResult)newInput);
	}

	protected void insert(Object child, boolean refreshViewer) {
		Object parent= getParent(child);
		while (parent != null) {
			if (insertChild(parent, child)) {
				if (refreshViewer)
					viewer.add(parent, child);
			} else {
				if (refreshViewer)
					viewer.refresh(parent);
				return;
			}
			child= parent;
			parent= getParent(child);
		}
		if (insertChild(result, child)) {
			if (refreshViewer)
				viewer.add(result, child);
		}
	}

	/**
	 * returns true if the child already was a child of parent.
	 * 
	 * @param parent
	 * @param child
	 * @return Returns <code>trye</code> if the child was added
	 */
	private boolean insertChild(Object parent, Object child) {
		Set children= (Set) childrenMap.get(parent);
		if (children == null) {
			children= new HashSet();
			childrenMap.put(parent, children);
		}
		return children.add(child);
	}

	protected void remove(Object element, boolean refreshViewer) {
		// precondition here:  result.getMatchCount(child) <= 0
	
		if (hasChildren(element)) {
			if (refreshViewer)
				viewer.refresh(element);
		} else {
			if (result.getMatchCount(element) == 0) {
				childrenMap.remove(element);
				Object parent= getParent(element);
				if (parent != null) {
					removeFromSiblings(element, parent);
					remove(parent, refreshViewer);
				} else {
					removeFromSiblings(element, result);
					if (refreshViewer)
						viewer.refresh();
				}
			} else {
				if (refreshViewer) {
					viewer.refresh(element);
				}
			}
		}
	}

	private void removeFromSiblings(Object element, Object parent) {
		Set siblings= (Set) childrenMap.get(parent);
		if (siblings != null) {
			siblings.remove(element);
		}
	}
	
	public void elementsChanged(Object[] updatedElements) {
		for (int i= 0; i < updatedElements.length; i++) {
			if (result.getMatchCount(updatedElements[i]) > 0)
				insert(updatedElements[i], true);
			else
				remove(updatedElements[i], true);
		}
	}
	
	public void clear() {
		init(result);
		viewer.refresh();
	}
}
