/*******************************************************************************
 * Copyright (c) 2006, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestLogSearchPage.java,v 1.11 2009/09/03 13:39:59 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.search;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleSearchQuery;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.forms.editor.TestLogViewer;
import org.eclipse.hyades.test.ui.forms.extensions.provisional.ITestLogSearchProvider;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.DialogPage;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.search.ui.ISearchPage;
import org.eclipse.search.ui.ISearchPageContainer;
import org.eclipse.search.ui.ISearchQuery;
import org.eclipse.search.ui.NewSearchUI;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.custom.SashForm;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.widgets.ScrolledPageBook;

/**
 * <p>The Test Log Search page of the Search dialog (<i>Search &gt;&gt; Search... Ctrl+H</i>).</p>
 * 
 * 
 * @author  Bianca Xue Jiang 
 * @author  Paul E. Slauenwhite
 * @version September 3, 2009
 * @since   March 15, 2006
 */
public class TestLogSearchPage extends DialogPage implements ISearchPage, ModifyListener, SelectionListener {
		
	private class SearchExtension
	{
		private IConfigurationElement configElement;
		private String label;
		private Image icon;
		private ITestLogSearchProvider provider;
		
		public SearchExtension(IConfigurationElement ce)
		{
			this.configElement = ce;
		}
		
		public String getLabel()
		{
			if(label == null)
				label = configElement.getAttribute(PROVIDER_EXT_ATTR_LABEL);
			return label;
		}
		public Image getIcon()
		{
			if(icon == null)
			{
				String imageKey = configElement.getAttribute(PROVIDER_EXT_ATTR_ICON);
				if(imageKey != null) {
                    ImageDescriptor imageDescriptor = UIUtil.getImageDescriptorFromPlugin(Platform.getBundle(configElement.getDeclaringExtension().getNamespace()), imageKey);
                    if(imageDescriptor != null) {
                    	icon = imageDescriptor.createImage();
                        UiPlugin.getInstance().getImageRegistry().put(imageKey, imageDescriptor);
                    }
                }
			}
			return icon;
		}
		
		public ITestLogSearchProvider getSearchProvider()
		{
			if(provider == null)
			{
				try {
					provider = (ITestLogSearchProvider)configElement.createExecutableExtension(PROVIDER_EXT_ELEM_PROVIDER);
				}
				catch(Exception e){
					UiPlugin.logError(e);
				}
			}
			return provider;
		}
	}

	/**
	 * The ID of the Test Log Search page (implementation of the <code>org.eclipse.search.searchPages</code> 
	 * extension point).
	 */
	public static final String TEST_LOG_SEARCH_PAGE_ID = "org.eclipse.hyades.test.ui.search.TestLogSearchPage";

	public static final String SEARCH_OPTION_CASE_SENSITIVE = "caseSensitive";
	public static final String SEARCH_OPTION_DEPTH = "depth";	
	public static final String WILD_CARD = "*";
	
	public static final String PROVIDER_EXT_POINT_ID = ".testLogSearchProvider";
	public static final String PROVIDER_EXT_ELEM_PROVIDER = "provider";
	public static final String PROVIDER_EXT_ATTR_LABEL = "label";
	public static final String PROVIDER_EXT_ATTR_ICON = "icon";
	
	private TPFExecutionResult testLog;
	private EObject eventSelection;
	private EObject rootElement;
	private IEditorInput editorInput;
	private ResourceSet resourceSet;
	private ISearchPageContainer pageContainer;
	private Table elementsTable;
	private List checkedElements = new ArrayList();
	private CLabel elementTitle;
	private ScrolledPageBook detailsPageBook;
	private SearchExtension[] elementSearchProviders;
	
	private String searchText;
	private Text searchTextField;
	private boolean caseSensitive;
	private Button caseSensitiveCheck;
	private boolean inChildrenOnly;
	private Button inChildrenOnlyCheck;
	private boolean recursive;
	private Button recursiveCheck;
	//private int searchLevel;
	//private CCombo searchLevelCombo; 
	
	/* (non-Javadoc)
	 * @see org.eclipse.search.ui.ISearchPage#performAction()
	 */
	public boolean performAction() {
		ISearchQuery query = createSearchQuery();
		if(query != null)
			NewSearchUI.runQueryInBackground(query);
			
		return true;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.search.ui.ISearchPage#setContainer(org.eclipse.search.ui.ISearchPageContainer)
	 */
	public void setContainer(ISearchPageContainer container) {
		pageContainer = container;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		//parent.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_YELLOW));
		
		Composite control = new Composite(parent, SWT.NONE);
		control.setFont(parent.getFont());
		control.setLayout(new GridLayout());
		control.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		//control.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_DARK_BLUE));
		
		IEditorPart editorPart = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().getActiveEditor();
		if(editorPart instanceof TestLogViewer)
		{
			testLog = (TPFExecutionResult)((TestLogViewer)editorPart).getEditorObject();
			rootElement = testLog;
			editorInput = editorPart.getEditorInput();
			resourceSet = ((TestLogViewer)editorPart).getResourceSet();
			IStructuredSelection sel = (IStructuredSelection)((TestLogViewer)editorPart).getSelection();
			if(sel != null && !sel.isEmpty())
			{
				Object first = sel.getFirstElement();
				// only take selections from the Events page, not Overview page.
				if(first instanceof TPFExecutionEvent || first instanceof TPFExecutionResult)
					eventSelection = (EObject)first;
				else
					eventSelection = null;
			}
		}
		
		if(testLog == null)
		{
			Label label= new Label(control, SWT.WRAP);
			label.setText(UiPluginResourceBundle.TestLogSearch_NoOpenLogMessage); 
			label.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_CENTER));
		}
		else
		{
			createTextSearchControls(control);
			createElementSearchControls(control);
		}
		
		setControl(control);
		
		Dialog.applyDialogFont(control);
	}
	
	protected void createTextSearchControls(Composite parent)
	{
		Composite searchTextParent = new Composite(parent, SWT.NONE);
		searchTextParent.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_BEGINNING | GridData.FILL_HORIZONTAL));
		searchTextParent.setLayout(new GridLayout(3, false));
		
		Label searchText = new Label(searchTextParent, SWT.NONE);
		searchText.setText(UiPluginResourceBundle.TestLogSearch_SearchForText); 
		searchTextField = new Text(searchTextParent, SWT.BORDER);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		searchTextField.setLayoutData(gd);
		searchTextField.addModifyListener(this);

		caseSensitiveCheck = new Button(searchTextParent, SWT.CHECK);
		caseSensitiveCheck.setText(UiPluginResourceBundle.TestLogSearch_CaseSensitive); 
		caseSensitiveCheck.addSelectionListener(this);
				
		inChildrenOnlyCheck = new Button(searchTextParent, SWT.CHECK);
		inChildrenOnlyCheck.setText(UiPluginResourceBundle.TestLogSearch_RestrictToDescendants); 
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 2;
		inChildrenOnlyCheck.setLayoutData(gd);
		inChildrenOnlyCheck.addSelectionListener(this);
		if(eventSelection == null)
			inChildrenOnlyCheck.setEnabled(false);
	}
	
	protected void createElementSearchControls(Composite parent)
	{
		SashForm sashForm = new SashForm(parent, SWT.HORIZONTAL);
		GridData gd = new GridData(SWT.FILL, SWT.FILL, true, true);
		sashForm.setLayoutData(gd);
		sashForm.setLayout(new FillLayout());
		//sashForm.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_CYAN));
		
		Composite searchElement = new Composite(sashForm, SWT.NONE);
		searchElement.setLayoutData(new GridData(GridData.FILL_VERTICAL));
		searchElement.setLayout(new GridLayout());
		Label searchElementLabel = new Label(searchElement, SWT.NONE);
		searchElementLabel.setText(UiPluginResourceBundle.TestLogSearch_ElementToSearch); 
		elementsTable = new Table(searchElement, SWT.CHECK | SWT.BORDER);
		gd = new GridData(GridData.FILL_BOTH);
		gd.heightHint = 200;
		elementsTable.setLayoutData(gd);
		elementsTable.addSelectionListener(new SelectionListener(){
			public void widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent e) {
				widgetSelected(e);
			}
			public void widgetSelected(org.eclipse.swt.events.SelectionEvent e) {				
				// table item selection changed, update details
				if(e.item instanceof TableItem)
				{
					// keep track of the checked elements.
					if(e.detail == SWT.CHECK)
					{
						if(((TableItem)e.item).getChecked())
							checkedElements.add(e.item);
						else
							checkedElements.remove(e.item);
					}
					
					updateSearchButton();
					
					TableItem[] selection = elementsTable.getSelection();
					// when e.detail == SWT.CHECK, also select the item
					if(selection.length == 0 || selection[0] != e.item)
						elementsTable.setSelection((TableItem)e.item);
					
					elementTitle.setImage(((TableItem)e.item).getImage());
					elementTitle.setText(((TableItem)e.item).getText());
					// Verdict Event details
					detailsPageBook.showPage(e.item);
				}
			}
		});
		checkedElements.clear();
		
		Composite elementDetails = new Composite(sashForm, SWT.NONE);
		elementDetails.setLayout(new GridLayout());
		elementDetails.setLayoutData(new GridData(GridData.FILL_HORIZONTAL | GridData.GRAB_VERTICAL));
		elementTitle = new CLabel(elementDetails, SWT.LEFT|SWT.SHADOW_OUT|SWT.FLAT|SWT.BORDER);
		elementTitle.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		elementTitle.setBackground(Display.getCurrent().getSystemColor(SWT.COLOR_WHITE));
		elementTitle.setText(UiPluginResourceBundle.TestLogSearch_MoreOptions); 
		detailsPageBook = new ScrolledPageBook(elementDetails, SWT.NONE);
		detailsPageBook.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
	
		SearchExtension[] providers = getElementSearchProviders();
		for(int i = 0; i < providers.length; i++)
		{
			TableItem item = new TableItem(elementsTable, SWT.NONE);
			item.setText(providers[i].getLabel());
			item.setImage(providers[i].getIcon());
			item.setData(providers[i].getSearchProvider());
		
			Composite page = detailsPageBook.createPage(item);
			providers[i].getSearchProvider().createContents(page);
			providers[i].getSearchProvider().setContainer(this.pageContainer);
		}
		
		sashForm.setWeights(new int[]{40, 60});
	}

	protected void updateSearchButton(){
		pageContainer.setPerformActionEnabled((checkedElements.size() > 0) && (searchText != null) && (searchText.length() > 0));
	}
	
	/*
	 * (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.DialogPage#setVisible(boolean)
	 */
	public void setVisible(boolean visible) {
		
		updateSearchButton();
		
		super.setVisible(visible);
	}
	
	public void dispose() {
		super.dispose();
	}
	
	public ISearchPageContainer getContainer()
	{
		return pageContainer;
	}
	
	protected ISearchQuery createSearchQuery()
	{
		if(checkedElements.size() == 0)
			return null;
		
		// create queries from providers
		Map options = new Hashtable();
		options.put(SEARCH_OPTION_CASE_SENSITIVE, Boolean.valueOf(caseSensitive));
		options.put(SEARCH_OPTION_DEPTH, Boolean.valueOf(recursive));
		
		if(inChildrenOnly && eventSelection != null)
			rootElement = eventSelection;
			
		String searchQueryText = searchText;
		
		if((searchText == null) || (searchText.trim().length() == 0)){

			searchQueryText = searchText;
			searchText = WILD_CARD;
		}
		else{
			searchText = (WILD_CARD + searchText + WILD_CARD);
		}
		
		if(checkedElements.size() == 0)
			return null;		
		
		SimpleSearchQuery[] queries = new SimpleSearchQuery[checkedElements.size()];
		for(int i = 0; i < checkedElements.size(); i++)
		{
			ITestLogSearchProvider provider = (ITestLogSearchProvider)((TableItem)checkedElements.get(i)).getData();
			queries[i] = provider.createSearchQuery(rootElement, searchText, options);
		}
		
		return new TestLogSearchQuery(queries, resourceSet, editorInput, (testLog.getName()), searchQueryText);
	}
	
	public SearchExtension[] getElementSearchProviders()
	{
		if(elementSearchProviders == null)
		{
			IExtensionPoint ep = Platform.getExtensionRegistry().getExtensionPoint(UiPlugin.getID() + PROVIDER_EXT_POINT_ID);
			IConfigurationElement[] ce = ep.getConfigurationElements();
			elementSearchProviders = new SearchExtension[ce.length];
			for(int i = 0; i < ce.length; i++)
				elementSearchProviders[i] = new SearchExtension(ce[i]);
		}
		return elementSearchProviders;
	}
	
	public void modifyText(ModifyEvent e) {
		
		if(e.widget == searchTextField){

			searchText = searchTextField.getText();
			
			updateSearchButton();
		}
	}
	
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	
	public void widgetSelected(SelectionEvent e) {
		if(e.widget == caseSensitiveCheck)
			caseSensitive = caseSensitiveCheck.getSelection();
		else if(e.widget == inChildrenOnlyCheck)
			inChildrenOnly = inChildrenOnlyCheck.getSelection();
		else if(e.widget == recursiveCheck)
			recursive = recursiveCheck.getSelection();
	}
}
