/*******************************************************************************
 * Copyright (c) 2008, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EObjectEditorInput.java,v 1.2 2009/01/23 12:31:04 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.util;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IStorage;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.PlatformObject;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.PlatformUI;

/**
 * <p>Editor input for {@link EObject}s used to decrease the time to open editors 
 * for loaded {@link EObject}s.  Generally, this editor input is used for 
 * large models (for example, datapools).</p>
 * 
 * <p><b>Note:</b> Editor inputs are intended to be light weight since the navigation 
 * history (default: 50 items) holds a reference to each editor input while the 
 * Eclipse workbench is open as a means of reconstructing the editor.  As such, 
 * the life cycle of the underlying {@link EObject} terminates after the first 
 * invocation of the {@link #getEObject()} method.  Subsequent invocations will 
 * reload the {@link EObject} from the workspace resource (if exists), which may 
 * be a costly operation for large models (for example, datapools).</p>
 * 
 * <p>This editor input is not persisted due to the life cycle of the underlying 
 * {@link EObject}.  As such, these editor inputs are only available in the 
 * navigation history while the Eclipse workbench is open.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version January 22, 2009
 * @since   November 21, 2008
 * @see     IEditorInput
 */
public class EObjectEditorInput extends PlatformObject implements IFileEditorInput{

	/**
	 * The {@link EObject} editor input.
	 */
	private EObject eObject = null;
	
	/**
	 * The workspace resource associated with the {@link EObject} editor input.
	 */
	private IFile file = null;
	
	/**
	 * <p>Constructor for an editor input for {@link EObject}s.</p>
	 * 
	 * <p>If the specified {@link EObject} or the underlying workspace resource 
	 * for the specified {@link EObject} is <code>null</code>, a
	 * {@link IllegalArgumentException} is thrown.</p>
	 * 
	 * @param eObject The non-<code>null</code> {@link EObject} editor input.
	 */
	public EObjectEditorInput(EObject eObject){
				
		if (eObject == null){
			throw new IllegalArgumentException();
		}
		
		IFile workspaceFile = EMFUtil.getWorkspaceFile(eObject);
		
		if (workspaceFile == null){
			throw new IllegalArgumentException();
		}
		
		this.eObject = eObject;
		this.file = workspaceFile;
	}

	/**
	 * <p>Resolves the {@link EObject} editor input.</p>
	 * 
	 * <p><b>Note:</b> Editor inputs are intended to be light weight since the navigation 
	 * history (default: 50 items) holds a reference to each editor input while the 
	 * Eclipse workbench is open as a means of reconstructing the editor.  As such, 
	 * the life cycle of the underlying {@link EObject} terminates after the first 
	 * invocation of this method.  Subsequent invocations will reload the {@link EObject} 
	 * from the workspace resource (if exists), which may be a costly operation for large 
	 * models (for example, datapools).</p>
	 * 
	 * <p>If the {@link EObject} is <code>null</code> and the underlying workspace resource 
	 * for the original {@link EObject} does not exist or the {@link EObject} cannot be
	 * reloaded from the workspace resource, this method returns <code>null</code>.</p>
	 * 
	 * @return The {@link EObject} editor input, otherwise <code>null</code>.
	 */
	public EObject getEObject(){

		if(eObject != null){
			
			//Return and dispose the EOject:
			try {
				return eObject;
			} 
			finally{
				eObject = null;
			}
		}
		else if(exists()){

			//Reload the EObject from the workspace resource:
			EObject[] eObjects = EMFUtil.load(new ResourceSetImpl(), file);
			
			if(eObjects.length == 1){
				return (eObjects[0]);
			}
		}
		
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IEditorInput#exists()
	 */
	public boolean exists() {
		return (file.exists());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IEditorInput#getImageDescriptor()
	 */
	public ImageDescriptor getImageDescriptor() {
		return (PlatformUI.getWorkbench().getEditorRegistry().getImageDescriptor(file.getName()));
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IEditorInput#getName()
	 */
	public String getName() {
		return (file.getName());
	}

	/**
	 * This editor input is not persisted due to the 
	 * life cycle of the underlying {@link EObject}.
	 * 
	 * @return Returns <code>null</code>.
	 * @see org.eclipse.ui.IEditorInput#getPersistable()
	 */
	public IPersistableElement getPersistable() {
		return null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IEditorInput#getToolTipText()
	 */
	public String getToolTipText() {
		return (file.getFullPath().makeRelative().toString());
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object obj) {
		return ((this == obj) || ((obj instanceof IFileEditorInput) && (file.equals(((IFileEditorInput)(obj)).getFile()))));
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	public int hashCode() {
		return (file.hashCode());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IFileEditorInput#getFile()
	 */
	public IFile getFile() {
		return file;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IStorageEditorInput#getStorage()
	 */
	public IStorage getStorage() throws CoreException {
		return file;
	}
}
