/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestUIUtilities.java,v 1.24 2010/10/27 15:40:55 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.util;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.hyades.models.common.common.CMNNodeType;
import org.eclipse.hyades.security.internal.util.ConnectUtil;
import org.eclipse.hyades.test.core.TestCorePreferences;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyManager;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyNodeCache;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.trace.ui.provisional.ITraceUIHelper;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.ui.wizard.ICustomLocationNameValidation;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

/**
 * <p>Test UI utilities.</p>
 * 
 * <p>See {@link TestUIUtil} for more test UI utilities.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version October 27, 2010
 * @since   March 5, 2008
 * @see     TestUIUtil
 */
public class TestUIUtilities implements ICustomLocationNameValidation {

	/**
	 * Singleton instance.
	 */
	private static TestUIUtilities instance = null;

	/**
	 * Privatized constructor.
	 */
	private TestUIUtilities() {	
		//No-operation.
	}
	
	/**
	 * Resolves the singleton instance of this class.
	 * 
	 * @return The singleton instance of this class.
	 */
	public static TestUIUtilities instance() {
		
		if (instance == null) {
			instance = new TestUIUtilities();
		}
		
		return instance;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.tptp.platform.common.ui.wizard.ICustomLocationNameValidation#isNameValid(java.lang.String, org.eclipse.core.resources.IContainer)
	 */
	public boolean isNameValid(String resourceName, IContainer container) {
		return (TestUIUtil.validateResourcePath(resourceName, container) == TestUIUtil.VALID_RESOURCE_PATH);
	}
	
	public static void testConnection() {
		testConnection(UiPlugin.getInstance().getPreferenceStore().getString(TestCorePreferences.DEFAULT_HOST_NAME));
	}	

	public static void testConnection(CMNNodeType node) {
		testConnection(node.getHostname());			
	}

	public static void testConnection(String hostName) {
		testConnection(hostName, UiPlugin.getInstance().getPreferenceStore().getString(TestCorePreferences.DEFAULT_PORT));
	}

	public static void testConnection(final String hostName, final String port) {

		final Shell shell = Display.getDefault().getActiveShell();
		
		//Verify the host name:
		if (!isValidHostName(hostName)){
			MessageDialog.openError(shell, UiPluginResourceBundle.TEST_CONNECTION_NAME, UiPluginResourceBundle._ERROR_INVALID_HOST_NAME);
		}
		else if (!isValidPort(port)){
			MessageDialog.openError(shell, UiPluginResourceBundle.TEST_CONNECTION_NAME, UiPluginResourceBundle._ERROR_INVALID_PORT_RANGE);
		}
		else{

			Runnable runnable = new Runnable() {

				public void run() {
					
					//Test the connection:
					ITraceUIHelper traceUIHelper = ITraceUIHelper.INSTANCE;
			
					if(traceUIHelper!=null){
			
						ConnectUtil connectUtil = new ConnectUtil(hostName.trim(), port.trim(),  traceUIHelper.getTraceUIApplication());

						if (connectUtil.connect() == ConnectUtil.CONNECTION_SUCCESS){
							MessageDialog.openInformation(shell, UiPluginResourceBundle.TEST_CONNECTION_NAME, UiPluginResourceBundle.TEST_CONNECTION_SUCCESS);
						}
					}
				}
			};
			
			//Change to a busy cursor while testing the connection:
			BusyIndicator.showWhile(shell.getDisplay(), runnable);
		}
	}
	
	public static boolean isValidPort(String port){
		
		if ((port != null) && (port.trim().length() > 0)){
			
			try{
				
				int portNumber = Integer.parseInt(port.trim());
				
				return ((portNumber >= 0) && (portNumber <= 65535));
			}
			catch (NumberFormatException n){
				//Ignore and return false;
			}
		}
		
		return false;
	}
	
	/**
	 * <p>Validates the host name (or IPV4 or IPV6 address) according to 
	 * <a href="http://tools.ietf.org/html/rfc1123">RFC 1123</a> and 
	 * <a href="http://tools.ietf.org/html/rfc952">RFC 952</a>.</p>
	 * 
	 * 
	 * @param hostName The host name (or IPV4 or IPV6 address).
	 * @return <code>true</code> if the host name (or IPV4 or IPV6 address) is valid, otherwise <code>false</code>.
	 */
	public static boolean isValidHostName(String hostName){
		
		if ((hostName != null) && (hostName.length() > 0) && (hostName.length() < 256) && (hostName.charAt(0) != '.') && (hostName.charAt(hostName.length() - 1) != '.')){ 

			String[] segments = hostName.split("\\."); //$NON-NLS-1$

			if(segments.length > 0){

				for(String segment : segments){

					char[] characters = segment.toCharArray();

					if((characters.length < 1) || (characters.length > 63) || (characters[0] == '-') || (characters[characters.length - 1] == '-')){
						return false;
					}

					for (char character : characters) {

						//Support host names, IPv4/IPv6 addresses (may start contain colon characters), and DNS names (may contain underscore characters):
						if(((character < '0') || (character > '9')) && ((character < 'A') || (character > 'Z')) && ((character < 'a') || (character > 'z')) && (character != '-') && (character != ':') && (character != '_')){
							return false;
						}
					}
				}
				
				return true;
			}
		}
		
		return false;
	}
	
	/**
	 * <p>Opens the Test Perspective based on its identifier.</p>
	 * 
	 * <p>If the Test Perspective is the current perspective, this 
	 * method gracefully exits.</p>
	 * 
	 * <p>If the Test Perspective cannot be opened, this method logs the exception 
	 * in the Eclipse <code>.log</code> file and gracefully exits.</p>
	 * 
	 * @see TestUIConstants#TEST_PERSPECTIVE_ID
	 */
	public static void openTestPerspective(){

		Display.getDefault().syncExec(new Runnable() {

			public void run() {

				try{

					IWorkbench workbench = PlatformUI.getWorkbench();
					IWorkbenchWindow activeWorkbenchWindow = workbench.getActiveWorkbenchWindow();

					if (!activeWorkbenchWindow.getActivePage().getPerspective().getId().equals(TestUIConstants.TEST_PERSPECTIVE_ID)) {
						workbench.showPerspective(TestUIConstants.TEST_PERSPECTIVE_ID, activeWorkbenchWindow);
					}
				} 
				catch (Exception e) {
					UiPlugin.logError(e);
				}
			}
		});
	}
	
	/**
	 * Enables or disables the parent Composite and each of the child
	 * Controls.  Child Composites are recursively enabled or disabled.  
	 * <p/>
	 * 
	 * @param composite The parent Composite to be enabled or disabled.
	 * @param enabled True if enabled, otherwise false. 
	 */
	public static void setEnabled(Composite composite, boolean enabled){
		
		composite.setEnabled(enabled);
		
		Control[] children = composite.getChildren();
		
		for (int counter = 0; counter < children.length; counter++) {
			
			if(children[counter] instanceof Composite){	
				setEnabled(((Composite)(children[counter])), enabled);
			}
			else{
				children[counter].setEnabled(enabled);
			}
		}
	}
	
	public static void recursiveRefreshProxy(IResource resource) {

		IFileProxyManager proxManager = TestNavigator.getFileProxyManager();

		if (proxManager == null) {
			proxManager = new FileProxyManager();
		}
		
		internalRecursiveRefreshProxy(proxManager,resource);
	}
	
	private static void internalRecursiveRefreshProxy(IFileProxyManager proxManager, IResource resource) {
		
		if (resource.exists()) {
			
			if (resource instanceof IFile) {
				
				proxManager.updateProxy(((IFile)(resource)), null);
				
				return;
			}
			else if (resource instanceof IContainer) {
				
				IContainer container = ((IContainer)(resource));
				
				try {
					
					IResource [] children = container.members();
					
					for (int counter = 0; counter < children.length; counter++) {
						internalRecursiveRefreshProxy(proxManager, children[counter]);
					}
				} 
				catch (CoreException e) {
					UiPlugin.logError(e);
				}
			}
		}
	}	
	
	/**
	 * <p>Create the proxy nodes under a container (e.g. workspace root,
	 * project, or folder).</p>
	 * 
	 * <p>Note: This method causes proxies to be resolved for all test assets below 
	 * a container (e.g. workspace root, project, or folder).  As such, this method 
	 * <i>may</i> be long running.</a>
	 *   
	 * @param container The container (e.g. workspace root, project, or folder). 
	 * @param progressMonitor The progress monitor, otherwise <code>null</code>.
	 * @throws OperationCanceledException If the non-<code>null</code> progress monitor is canceled.
	 */
	public static void createProxyNodes(IContainer rootContainer) {
		createProxyNodes(rootContainer, null);
	}
	
	/**
	 * <p>Create the proxy nodes under a container (e.g. workspace root,
	 * project, or folder).</p>
	 * 
	 * <p>Note: This method causes proxies to be resolved for all test assets below 
	 * a container (e.g. workspace root, project, or folder).  As such, this method 
	 * <i>may</i> be long running.</a>
	 * 
	 * @param container The container (e.g. workspace root, project, or folder). 
	 * @param progressMonitor The progress monitor, otherwise <code>null</code>.
	 * @throws OperationCanceledException If the non-<code>null</code> progress monitor is canceled.
	 */
	public static void createProxyNodes(IContainer container, IProgressMonitor progressMonitor) throws OperationCanceledException{
		
		if(progressMonitor != null){
			progressMonitor.beginTask(RefactoringMessages.TestUIUtilities_CREATING_PROXIES_TASK, IProgressMonitor.UNKNOWN);
		}

		try {
			internalCreateProxyNodes(container, progressMonitor);
		}
		finally{			

			if (progressMonitor != null){
				progressMonitor.done();
			}
		}		
	}

	/**
	 * <p>Create the proxy nodes under a container (e.g. workspace root,
	 * project, or folder).</p>
	 * 
	 * <p>Note: This method causes proxies to be resolved for all test assets below 
	 * a container (e.g. workspace root, project, or folder).  As such, this method 
	 * <i>may</i> be long running.</a>
	 * 
	 * @param container The container (e.g. workspace root, project, or folder). 
	 * @param progressMonitor The progress monitor, otherwise <code>null</code>.
	 * @throws OperationCanceledException If the non-<code>null</code> progress monitor is canceled.
	 */
	private static void internalCreateProxyNodes(IContainer container, IProgressMonitor progressMonitor) throws OperationCanceledException{

		if((container != null) && (container.isAccessible())){

			try {

				IResource[] resources = container.members();

				for (int counter = 0; counter < resources.length; counter++) {					

					if((progressMonitor != null) && (progressMonitor.isCanceled())){
						throw new OperationCanceledException();
					}

					if (resources[counter] instanceof IFile) {					
						FileProxyNodeCache.getInstance().getProxy(((IFile)(resources[counter])));
					} 
					else if (resources[counter] instanceof IContainer) {						
						internalCreateProxyNodes(((IContainer)(resources[counter])), progressMonitor);
					}
				}
			} 
			catch (CoreException c) {
				UiPlugin.logError(c);
			}
		}
	}

}