/********************************************************************** 
 * Copyright (c) 2008, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ZipUtils.java,v 1.2 2009/12/16 14:01:59 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.util;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * this class contains utility to zip and unzip files and folders
 * to create a zip of a file or a container :
 * 		ZipUtils myZip = new ZipUtils("resource name","zip name");
 * 		myZip.zip(); 
 * to unzip a file or a container :
 * 		ZipUtils myZip = new ZipUtils("resource name","zip name");
 * 		myZip.unzip(); 
 * 
 * @author Jerome Bozier
 * @version November 20, 2009
 *
 */
public class ZipUtils {

	private String source = null;
	private String zipFileName = null;
	private ZipOutputStream zipOutputStream = null;
	private ZipInputStream  zipInputStream = null;

	/**
	 * @param source : full resource name of the file or container
	 * @param zipFileName : full zip name
	 */
	public ZipUtils(String source, String zipFileName) {
		this.source = source;
		this.zipFileName = zipFileName;
	}
		
	/**
	 * delete the zipped file
	 */
	public void deleteteZip() {
		File file = new File(zipFileName);
		if (file.exists()) {
			file.delete();
		}
	}
	
	
	private void createParentFolderIfNeeded(String entryName) throws Exception {
		File file = new File(entryName);
		String parentName = file.getParent();
		if (parentName != null) {
			File parentFile = new File(parentName);
			if (!parentFile.exists()) {
				if (!parentFile.mkdirs()) {
					throw new Exception("can't create folder <"+parentName+">");
				}
			}
		}
	}
	
	/**
	 * unzip the zipped file to re-create resource
	 * @throws Exception
	 */
	public void unzip() throws Exception {
		FileInputStream fis = new FileInputStream(zipFileName);
		zipInputStream = new ZipInputStream(fis);
		ZipEntry entry;
		while ((entry = zipInputStream.getNextEntry()) != null) {
			createParentFolderIfNeeded(entry.getName());
			if (entry.isDirectory()) {
				File dir = new File(entry.getName());
				if (!dir.exists()) {
					if (!dir.mkdir()) {
						throw new Exception("can't create folder <"+entry.getName()+">");
					}
				}
			} else {
				int size;
				byte[] buffer = new byte[2048];
				FileOutputStream fos = new FileOutputStream(entry.getName());
				BufferedOutputStream bos = new BufferedOutputStream(fos, buffer.length);
				while ((size = zipInputStream.read(buffer, 0, buffer.length)) != -1) {
					bos.write(buffer, 0, size);
				}
				bos.flush();
				bos.close();
			}
		}
		zipInputStream.close();
		fis.close();
	}
	
	/**
	 * create the zip file from the resource name (file or folder)
	 * @throws Exception
	 */
	public void zip() throws Exception {
		File cpFile = new File(source);

		// Make sure the source is a file or a directory
		if (!cpFile.isFile() && !cpFile.isDirectory()) {
			throw new Exception("invalid source");
		}	 
		// Set flag is source is a directory
		// Create the zip output stream
		zipOutputStream = new ZipOutputStream(new FileOutputStream(zipFileName));
		zipOutputStream.setLevel(9);
		//  Zip the file(s)
		zipFiles(cpFile);
		// Cleanup the zip output stream
		zipOutputStream.finish();
		zipOutputStream.close();
	}

	private void zipFiles(File cpFile) throws Exception {
		// If we are working with a directory then get the list of files
		// in the directory and call zipFiles again
        String zipEntryName  = cpFile.getAbsolutePath();	            
		if (cpFile.isDirectory()) {
			try { 
				ZipEntry dirEntry = new ZipEntry(zipEntryName+"/");
				zipOutputStream.putNextEntry(dirEntry);
				zipOutputStream.closeEntry();
			} catch (IOException e) {
				throw(e);
			}
			File [] fileList = cpFile.listFiles();
	      	// Call zipFiles again with the new list of files.
	        for (int i=0; i< fileList.length; i++) {
	           zipFiles(fileList[i]);
	        }
		} else {
			// Working with a file.  Put it in the zip file.
			FileInputStream theInputStream = null;
			try {
				// Create the zip entry name based on the absolute path of
	            // the file minus the path of the source (if the source was
	            // a directory).  For example:  if source was c:\a_dir then
	            // each zip entry name would start with a_dir\...
	            // Read the file and put it to the zip output stream
	            byte[] b = new byte[(int)(cpFile.length())];
	            theInputStream = new FileInputStream(cpFile);
	            theInputStream.read(b, 0, (int)cpFile.length());
	            ZipEntry theZipEntry = new ZipEntry(zipEntryName);
	            zipOutputStream.putNextEntry(theZipEntry );
	            zipOutputStream.write(b, 0, (int)cpFile.length());
	            zipOutputStream.closeEntry();
	         } catch (Exception e) {
	            // Close the input stream.
	            if (theInputStream != null) {
	            	theInputStream.close();
	            }
	            throw(e);
	         }
	         // Close the input stream.
	         if (theInputStream != null) {
	        	 theInputStream.close();
	         }
		}
	}

}
