/*******************************************************************************
 * Copyright (c) 2009, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CSVFileSelectionComposite.java,v 1.4 2010/02/03 13:57:16 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.io.File;
import java.util.Locale;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.util.CSVImportExportUtil;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;

/**
 * <p>CSV file selection composite.</p>
 * 
 * <p>This composite requires a three/unequal-width column layout 
 * (see {@link #setLayout(org.eclipse.swt.widgets.Layout)}).</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version February 3, 2010
 * @since   April 1, 2009
 */
public class CSVFileSelectionComposite extends Composite {
	
	private Text csvFilePathText = null;
	private Button csvFileBrowseButton = null;
	private Combo csvFileEncodingCombo = null;
	private Button csvFileFirstRowVariableNamesButton = null;
	private Button csvFileFirstColumnEquivalenceClassNamesButton = null;
	private Button csvFileCellsEnclosedInXMLTagsButton = null;

	private static final String DATAPOOL_SECTION_KEY = "Datapool"; //$NON-NLS-1$
	private static final String DATAPOOL_DIR_KEY = "CSV directory"; //$NON-NLS-1$

	private static final String[] ENCODINGS = {UiPluginResourceBundle.WIZ_DATAPOOL_CSV_ENC_DEFAULT, 
			UiPluginResourceBundle.WIZ_DATAPOOL_CSV_ENC_ASCII, 
			UiPluginResourceBundle.WIZ_DATAPOOL_CSV_ENC_ISOLATIN, 
			UiPluginResourceBundle.WIZ_DATAPOOL_CSV_ENC_UTF8, 
			UiPluginResourceBundle.WIZ_DATAPOOL_CSV_ENC_UTF16, 
			UiPluginResourceBundle.WIZ_DATAPOOL_CSV_ENC_UTF16LE, 
			UiPluginResourceBundle.WIZ_DATAPOOL_CSV_ENC_UTF16BE}; 
	
	public static final String[] ENCODING_VALUES = {new String(), 
			CSVImportExportUtil.ASCII,
			CSVImportExportUtil.ISONLATIN,
			CSVImportExportUtil.UTF8,
			CSVImportExportUtil.UTF16,
			CSVImportExportUtil.UTF16LE,
			CSVImportExportUtil.UTF16BE};

	public CSVFileSelectionComposite(Composite parent, int style){
		this(parent, style, null);
	}
	
	public CSVFileSelectionComposite(Composite parent, int style, String csvFileCellsEnclosedInXMLTagsButtonLabel){

		super(parent, style);

		createControl(csvFileCellsEnclosedInXMLTagsButtonLabel);
	}

	public void init(){
		
		csvFilePathText.setText("");
		csvFileEncodingCombo.select(0);
		csvFileFirstRowVariableNamesButton.setSelection(true);
		csvFileFirstColumnEquivalenceClassNamesButton.setSelection(true);
		
		if(csvFileCellsEnclosedInXMLTagsButton != null){
			csvFileCellsEnclosedInXMLTagsButton.setSelection(false);
		}
	}
	
	public Text getCSVFilePathText() {
		return csvFilePathText;
	}

	public Combo getCSVFileEncodingCombo(){		
		return csvFileEncodingCombo;
	}

	public Button getCSVFileFirstRowVariableNamesButton() {
		return csvFileFirstRowVariableNamesButton;
	}

	public Button getCSVFileFirstColumnEquivalenceClassNamesButton() {
		return csvFileFirstColumnEquivalenceClassNamesButton;
	}

	public Button getCSVFileCellsEnclosedInXMLTagsButton() {
		return csvFileCellsEnclosedInXMLTagsButton;
	}
	
	private void createControl(String csvFileCellsEnclosedInXMLTagsButtonText){
		
		//Create CSV file inputs:
		Label csvFileLabel = new Label(this, SWT.NULL);
		csvFileLabel.setText(UiPluginResourceBundle.WIZ_DATAPOOL_CSV_PG_SEL);

		csvFilePathText = new Text(this, SWT.BORDER);
		csvFilePathText.setLayoutData(GridDataUtil.createHorizontalFill());
		
		csvFileBrowseButton = new Button(this, SWT.PUSH);
		csvFileBrowseButton.setText(UiPluginResourceBundle.BROWSE);	
		csvFileBrowseButton.addSelectionListener(new SelectionAdapter(){
		
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e){
				
				String csvFileName = getNewCSVFilePathFromDialog();
				
				if (csvFileName != null){
					csvFilePathText.setText(csvFileName);
				}
			}
		});

		//Create reusable layout data:
		GridData threeColumnHorizontalFillGridData = GridDataUtil.createHorizontalFill();	
		threeColumnHorizontalFillGridData.horizontalSpan = 3;

		//Create CSV file options group:
		Group csvFileOptionsGroup = new Group(this, SWT.NONE);
		csvFileOptionsGroup.setText(UiPluginResourceBundle.DatapoolWizard_options); 
		csvFileOptionsGroup.setLayout(new GridLayout(3, false));
		csvFileOptionsGroup.setLayoutData(threeColumnHorizontalFillGridData);
		
		//Create CSV file options:
		Label csvFileEncodingLabel = new Label(csvFileOptionsGroup, SWT.NULL);
		csvFileEncodingLabel.setText(UiPluginResourceBundle.WIZ_DATAPOOL_CSV_PG_ENC); 
		
		csvFileEncodingCombo = new Combo(csvFileOptionsGroup, SWT.DROP_DOWN | SWT.READ_ONLY);
		csvFileEncodingCombo.setItems(ENCODINGS);		
		
		csvFileFirstRowVariableNamesButton = new Button(csvFileOptionsGroup, SWT.CHECK);
		csvFileFirstRowVariableNamesButton.setText(UiPluginResourceBundle.WIZ_DATAPOOL_CSV_PG_COL_HEADER); 
		csvFileFirstRowVariableNamesButton.setLayoutData(threeColumnHorizontalFillGridData);

		csvFileFirstColumnEquivalenceClassNamesButton = new Button (csvFileOptionsGroup, SWT.CHECK);
		csvFileFirstColumnEquivalenceClassNamesButton.setText(UiPluginResourceBundle.WIZ_DATAPOOL_CSV_PG_ROW_HEADER); 		
		csvFileFirstColumnEquivalenceClassNamesButton.setLayoutData(threeColumnHorizontalFillGridData);

		if(csvFileCellsEnclosedInXMLTagsButtonText != null){
			
			csvFileCellsEnclosedInXMLTagsButton = new Button (csvFileOptionsGroup, SWT.CHECK);
			csvFileCellsEnclosedInXMLTagsButton.setText(csvFileCellsEnclosedInXMLTagsButtonText); 		
			csvFileCellsEnclosedInXMLTagsButton.setLayoutData(threeColumnHorizontalFillGridData);
		}

		init();
		
		layout(false,true);
	}

	private String getNewCSVFilePathFromDialog() {
		
		FileDialog dialog = new FileDialog(getShell());

		String csvFilePath = csvFilePathText.getText().trim();

		if ((csvFilePath == null) || (csvFilePath.length() == 0) || (!(new File(csvFilePath).exists()))) {
		
			String path = null;
			IDialogSettings settings = getDatapoolDialogSettings();

			if (settings != null) {
				path = settings.get(DATAPOOL_DIR_KEY);
			}
				
			if ((path == null) || (path.length() == 0) || (!(new File(path).exists()))) {
				dialog.setFilterPath(ResourcesPlugin.getWorkspace().getRoot().getLocation().makeAbsolute().toOSString());
			} 
			else {
				dialog.setFilterPath(path);
			}
		} 
		else {
			dialog.setFilterPath(csvFilePath);
		}
		
		dialog.setFilterExtensions(new String[] { "*." + TestUI.CSV_FILE_EXTENSION}); //$NON-NLS-1$
		dialog.setText(UiPluginResourceBundle.WIZ_DATAPOOL_CSV_FILE_OPEN); 

		String newCSVFilePath = dialog.open();
		
		if ((newCSVFilePath != null) && (newCSVFilePath.trim().length() > 0)){
			
			newCSVFilePath = newCSVFilePath.trim();
			
			//Append the file extension, if required:
			//Note: The CSV file extension (TestUI.CSV_FILE_EXTENSION) is in lower-case.
			if(!newCSVFilePath.toLowerCase(Locale.ENGLISH).endsWith("." + TestUI.CSV_FILE_EXTENSION)){ //$NON-NLS-1$
				
				if(!newCSVFilePath.endsWith(".")){ //$NON-NLS-1$
					newCSVFilePath += "."; //$NON-NLS-1$
				}
				
				newCSVFilePath += TestUI.CSV_FILE_EXTENSION;
			}
			
			IDialogSettings settings = getDatapoolDialogSettings();
			
			if (settings != null) {
				settings.put(DATAPOOL_DIR_KEY, new Path(newCSVFilePath).removeLastSegments(1).makeAbsolute().toOSString());
			}
		}
		
		return newCSVFilePath;
	}

	private IDialogSettings getDatapoolDialogSettings() {
		
		IDialogSettings pluginDialogSettings = UiPlugin.getDefault().getDialogSettings();
		IDialogSettings dataPoolDialogSettings = pluginDialogSettings.getSection(DATAPOOL_SECTION_KEY);
		
		if (dataPoolDialogSettings == null) {
			dataPoolDialogSettings = pluginDialogSettings.addNewSection(DATAPOOL_SECTION_KEY);
		}
		
		return dataPoolDialogSettings;
	}
}
