/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolAttributeWizardPage.java,v 1.9 2010/03/31 19:18:09 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import org.eclipse.hyades.test.ui.datapool.internal.dialog.DatapoolConstants;
import org.eclipse.hyades.test.ui.datapool.internal.util.CSVImportExportUtil;
import org.eclipse.hyades.test.ui.datapool.internal.util.CorruptCSVFileException;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.internal.wizard.AttributeWizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Spinner;

/**
 * <p>Wizard page that allows the user to define the description, initial
 * dimension, and CSV file to append to the new datapool.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Paul Slauenwhite
 * @version March 31, 2010
 * @since   March 3, 2005
 */
public class DatapoolAttributeWizardPage extends AttributeWizardPage {
 	
	private Spinner columnSpinner = null;
	private Spinner rowSpinner = null;
	private Button importCSVFileButton = null;
	private CSVFileSelectionComposite csvFileSelectionComposite = null;

	/**
	 * @param pageName
	 */
	public DatapoolAttributeWizardPage(String pageName) {
		super(pageName);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.wizard.AttributeWizardPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		
		super.createControl(parent);
		
		Group dimensionGroup = new Group(((Composite)(getControl())), SWT.NONE);
		dimensionGroup.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		dimensionGroup.setLayout(new GridLayout(2, false));
		dimensionGroup.setText(UiPluginResourceBundle.DatapoolNewWizard_InitialTableDimension); 
				
		Label columnNumberLabel = new Label(dimensionGroup, SWT.NONE);
		columnNumberLabel.setText(UiPluginResourceBundle.DatapoolNewWizard_numberOfColumns);
		
		initializeDialogUnits(dimensionGroup);

		GridData gridData = new GridData();
		gridData.widthHint = convertWidthInCharsToPixels(Math.max(String.valueOf(DatapoolConstants.MAXIMUM_VARIABLE_LIMIT).length(), String.valueOf(Integer.MAX_VALUE).length()) + 2);
		
		columnSpinner = new Spinner(dimensionGroup, SWT.BORDER);
		columnSpinner.setLayoutData(gridData);
		columnSpinner.setValues(1, 1, DatapoolConstants.MAXIMUM_VARIABLE_LIMIT, 0, 1, 10);
		
		Label rowNumberLabel = new Label(dimensionGroup, SWT.NONE);
		rowNumberLabel.setText(UiPluginResourceBundle.DatapoolNewWizard_numberOfRows); 
		
		rowSpinner = new Spinner(dimensionGroup, SWT.BORDER);
		rowSpinner.setLayoutData(gridData);
		rowSpinner.setValues(1, 0, Integer.MAX_VALUE, 0, 1, 10);
		
		importCSVFileButton = new Button(((Composite)(getControl())), SWT.CHECK);
		importCSVFileButton.setText(UiPluginResourceBundle.DatapoolNewWizard_importCSVFile); 	
		
		//Indent the CSV file selection composite:
		gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.horizontalIndent = 12;
		
		GridLayout gridLayout = new GridLayout(3, false);
		gridLayout.marginWidth = 0;
		gridLayout.marginHeight = 0;
		
		SelectionAdapter selectionAdapter = new SelectionAdapter(){

			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e){
				validateSelection();
			}
		};
		
		csvFileSelectionComposite = new CSVFileSelectionComposite(((Composite)(getControl())), SWT.NONE);
		csvFileSelectionComposite.setLayoutData(gridData);
		csvFileSelectionComposite.setLayout(gridLayout);
		csvFileSelectionComposite.setVisible(false);
		csvFileSelectionComposite.getCSVFileFirstRowVariableNamesButton().addSelectionListener(selectionAdapter);
		csvFileSelectionComposite.getCSVFileFirstColumnEquivalenceClassNamesButton().addSelectionListener(selectionAdapter);
		csvFileSelectionComposite.getCSVFilePathText().addModifyListener(new ModifyListener(){
		
			public void modifyText(ModifyEvent e) {
				validateSelection();
			}
		});
		
		importCSVFileButton.addSelectionListener(new SelectionAdapter(){
			
			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {	
				
				boolean isVisible = importCSVFileButton.getSelection();
				
				csvFileSelectionComposite.setVisible(isVisible);
				
				if(isVisible){
					csvFileSelectionComposite.getCSVFilePathText().setFocus();
				}
				else{
					
					itemDescriptionText.setFocus();
					
					csvFileSelectionComposite.init();
					
					//Reset the error message:
					validateSelection();
				}
			}
		});
	}
	
	/**
	 * @return Returns the numOfColumns.
	 */
	public int getNumOfColumns() {
		return (columnSpinner.getSelection());
	}
	
	/**
	 * @return Returns the numOfRows.
	 */
	public int getNumOfRows() {
		return (rowSpinner.getSelection());
	}	
	
	public String getCSVFilePath(){
		return (csvFileSelectionComposite.getCSVFilePathText().getText());
	}
	
	public boolean getCSVFileFirstRowContainsVariableNames(){
		return (csvFileSelectionComposite.getCSVFileFirstRowVariableNamesButton().getSelection());
	}
	
	public boolean getCSVFileFirstColumnContainsEquivalenceClassNames(){
		return (csvFileSelectionComposite.getCSVFileFirstColumnEquivalenceClassNamesButton().getSelection());
	}

	public String getCSVFileEncoding(){		
		return (CSVFileSelectionComposite.ENCODING_VALUES[csvFileSelectionComposite.getCSVFileEncodingCombo().getSelectionIndex()]);
	}
		
	/**
	 * <p>Validates the CSV file selection.</p>
	 * 
	 * <p>The CSV file selection is valid if either 
	 * of the following conditions are true:</p>
	 * 
	 * <ul>
	 * <li>The CSV file name is <code>null</code> or empty.</li>
	 * <li>The CSV file name (neither <code>null</code> or empty) and the CSV file contents are valid 
	 * (see {@link CSVImportExportUtil#validateCSVFile(String, boolean, boolean, String)}).</li>
	 * </ul>
	 * 
	 * @see CSVImportExportUtil#validateCSVFile(String, boolean, boolean, String)
	 */
	protected void validateSelection(){
		
		//Note: The error message is not initially reset to reduce flicker when setting the same error message.		
		boolean isValid = false;

		if(importCSVFileButton.getSelection()){

			String csvFilePath = getCSVFilePath();		

			if(csvFilePath != null){			

				Shell shell = getShell();

				//Set the wait cursor:
				shell.setCursor(new Cursor(shell.getDisplay(), SWT.CURSOR_WAIT));

				try{

					isValid = CSVImportExportUtil.getInstance().validateCSVFile(csvFilePath, getCSVFileFirstRowContainsVariableNames(), getCSVFileFirstColumnContainsEquivalenceClassNames(), getCSVFileEncoding());

					setErrorMessage(null);	
				}
				catch(CorruptCSVFileException c){			
					setErrorMessage(c.getMessage());
				}		
				finally{

					//Set the default cursor:
					shell.setCursor(null);
				}
			}
		}
		else{
			
			isValid = true;
			
			setErrorMessage(null);	
		}
		
		setPageComplete(isValid);
	}
}
