/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolExportWizard.java,v 1.18 2009/04/03 12:28:34 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.io.File;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.models.common.datapool.util.DatapoolEncryptManager;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.util.CSVImportExportUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IExportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.help.IWorkbenchHelpSystem;

/**
 * <p>DatapoolExportWizard.java</p>
 * 

 * @author  Peter Sun
 * @author  Paul E. Slauenwhite
 * @version April 3, 2009
 * @since   February 1, 2005
 */
public class DatapoolExportWizard extends Wizard implements IExportWizard {

	private DatapoolSelectionPage datapoolSelectionPage = null;
	private DatapoolExportCSVFileLocationPage csvFileLocationPage = null;
	private IStructuredSelection selection = null;
	
	public DatapoolExportWizard()
	{
		super();
		setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_EXPORT_DATAPOOL));
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.ui.IWorkbenchWizard#init(org.eclipse.ui.IWorkbench, org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) 
	{
		setWindowTitle(UiPluginResourceBundle.WIZ_EXP_DATAPOOL_TTL);
		this.selection = selection;
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#createPageControls(Composite)
	 */

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#addPages()
	 */
	public void addPages(){
		
		datapoolSelectionPage = new DatapoolSelectionPage("datapoolSelection", selection);
		datapoolSelectionPage.setTitle(UiPluginResourceBundle.WIZ_EXP_DATAPOOL_TTL);
		datapoolSelectionPage.setDescription(UiPluginResourceBundle.WIZ_EXP_DATAPOOL_SEL_PG_DSC);
		addPage(datapoolSelectionPage);

		csvFileLocationPage = new DatapoolExportCSVFileLocationPage("csvFileSelection");
		csvFileLocationPage.setTitle(UiPluginResourceBundle.WIZ_EXP_DATAPOOL_TTL);
		csvFileLocationPage.setDescription(UiPluginResourceBundle.WIZ_EXP_DATAPOOL_CSV_PG_DSC);
		addPage(csvFileLocationPage);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#createPageControls(org.eclipse.swt.widgets.Composite)
	 */
	public void createPageControls(Composite parent){
		
		super.createPageControls(parent);
		
		//Set the context sensitive help:
		IWorkbenchHelpSystem workbenchHelpSystem = PlatformUI.getWorkbench().getHelpSystem();
		workbenchHelpSystem.setHelp(datapoolSelectionPage.getControl(), (UiPlugin.getID() + ContextIds.EXPORT_DATAPOOL_AS_CSV_FILE_WIZARD));
		workbenchHelpSystem.setHelp(csvFileLocationPage.getControl(), (UiPlugin.getID() + ContextIds.EXPORT_DATAPOOL_AS_CSV_FILE_WIZARD));
	}
	
	public boolean canFinish()
	{
		return csvFileLocationPage.isPageComplete() && datapoolSelectionPage.isPageComplete();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.IWizard#performFinish()
	 */	
	public boolean performFinish() {

		//Assumption: The datapool is not null.
		IDatapool datapool = datapoolSelectionPage.getSelectedDatapool();
		String csvFilePath = csvFileLocationPage.getCSVFilePath();
		boolean isEncrypted = datapoolSelectionPage.isSelectedDatapoolEncrypted();
		String password = datapoolSelectionPage.getEncryptedDatapoolPassword();

		try {

			//Decrypt the datapool if encrypted: 
			if(isEncrypted){		
				DatapoolEncryptManager.decryptDatapool(datapool, password);
			}

			if (new File(csvFilePath).exists()){

				MessageDialog overwriteDialog = new MessageDialog(getShell(),
						UiPluginResourceBundle.WIZ_EXP_DATAPOOL_TTL, 
						null,
						NLS.bind(UiPluginResourceBundle.WIZ_EXP_DATAPOOL_MSG_EXISTS, csvFilePath), 
						MessageDialog.WARNING, 
						new String[] {IDialogConstants.YES_LABEL, IDialogConstants.NO_LABEL}, 
						0);

				//Return if cannot replace existing CSV file:
				if (overwriteDialog.open() != IDialogConstants.OK_ID){
					return false;
				}
			}

			//Export the datapool to the CSV file:
			CSVImportExportUtil.getInstance().exportCSV(datapool, csvFilePath, 
					csvFileLocationPage.getCSVFileFirstRowContainsVariableNames(), 
					csvFileLocationPage.getCSVFileFirstColumnContainsEquivalenceClassNames(), 
					csvFileLocationPage.getCSVFileCellsEnclosedInXMLTags(),
					csvFileLocationPage.getCSVFileEncoding());

			//Refresh the CSV file if exported to the workspace:
			IFile[] files = ResourcesPlugin.getWorkspace().getRoot().findFilesForLocation(new Path(csvFilePath));

			for (int counter = 0; counter < files.length; counter++) {

				try {
					files[counter].refreshLocal(0, null);
				} 
				catch (CoreException e) {
					UiPlugin.logError(e);
				}
			}

			return true;
		}
		catch (Exception e) {
			
			//Log the error:
			UiPlugin.logError(e);			

			//Report the error:
			new ErrorDialog(getShell(), 
							UiPluginResourceBundle.W_ERROR, 
							NLS.bind(UiPluginResourceBundle._ERROR_WIZ_DATAPOOL_CSV_ERRDLG_EXPFILE, new String[] {datapool.getName(), csvFilePath}), 
							new Status(IStatus.ERROR,
									UiPlugin.PLUGIN_ID,
									IStatus.OK,
									e.toString(),
									e),
							IStatus.ERROR).open();			
		}
		finally{

			//Encrypted the datapool if decrypted: 
			if(isEncrypted){		
				DatapoolEncryptManager.encryptDatapool(datapool, password);
			}
		}
		
		return false;
	}
}
