/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolImportPageOne.java,v 1.4 2009/04/03 12:28:34 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import org.eclipse.hyades.test.ui.datapool.internal.util.CSVImportExportUtil;
import org.eclipse.hyades.test.ui.datapool.internal.util.CorruptCSVFileException;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;

/**
 * <p>Datapool import wizard first wizard page to select a CSV file for import 
 * into a new or existing datapool.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version April 3, 2009
 * @since   April 2, 2009
 */
public class DatapoolImportPageOne extends WizardPage {

	private CSVFileSelectionComposite csvFileSelectionComposite = null;
	
	public DatapoolImportPageOne(String pageName){
		super(pageName);
	}	

    /* (non-Javadoc)
     * @see org.eclipse.jface.dialogs.DialogPage#setVisible(boolean)
     */
    public void setVisible(boolean visible) {
        
    	super.setVisible(visible);
        
    	if (visible) {
    		csvFileSelectionComposite.getCSVFilePathText().setFocus();
        }
    }
    
	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent){
		
		SelectionAdapter selectionAdapter = new SelectionAdapter(){

			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e){
				validateSelection();
			}
		};
		
		csvFileSelectionComposite = new CSVFileSelectionComposite(parent, SWT.NONE);
		csvFileSelectionComposite.setLayoutData(GridDataUtil.createHorizontalFill());
		csvFileSelectionComposite.setLayout(new GridLayout(3, false));
		csvFileSelectionComposite.getCSVFileFirstRowVariableNamesButton().addSelectionListener(selectionAdapter);
		csvFileSelectionComposite.getCSVFileFirstColumnEquivalenceClassNamesButton().addSelectionListener(selectionAdapter);
		csvFileSelectionComposite.getCSVFilePathText().addModifyListener(new ModifyListener(){
		
			public void modifyText(ModifyEvent e) {
				validateSelection();
			}
		});
		
		setControl(csvFileSelectionComposite);
		
		setPageComplete(false);
	}
	
	public String getCSVFilePath(){
		return (csvFileSelectionComposite.getCSVFilePathText().getText());
	}
	
	public boolean getCSVFileFirstRowContainsVariableNames(){
		return (csvFileSelectionComposite.getCSVFileFirstRowVariableNamesButton().getSelection());
	}
	
	public boolean getCSVFileFirstColumnContainsEquivalenceClassNames(){
		return (csvFileSelectionComposite.getCSVFileFirstColumnEquivalenceClassNamesButton().getSelection());
	}

	public String getCSVFileEncoding(){		
		return (CSVFileSelectionComposite.ENCODING_VALUES[csvFileSelectionComposite.getCSVFileEncodingCombo().getSelectionIndex()]);
	}
	
	/**
	 * <p>Validates the CSV file selection.</p>
	 * 
	 * <p>The CSV file selection is valid if both 
	 * of the following conditions are true:</p>
	 * 
	 * <ul>
	 * <li>The CSV file name is neither <code>null</code> or empty.</li>
	 * <li>The CSV file contents are valid 
	 * (see {@link CSVImportExportUtil#validateCSVFile(String, boolean, boolean, String)}).</li>
	 * </ul>
	 * 
	 * @see CSVImportExportUtil#validateCSVFile(String, boolean, boolean, String)
	 * @see org.eclipse.hyades.test.ui.internal.wizard.DatapoolCSVImportPage#validateSelection()
	 */
	protected void validateSelection(){
		
		//Note: The error message is not initially reset to reduce flicker when setting the same error message.		
		boolean isValid = false;
		String csvFilePath = getCSVFilePath();		

		if(csvFilePath != null){			

			Shell shell = getShell();
			
			//Set the wait cursor:
			shell.setCursor(new Cursor(shell.getDisplay(), SWT.CURSOR_WAIT));

			try{
				
				isValid = CSVImportExportUtil.getInstance().validateCSVFile(csvFilePath, getCSVFileFirstRowContainsVariableNames(), getCSVFileFirstColumnContainsEquivalenceClassNames(), getCSVFileEncoding());
				
				setErrorMessage(null);					
			}
			catch(CorruptCSVFileException c){
				setErrorMessage(c.getMessage());
			}		
			finally{
					
				//Set the default cursor:
				shell.setCursor(null);
			}			
		}
		else{
			setErrorMessage(null);	
		}
		
		setPageComplete(isValid);
	}
}
