/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolImportWizard.java,v 1.32 2010/05/05 12:32:43 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.URI;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.datapool.util.DatapoolEncryptManager;
import org.eclipse.hyades.models.common.util.DatapoolUtil;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.util.CSVImportExportUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.test.ui.wizard.TestLocationPage;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.help.IWorkbenchHelpSystem;

/**
 * <p>Datapool import wizard to import data from a CSV file to an 
 * existing datapool.</p>
 * 
 * 
 * @author  Bianca Xue Jiang
 * @author  Paul E. Slauenwhite
 * @version April 12, 2010
 * @since   March 18, 2005
 */
public class DatapoolImportWizard extends Wizard implements IImportWizard {

	private DatapoolImportPageOne datapoolImportPageOne = null;
	private DatapoolImportPageTwo datapoolImportPageTwo = null;
	private TestLocationPage testLocationPage = null;
	private IStructuredSelection selection = null;
	
	public DatapoolImportWizard(){
		
		super();
		
		setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_IMPORT_DATAPOOL));
		setWindowTitle(UiPluginResourceBundle.DatapoolImportWizard_title); 
		setNeedsProgressMonitor(true);
	}
	
	public DatapoolImportPageOne getDatapoolImportPageOne(){
		return datapoolImportPageOne;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#canFinish()
	 */
	public boolean canFinish() {
		return (datapoolImportPageOne.isPageComplete() && (datapoolImportPageTwo.isPageComplete() || testLocationPage.isPageComplete()));
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.IWorkbenchWizard#init(org.eclipse.ui.IWorkbench, org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection){
		this.selection = selection;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#addPages()
	 */
	public void addPages(){
		
		datapoolImportPageOne = new DatapoolImportPageOne("importDatapoolPageOne"); //$NON-NLS-1$
		datapoolImportPageOne.setTitle(UiPluginResourceBundle.DatapoolImportWizard_title); 
		datapoolImportPageOne.setDescription(UiPluginResourceBundle.DatapoolImportWizard_description); 		
		addPage(datapoolImportPageOne);	

		datapoolImportPageTwo = new DatapoolImportPageTwo("importDatapoolPageTwo", selection); //$NON-NLS-1$
		datapoolImportPageTwo.setTitle(UiPluginResourceBundle.DatapoolImportWizard_title); 
		datapoolImportPageTwo.setDescription(UiPluginResourceBundle.DatapoolImportWizard_PageTwo_description);
		addPage(datapoolImportPageTwo);

		testLocationPage = new TestLocationPage("locationPage", selection); //$NON-NLS-1$
		testLocationPage.setTitle(UiPluginResourceBundle.DatapoolImportWizard_title); 
		testLocationPage.setDescription(UiPluginResourceBundle.DatapoolImportWizard_LocationPage_description);
		testLocationPage.setFileExtension(ICommonConstants.DATAPOOL_FILE_EXTENSION);
		addPage(testLocationPage);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#createPageControls(org.eclipse.swt.widgets.Composite)
	 */
	public void createPageControls(Composite parent){
		
		super.createPageControls(parent);
		
		//Set the context sensitive help:
		IWorkbenchHelpSystem workbenchHelpSystem = PlatformUI.getWorkbench().getHelpSystem();
		workbenchHelpSystem.setHelp(datapoolImportPageOne.getControl(), (UiPlugin.getID() + ContextIds.IMPORT_CSV_FILE_AS_DATAPOOL_WIZARD));
		workbenchHelpSystem.setHelp(datapoolImportPageTwo.getControl(), (UiPlugin.getID() + ContextIds.IMPORT_CSV_FILE_AS_DATAPOOL_WIZARD));
		workbenchHelpSystem.setHelp(testLocationPage.getControl(), (UiPlugin.getID() + ContextIds.IMPORT_CSV_FILE_AS_DATAPOOL_WIZARD));
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.IWizard#performFinish()
	 */
	public boolean performFinish() {

		IRunnableWithProgress operation = new WorkspaceModifyOperation() {

			public void execute(IProgressMonitor monitor) throws InvocationTargetException {

				monitor.beginTask(UiPluginResourceBundle.DatapoolImportWizard_progressMonitor, 4);

				//Resolve the selected datapool and its password, if encrypted:
				IDatapool datapool = datapoolImportPageTwo.getSelectedDatapool();
				String csvFilePath = datapoolImportPageOne.getCSVFilePath();
				boolean isEncrypted = datapoolImportPageTwo.isSelectedDatapoolEncrypted();
				boolean isReplaceExistingRecords = datapoolImportPageTwo.isReplaceExistingRecords();
				String password = datapoolImportPageTwo.getEncryptedDatapoolPassword();
				
				//Flag to determine if the datapool was decrypted during the import operation:
				boolean isDecrypted = false;
				
				try {

					monitor.worked(1);

					if(datapool != null){

						if(isEncrypted){
							
							DatapoolEncryptManager.decryptDatapool(datapool, password);
							
							isDecrypted = true;
						}

						//Remove all of the records from the equivalence class(es): 
						if(isReplaceExistingRecords){

							for (int equivalenceClassIndex = 0; equivalenceClassIndex < datapool.getEquivalenceClassCount(); equivalenceClassIndex++) {

								IDatapoolEquivalenceClass equivalenceClass = ((IDatapoolEquivalenceClass)(datapool.getEquivalenceClass(equivalenceClassIndex)));
								int recordCount = equivalenceClass.getRecordCount();

								for (int counter = 0; counter < recordCount; counter++) {
									equivalenceClass.removeRecord(0);	
								}
							}
						}
					}    				
					else {

						//Assumption:  The file name will include the file extension (see org.eclipse.tptp.platform.common.ui.wizard.LocationPage.getFileName()).
						String datapoolFileName = testLocationPage.getFileName();

						//Create a new datapool:
						datapool = DatapoolUtil.createNewDatapool(datapoolFileName.substring(0, datapoolFileName.indexOf('.')), null);

						//Create the resource and add the datapool:
						EMFUtil.getResourceFactory(testLocationPage.getFileExtension()).createResource(URI.createPlatformResourceURI(testLocationPage.getContainerFullPath().append(datapoolFileName).toOSString(), false)).getContents().add(((DPLDatapool)(datapool)));
					}

					monitor.worked(1);

					if(datapool.getVariableCount() == 0){

						//Import the CSV file into the datapool:
						CSVImportExportUtil.getInstance().importCSV(datapool, 
								csvFilePath, 
								datapoolImportPageOne.getCSVFileFirstRowContainsVariableNames(),
								datapoolImportPageOne.getCSVFileFirstColumnContainsEquivalenceClassNames(),
								datapoolImportPageOne.getCSVFileEncoding());
					}
					else{

						//Append the CSV file to the datapool:
						CSVImportExportUtil.getInstance().appendFromCSV(datapool, 
								csvFilePath, 
								datapoolImportPageOne.getCSVFileFirstRowContainsVariableNames(),
								datapoolImportPageOne.getCSVFileFirstColumnContainsEquivalenceClassNames(),
								datapoolImportPageOne.getCSVFileEncoding());
						
						//Note: The datapool must be re-encrypted before being saved.
						//Re-encrypt the datapool, if required: 
						if(isDecrypted){
							
	    					DatapoolEncryptManager.encryptDatapool(datapool, password);
	    					
	    					isDecrypted = false;
	    				}
					}

					monitor.worked(1);

					//Save (or serialize) the datapool resource to the workspace:
					EMFUtil.save(((DPLDatapool)(datapool)).eResource());

					//Attempt to open the editor:
					TestUIUtil.openEditor(((DPLDatapool)(datapool)), TestUIExtension.DATAPOOL_EDITOR_PART_ID);

					monitor.worked(1);
				}
				catch(Exception e){

					//Re-encrypt the datapool, if required: 
					if(isDecrypted){						
    					DatapoolEncryptManager.encryptDatapool(datapool, password);
    				}
					
					//Log the error:
					UiPlugin.logError(e);			

					//Report the error:
					new ErrorDialog(getShell(), 
									UiPluginResourceBundle.W_ERROR, 
									NLS.bind(UiPluginResourceBundle._ERROR_WIZ_DATAPOOL_CSV_ERRDLG_IMPFILE, new String[] {csvFilePath, datapool.getName()}), 
									new Status(IStatus.ERROR,
											UiPlugin.PLUGIN_ID,
											IStatus.OK,
											e.toString(),
											e),
									IStatus.ERROR).open();
				}
				finally{
					monitor.done();		
				}
			}
		};

		try {        	
			getContainer().run(false, false, operation);
		} 
		catch (InvocationTargetException e) {

			//Log the error since all exceptions are handled by try/catch in the runnable operation:
			UiPlugin.logError(e);
		} 
		catch (InterruptedException e) {
			//Ignore since the operation cannot be canceled.
		}

		return true;
	}
}
