/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolNewWizard.java,v 1.23 2010/05/05 12:32:45 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.edit.datapool.IDatapoolEquivalenceClass;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.util.DatapoolUtil;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.datapool.internal.util.CSVImportExportUtil;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.test.ui.wizard.TestLocationPage;
import org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.tptp.platform.common.ui.wizard.LocationPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.help.IWorkbenchHelpSystem;

/**
 * <p>Extension of the {@link HyadesNewWizard} wizard for creating datapools.</p>
 * 
 * 
 * @author  Peter Sun
 * @author  Bianca Xue Jiang
 * @author  Tony Wang
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version May 4, 2010
 * @since   February 1, 2005
 */
public class DatapoolNewWizard extends HyadesNewWizard {

	private TestLocationPage testLocationPage = null;
	
	/**
	 * Constructor for DataPoolNewWizard
	 */
	public DatapoolNewWizard()
	{
		super();
		setWindowTitle(UiPluginResourceBundle.WIZ_DATAPOOL_TTL); 
		setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_NEWDATAPOOL));

		IDialogSettings dialogSettings = UiPlugin.getInstance().getDialogSettings();
		IDialogSettings settings = dialogSettings.getSection("DataPoolNewWizard"); //$NON-NLS-1$
		if(settings == null)
			settings = dialogSettings.addNewSection("DataPoolNewWizard"); //$NON-NLS-1$
		setDialogSettings(settings);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard#getLocationPage()
	 */
	protected LocationPage getLocationPage() {
		
		if(testLocationPage == null){

			testLocationPage = new TestLocationPage("datapoolLocation", selection); //$NON-NLS-1$

			String fileExtension = getFileExtension();
			
			if (fileExtension != null) {
				
				testLocationPage.setFileExtension(fileExtension);
				testLocationPage.setFileName(""); //$NON-NLS-1$
			}
		}

		return testLocationPage;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard#initPages()
	 */
	protected void initPages() {
		super.initPages();
		setAttributeWizardPage(new DatapoolAttributeWizardPage("datapool attribute page name")); //$NON-NLS-1$
	}

	/**
	 * Returns the type for the datapool created by this wizard.
	 * @return String
	 */
	protected String getType()
	{
		return TestUIExtension.GENERIC_DATAPOOL_TYPE;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard#getFileExtension()
	 */
	protected String getFileExtension()
	{
		return ICommonConstants.DATAPOOL_FILE_EXTENSION;
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#addPages()
	 */
	public void addPages()
	{
		getLocationPage().setTitle(UiPluginResourceBundle.WIZ_DATAPOOL_GEN_PG_TTL); 
		getLocationPage().setDescription(UiPluginResourceBundle.WIZ_DATAPOOL_GEN_PG_LOC_DSC); 
		addPage(getLocationPage());
		
		getAttributeWizardPage().setTitle(UiPluginResourceBundle.WIZ_DATAPOOL_GEN_PG_TTL); 
		getAttributeWizardPage().setDescription(UiPluginResourceBundle.WIZ_DATAPOOL_GEN_PG_ATT_DSC); 
		addPage(getAttributeWizardPage());		
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#createPageControls(org.eclipse.swt.widgets.Composite)
	 */
	public void createPageControls(Composite parent){
		
		super.createPageControls(parent);
		
		//Set the context sensitive help:
		IWorkbenchHelpSystem workbenchHelpSystem = PlatformUI.getWorkbench().getHelpSystem();
		workbenchHelpSystem.setHelp(getLocationPage().getControl(), (UiPlugin.getID() + ContextIds.NEW_DATAPOOL_WIZARD));
		workbenchHelpSystem.setHelp(getAttributeWizardPage().getControl(), (UiPlugin.getID() + ContextIds.NEW_DATAPOOL_WIZARD));
	}

	/**
	 * @see org.eclipse.hyades.ui.test.internal.wizard.HyadesNewWizard#createObject(org.eclipse.core.resources.IFile)
	 */
	protected boolean createObject(IFile file)
	throws Exception
	{
		Resource resource = createResource(file);
		EObject eObject = createEObject();
		if(eObject == null)
			return false;
		resource.getContents().add(eObject);
		EMFUtil.save(resource);
		return true;
	}
	
	/**
	 * Creates the EObject setting the attributes based on the information
	 * the user has defined in this wizard's pages.
	 * @return a not null EObject
	 */
	protected EObject createEObject(){
		
		DPLDatapool datapool = ((DPLDatapool)(DatapoolUtil.createNewDatapool(getLocationPage().getFileNameWithoutExtension(), getAttributeWizardPage().getItemDescription())));
		int numOfColumns = ((DatapoolAttributeWizardPage)(getAttributeWizardPage())).getNumOfColumns();
		int numOfRows = ((DatapoolAttributeWizardPage)(getAttributeWizardPage())).getNumOfRows();
		String csvFilePath = ((DatapoolAttributeWizardPage)(getAttributeWizardPage())).getCSVFilePath();
		
		if(csvFilePath.trim().length() != 0){
			
			try{

				boolean isRowVariableNameType = ((DatapoolAttributeWizardPage)(getAttributeWizardPage())).getCSVFileFirstRowContainsVariableNames();
				boolean isColEqClsName = ((DatapoolAttributeWizardPage)(getAttributeWizardPage())).getCSVFileFirstColumnContainsEquivalenceClassNames();
				String importEncoding = ((DatapoolAttributeWizardPage)(getAttributeWizardPage())).getCSVFileEncoding();

				//Overwrite and import the CSV if the default dimensions:
				if((numOfColumns == 1) && (numOfRows == 1)){
					CSVImportExportUtil.getInstance().importCSV(datapool, csvFilePath, isRowVariableNameType, isColEqClsName, importEncoding);
				}
				else{
					
					MessageDialog dialog = new MessageDialog(this.getShell(), UiPluginResourceBundle.DatapoolNewWizard_NewDatapool, null, UiPluginResourceBundle.DatapoolNewWizard_DimensionSizeWarning, MessageDialog.QUESTION, new String[]{UiPluginResourceBundle.Word_Yes, UiPluginResourceBundle.Word_No}, 0); 
					
					if(dialog.open() == Window.OK){
						CSVImportExportUtil.getInstance().importCSV(datapool, csvFilePath, isRowVariableNameType, isColEqClsName, importEncoding);
					}
					else{
						CSVImportExportUtil.getInstance().importCSV(datapool, csvFilePath, isRowVariableNameType, isColEqClsName, importEncoding, numOfColumns, numOfRows);
					}
				}
			}
			catch(Exception e){
				
				//Log the error:
				UiPlugin.logError(e);			

				//Report the error:
				new ErrorDialog(getShell(), 
								UiPluginResourceBundle.W_ERROR, 
								NLS.bind(UiPluginResourceBundle._ERROR_WIZ_DATAPOOL_CSV_ERRDLG_IMPFILE, new String[] {csvFilePath, datapool.getName()}), 
								new Status(IStatus.ERROR,
										UiPlugin.PLUGIN_ID,
										IStatus.OK,
										e.toString(),
										e),
								IStatus.ERROR).open();
			}
		}
		else{
			createDatapoolInitialDimension(datapool, numOfColumns, numOfRows);
		}
		
		return datapool;
	}
	
	protected void createDatapoolInitialDimension(IDatapool datapool, int numOfColumns, int numOfRows)
	{
		// create default equivalence class to start with.
		datapool.appendEquivalenceClass(datapool.constructEquivalenceClass());
		datapool.setDefaultEquivalenceClassIndex(0);
		
	    for(int i = 0; i < numOfColumns; i++)
	        datapool.appendVariable(datapool.constructVariable());
		
	    IDatapoolEquivalenceClass eqClass;
	    int index = datapool.getDefaultEquivalenceClassIndex();
		if(index > -1)
		    eqClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(index);
		else if(datapool.getEquivalenceClassCount() > 0)
		{
		    eqClass = (IDatapoolEquivalenceClass)datapool.getEquivalenceClass(0);
		    datapool.setDefaultEquivalenceClassIndex(0);
		}
		else
		{
		    eqClass = datapool.constructEquivalenceClass();
		    datapool.appendEquivalenceClass(eqClass);
		    datapool.setDefaultEquivalenceClassIndex(0);
		}
		
		for(int i = 0; i < numOfRows; i++)
		    eqClass.appendRecord(eqClass.constructRecord());
	}
	
	/**
	 * Creates the EMF resource in which the object is added to.
	 * @return a not null Resource
	 */
	protected Resource createResource(IFile file)
	{
		URI uri = URI.createPlatformResourceURI(file.getFullPath().toString(), false);
		Resource.Factory factory = EMFUtil.getResourceFactory(ICommonConstants.DATAPOOL_FILE_EXTENSION);
		return factory.createResource(uri);
	}

		
	/**
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard#openEditor(org.eclipse.core.resources.IFile)
	 */
	protected void openEditor(IFile file) 
	throws PartInitException
	{
		//Attempt to open the editor:
		TestUIUtil.openEditor(file, TestUIExtension.DATAPOOL_EDITOR_PART_ID);		
	}
}

