/*******************************************************************************
 * Copyright (c) 2008, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: DatapoolSelectionPage.java,v 1.4 2009/04/02 17:29:17 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.edit.datapool.IDatapool;
import org.eclipse.hyades.models.common.datapool.util.DatapoolEncryptManager;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.ui.internal.component.ProxyNodeSelectionViewer;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.navigator.EObjectProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Text;

/**
 * <p>DatapoolSelectionPage.java</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version April 2, 2009
 * @since   October 31, 2008
 */
public class DatapoolSelectionPage extends WizardPage {

	protected ProxyNodeSelectionViewer datapoolSelectionViewer = null;
	protected Group encryptedDatapoolPasswordGroup = null;
	protected Text encryptedDatapoolPasswordText = null;	
	protected IStructuredSelection selection = null;
	private IDatapool selectedDatapool = null;
	private boolean isSelectedDatapoolEncrypted = false;

	public DatapoolSelectionPage(String name){
		this(name, null);
	}
	
	public DatapoolSelectionPage(String name, IStructuredSelection selection){
		
		super(name);
		
		this.selection = selection;		
	}
	
    /* (non-Javadoc)
     * @see org.eclipse.jface.dialogs.DialogPage#setVisible(boolean)
     */
    public void setVisible(boolean visible) {
        
    	super.setVisible(visible);
        
    	if (visible){
    		
    		if(getSelectedDatapool() != null){
    			
    			if(isSelectedDatapoolEncrypted()){
        			encryptedDatapoolPasswordText.setFocus();
    			}
    			else{
    				datapoolSelectionViewer.getViewer().getTree().setFocus();
    			}
    		}    	
    		else {
    			datapoolSelectionViewer.getFilterButton().setFocus();
    		}
        }
    }

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		
		Composite pane = new Composite(parent, SWT.SHADOW_NONE);
		pane.setLayout(new GridLayout());
		pane.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
			    
		datapoolSelectionViewer = new ProxyNodeSelectionViewer(pane, 
				ResourcesPlugin.getWorkspace().getRoot(),
				new String[]{ICommonConstants.DATAPOOL_FILE_EXTENSION},
				(SWT.SINGLE | SWT.FULL_SELECTION | SWT.BORDER)){
			
			public void setSelection(){
				
				super.setSelection();
				
				selectedDatapool = null;
				isSelectedDatapoolEncrypted = false;
				
				IProxyNode[] selectedProxyNodes = datapoolSelectionViewer.getSelectedProxyNodes();

				if ((selectedProxyNodes.length > 0) && (selectedProxyNodes[0] instanceof EObjectProxyNode)) {

					EObject eObject = ((EObjectProxyNode)(selectedProxyNodes[0])).getEObject();

					if(eObject instanceof IDatapool){

						selectedDatapool = ((IDatapool)(eObject));

						//Determine if the selected datapool is encrypted:
						isSelectedDatapoolEncrypted = DatapoolEncryptManager.isDatapoolEncrypted(selectedDatapool);

						if((isSelectedDatapoolEncrypted) && (!encryptedDatapoolPasswordGroup.isVisible())){
						
							//Set the title of the encrypted datapool password group:
							encryptedDatapoolPasswordGroup.setText(NLS.bind(UiPluginResourceBundle.DatapoolExportWizard_password, selectedDatapool.getName()));
	
							//Reset and set focus to the encrypted datapool password:				
							encryptedDatapoolPasswordText.setText("");
							encryptedDatapoolPasswordText.setFocus();				
						}						
					}
				}
				
				//Show or hide the encrypted datapool password group:
				encryptedDatapoolPasswordGroup.setVisible(isSelectedDatapoolEncrypted);						
				
				setPageComplete(validatePage());			
			}
		};
		
		datapoolSelectionViewer.getViewer().setSelection(selection);
				
		encryptedDatapoolPasswordGroup = new Group(pane,SWT.NULL);			
		encryptedDatapoolPasswordGroup.setText(UiPluginResourceBundle.DatapoolExportWizard_password);
		encryptedDatapoolPasswordGroup.setLayoutData(GridDataUtil.createHorizontalFill());
		encryptedDatapoolPasswordGroup.setLayout(new GridLayout());
		encryptedDatapoolPasswordGroup.setVisible(false);

		encryptedDatapoolPasswordText = new Text(encryptedDatapoolPasswordGroup, (SWT.SINGLE | SWT.FULL_SELECTION | SWT.BORDER | SWT.PASSWORD));				
		encryptedDatapoolPasswordText.setLayoutData(GridDataUtil.createHorizontalFill());		
		encryptedDatapoolPasswordText.addModifyListener(new ModifyListener(){
			
			public void modifyText(ModifyEvent e){
				setPageComplete(validatePage());
			}
		});

		//Force the validation of the selection:
		datapoolSelectionViewer.setSelection();

		pane.layout(false, true);
	    				
		setControl(pane);
	}
	
	public IDatapool getSelectedDatapool(){
		return selectedDatapool;
	}
	
	public boolean isSelectedDatapoolEncrypted(){
		return isSelectedDatapoolEncrypted;
	}	

	public String getEncryptedDatapoolPassword(){
		return (encryptedDatapoolPasswordText.getText().trim());
	}

	protected boolean validatePage() {
		
		//Note: The error message is not initially reset to reduce flicker when setting the same error message.		
		boolean isValid = false;
		
		//Validate the selected datapool:
		if(selectedDatapool != null){

			//Validate the selected encrypted datapool and its password:
			if(isSelectedDatapoolEncrypted){

				String password = getEncryptedDatapoolPassword();

				if(password.length() == 0){

					setErrorMessage(NLS.bind(UiPluginResourceBundle.DatapoolExportWizard_encrypted, selectedDatapool.getName()));

					//Set focus to the encrypted datapool password:				
					encryptedDatapoolPasswordText.setFocus();
				}
				else if(!DatapoolEncryptManager.isKeyCorrect(selectedDatapool, password)){

					setErrorMessage(UiPluginResourceBundle.DatapoolExportWizard_wrongpass);

					//Set focus to the encrypted datapool password:				
					encryptedDatapoolPasswordText.setFocus();
				}
				else{					
					isValid = true;
				}
			}
			else{
				isValid = true;
			}
		}
		else{			
			
			//Invalid but reset the error message:
			setErrorMessage(null);			
		}

		if(isValid){
			setErrorMessage(null);			
		}

		return isValid;
	}	
}
