/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ReportExtensionsWizard.java,v 1.9 2010/05/05 12:32:43 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.io.InputStream;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.wizard.TestLocationPage;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.hyades.ui.report.IReportGenerator;
import org.eclipse.hyades.ui.report.ReportGeneratorWizard;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.util.NLS;


/**
 * <p>This wizard is used for the migration from the <code>reportExtensions</code> to the new 
 * <code>reportGenerators</code> extension point.</p>
 * 
 *  
 * @author  Paul E. Slauenwhite
 * @author  Patrick Nedelec
 * @version May 4, 2010
 * @since   March 17, 2005
 */
public class ReportExtensionsWizard extends ReportGeneratorWizard {
    
	private TestLocationPage testLocationPage = null;
    private IReportGenerator generator = null;

    /**
     * 
     */
    public ReportExtensionsWizard(IReportGenerator generator) {
        super();
        setWindowTitle(UiPluginResourceBundle.WIZ_TST_REP_PG_TTL); 
        setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_REPORT));
        this.generator = generator;
    }

    /**
     * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
     */
    protected void initPages() {
    	testLocationPage = new TestLocationPage("location", adjustLocation()); //$NON-NLS-1$
    	testLocationPage.setTitle(UiPluginResourceBundle.WIZ_TST_REP_PG_TTL); 
    	testLocationPage.setDescription(UiPluginResourceBundle.WIZ_TST_REP_PG_LOC_DSC); 
        String fileExtension = getFileExtension();
        if (fileExtension != null) {
        	testLocationPage.setFileExtension(fileExtension);
        	testLocationPage.setFileName("*." + fileExtension); //$NON-NLS-1$
        }
    }

    protected String getFileExtension() {
        return null;
    }

    protected IStructuredSelection adjustLocation() {
        IStructuredSelection structuredSelection = getSelection();
        Object selection = structuredSelection.getFirstElement();
        if (selection instanceof IResource) {
            IResource res = (IResource) selection;
            IContainer parent = res.getParent();
            if ((parent != null) && parent.exists()) {
                structuredSelection = new StructuredSelection(parent);
            }
        }
        return structuredSelection;
    }

    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#addReportPages()
     */
    protected void addReportPages() throws Exception {
        addPage(testLocationPage);
    }

    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#generate(org.eclipse.core.runtime.IProgressMonitor)
     */
    public IFile generate(IProgressMonitor monitor) throws Exception {
        IFile file = null;
        IPath path = testLocationPage.getContainerFullPath();
        IResource res = createContainer(path);

        String fileExtension = TestUI.REPORT_FILE_EXTENSION;
        if (generator != null && generator.getReportFileExtension(getSelection()) != null) fileExtension = generator.getReportFileExtension(getSelection());

        path = new Path(testLocationPage.getFileName() + "." + fileExtension); //$NON-NLS-1$
        file = ((IContainer) res).getFile(path);
        if (file == null) {
            throw new Exception("Exception caught when retrieving the file"); //$NON-NLS-1$
        }

        if (file.exists()) {
            if (!MessageDialog.openQuestion(getShell(), UiPluginResourceBundle.W_QUESTION, 
                    UiPluginResourceBundle.FILE_EXISTS)) 
                return null;
        }

        InputStream inputStream = null;
        try {
            if (generator != null) {
                inputStream = generator.generate(file, getSelection());
            }
        } catch (Exception e) {
            UiPlugin.logError(e);

            String error = e.getLocalizedMessage();
            if ((error == null) || (error.length() == 0)) error = e.getMessage();
            if ((error == null) || (error.length() == 0)) error = e.getClass().getName();

            MessageDialog.openError(getShell(), UiPluginResourceBundle.W_ERROR, 
                    NLS.bind(UiPluginResourceBundle.GENERATE_ERROR_, error)); 
            return null;
        }

        if (inputStream == null) {
            MessageDialog.openError(getShell(), UiPluginResourceBundle.W_ERROR, 
                    UiPluginResourceBundle.REPORT_NO_CONTENT_ERROR_); 
            return null;
        }

        try {
            if (file.exists()) {
                file.delete(true, true, null);
            }
            file.create(inputStream, true, null);
        } catch (CoreException e) {
            UiPlugin.logError(e);
            UIUtil.openSaveFileErrorDialog(getShell(), file.getFullPath().toString(), e);
            return null;
        }
        return file;
    }

    private IContainer createContainer(IPath path) {
        IContainer container = null;
        IWorkspace workbench = ResourcesPlugin.getWorkspace();
        int segCount = path.segmentCount();

        for (int idx = 0; idx < segCount; idx++) {
            if (idx == 0) {
                //project
                IProject project = workbench.getRoot().getProject(path.uptoSegment(idx + 1).toString());
                if (project == null || !project.exists()) {
                    //create the project
                    try {
                        project.create(null);
                    } catch (Exception e) {
                        UiPlugin.logError(e);
                        e.printStackTrace();
                    }
                }
                try {
                    project.open(null);
                } catch (Exception e) {
                }
                container = project;
            } else // (idx > 1)
            { //folder
                IFolder folder = workbench.getRoot().getFolder(path.uptoSegment(idx + 1));
                if (folder == null || !folder.exists()) {
                    //create the folder
                    try {
                        folder.create(false, true, null);
                    } catch (Exception e) {
                        UiPlugin.logError(e);
                        e.printStackTrace();
                    }
                }
                container = folder;
            }
        }
        try {
            container.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        } catch (CoreException exc) {
            UiPlugin.logError(exc);
            exc.printStackTrace();
        }
        return container;
    }

    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#isAvailable(org.eclipse.jface.viewers.ISelection)
     */
    public boolean isAvailable(ISelection selection) {
        return true;
    }

}
