/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ReportWizard.java,v 1.22 2009/10/23 18:15:13 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.ui.internal.report.ReportDescription;
import org.eclipse.hyades.ui.internal.report.ReportManager;
import org.eclipse.hyades.ui.internal.wizard.HyadesWizard;
import org.eclipse.hyades.ui.internal.wizard.selection.IWizardElement;
import org.eclipse.hyades.ui.internal.wizard.selection.WizardElement;
import org.eclipse.hyades.ui.report.IReportDescription;
import org.eclipse.hyades.ui.report.IReportGenerator;
import org.eclipse.hyades.ui.report.ReportGeneratorWizard;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.osgi.framework.Bundle;

/**
 * <p>This wizard allows the user to generate a report from its workbench selection.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Patrick Nedelec
 * @version October 23, 2009
 * @since   February 1, 2005
 */
public class ReportWizard extends HyadesWizard {
    private static String REPORT_WIZARD_KEY = "ReportWizard"; //$NON-NLS-1$

    ReportSelectionWizardPage reportSelectionPage;

    /**
     * Constructor for ReportWizard
     */
    public ReportWizard() {
        super();
        setWindowTitle(UiPluginResourceBundle.WIZ_TST_REP_PG_TTL); 
        setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_REPORT));

        IDialogSettings dialogSettings = UiPlugin.getInstance().getDialogSettings();
        IDialogSettings settings = dialogSettings.getSection(REPORT_WIZARD_KEY);
        if (settings == null) settings = dialogSettings.addNewSection(REPORT_WIZARD_KEY);
        setDialogSettings(settings);
        setForcePreviousAndNextButtons(true);
    }

    protected void initPages() {
        reportSelectionPage = new ReportSelectionWizardPage("report", getWorkbench(), getSelection(), getWizardElements(), UiPluginResourceBundle.ReportWizard_Label); 
        reportSelectionPage.setTitle(UiPluginResourceBundle.WIZ_TST_REP_PG_TTL); 
        reportSelectionPage.setDescription(UiPluginResourceBundle.WIZ_TST_REPP_PG_LOC_DSC); 
    }

    /**
     * @return
     */
    private IWizardElement[] getWizardElements() {
        List wizardElems = new LinkedList();
        try {
            //- Get all the contributors to reportGenerators
            Iterator it = ReportGeneratorsExtensionRegistryReader.getInstance().getWizardCategoryElements().iterator();
            while (it.hasNext()) {
                IConfigurationElement elem = (IConfigurationElement) it.next();
                try {
	                ReportGeneratorWizard wiz = (ReportGeneratorWizard) elem.createExecutableExtension("class"); //$NON-NLS-1$
	                if (wiz.isAvailable(getSelection())) {
	                    WizardElement wizardElement = new WizardElement(wiz); //$NON-NLS-1$                    
	                    wizardElement.setId(elem.getAttribute(ReportGeneratorsExtensionRegistryReader.ID));
	                    wiz.setId(wizardElement.getId());
	                    wizardElement.setLabel(elem.getAttribute(ReportGeneratorsExtensionRegistryReader.NAME));
	                    wizardElement.setDescription(elem.getAttribute(ReportGeneratorsExtensionRegistryReader.DESCRIPTION));
	                    ImageDescriptor imageDescriptor = null;
	                    String symbolicName = elem.getDeclaringExtension().getNamespace();
	                    Bundle bundle = Platform.getBundle(symbolicName);
	                    if (bundle != null && bundle.getState() == Bundle.ACTIVE) {
	                        String icon_path = elem.getAttribute(ReportGeneratorsExtensionRegistryReader.ICON);
	                        if (icon_path != null) {
	                            URL url = new URL(bundle.getEntry("/"), icon_path); //$NON-NLS-1$
	                            imageDescriptor = ImageDescriptor.createFromURL(url);
	                        }
	                    }
	                    wizardElement.setImageDescriptor(imageDescriptor);
	                    wizardElems.add(wizardElement);
	                }
				} catch (Exception e) {
					//if there's a problem with the wizard still process the other wizard extension points
		            UiPlugin.logError(e);
				}
                
            }

            //- Ensure that the old extension point named reportExtensions still read
            ReportManager manager = TestUI.getReportManager();
            IReportDescription[] list = manager.getReportDescriptions();
            for (int idx = 0; idx < list.length; idx++) {
                ReportDescription data = (ReportDescription) list[idx];
                IReportGenerator gen = null;
                gen = data.getReportGenerator();
                if (gen != null && gen.isAvailable(getSelection())) {
                    WizardElement wizardElement = new WizardElement(new ReportExtensionsWizard(gen));
                    wizardElement.setId(data.getName());
                    wizardElement.setLabel(data.getName());
                    wizardElement.setDescription(UiPluginResourceBundle.wizard_Report_Description); 
                    wizardElement.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_REPORT));
                    wizardElems.add(wizardElement);
                }
            }
        } catch (Exception e) {
            UiPlugin.logError(e);
        }
        IWizardElement[] arrayOfWizardElement = new IWizardElement[wizardElems.size()];
        wizardElems.toArray(arrayOfWizardElement);
        return arrayOfWizardElement;
    }

    public void addPages() {
        addPage(reportSelectionPage);
    }

    public boolean performFinish() {
        try {
            FinishOperation op = new FinishOperation();
            getContainer().run(false, false, op);
            IFile reportFile = op.getReportFile();
            reportSelectionPage.saveWidgetValues();
            if (reportFile == null) {
                return false;
            }
            
            //Attempt to refresh the project containing the report file:
            try {
            	reportFile.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
			} 
            catch (CoreException c) {
				//Ignore since the project containing the report file could not be refreshed.
			}

            //Attempt to open the editor:
    		TestUIUtil.openEditor(reportFile, op.getEditorId());
        } catch (InterruptedException e) {
            UiPlugin.logError(e);
            return false;
        } catch (InvocationTargetException e) {
            Throwable exception = e.getTargetException();
            UiPlugin.logError(exception);

            String error = exception.getLocalizedMessage();
            if ((error == null) || (error.length() == 0)) error = exception.getMessage();
            if ((error == null) || (error.length() == 0)) error = exception.getClass().getName();

            MessageDialog.openError(getShell(), UiPluginResourceBundle.W_ERROR, 
                    NLS.bind(UiPluginResourceBundle.GENERATE_ERROR_, error)); 
            return false;
        }
        return true;
    }

    class FinishOperation extends WorkspaceModifyOperation {
        private IFile indexReport = null;
        private String editorid = null;

        /**
         * @see org.eclipse.ui.actions.WorkspaceModifyOperation#execute(org.eclipse.core.runtime.IProgressMonitor)
         */
        protected void execute(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
            try {
                IWizard selectedWizard = reportSelectionPage.getSelectedNode().getWizard();
                if (selectedWizard instanceof ReportGeneratorWizard) {
                    monitor.beginTask("", 1); //$NON-NLS-1$             
                    indexReport = ((ReportGeneratorWizard) selectedWizard).generate(new SubProgressMonitor(monitor, 1, SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
                    editorid = ((ReportGeneratorWizard) selectedWizard).getEditorId();
                }
            } catch (Throwable e) {
                throw new InvocationTargetException(e);
            } finally {
                monitor.done();
            }
        }

        public IFile getReportFile() {
            return indexReport;
        }
        
        public String getEditorId(){
        	return editorid;
        }

    }

}
