/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestCaseDefinitionPage.java,v 1.9 2009/03/08 23:13:47 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.TestOnlyFilter;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.TestProvider;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestSuiteProxyNode;
import org.eclipse.hyades.test.ui.wizard.ITestCaseWizard;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.extension.ITypeValidator;
import org.eclipse.hyades.ui.internal.wizard.selection.IWizardElement;
import org.eclipse.hyades.ui.internal.wizard.selection.WizardElement;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.dialogs.ISelectionStatusValidator;
import org.eclipse.ui.views.navigator.ResourceSorter;

/**
 * <p>A page to define common attributes for the Test Case generation.</p>
 * 
 * 
 * @author  Patrick Nedelec
 * @author  Paul Slauenwhite
 * @version March 8, 2009
 * @since   Janiary 2, 2005
 */
public class TestCaseDefinitionPage extends WizardPage {
    private Text TSNameText;
    private ITestSuite TSElem;

    /**
     * Constructor
     */
    public TestCaseDefinitionPage(String pageName, IWorkbench wkbench, IStructuredSelection selection) {
        super(pageName);
        this.setTitle(UiPluginResourceBundle.WIZ_TST_CASE_DESC_PG_TTL); 
        this.setDescription(UiPluginResourceBundle.WIZ_TST_CASE_DESC_PG_DSC); 

        init(selection);
    }

    private void init(IStructuredSelection selection) {
        Object obj = selection.getFirstElement();
        if (obj instanceof ITestSuiteProxyNode) {
            setTestSuite(((ITestSuiteProxyNode) obj).getTestSuite());
        } else if (obj instanceof IProxyNode) {

        }
    }

    /**
     * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
     */
    public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();
        layout.numColumns = 3;
        composite.setLayout(layout);

        createTestSuiteControl(composite);

        setPageComplete(validatePage());
        setControl(composite);
    }

    private void createTestSuiteControl(Composite composite) {
        Label label = new Label(composite, SWT.LEFT | SWT.WRAP);
        label.setFont(composite.getFont());
        label.setText(UiPluginResourceBundle.WIZ_SEL_TST_SUITE); 
        GridData gd = new GridData();
        label.setLayoutData(gd);

        TSNameText = new Text(composite, SWT.SINGLE | SWT.BORDER);
        TSNameText.setText(getTestSuite() == null ? "" : getTestSuite().getName()); //$NON-NLS-1$
        TSNameText.setEnabled(false);
        TSNameText.setFont(composite.getFont());
        gd = new GridData();
        gd.horizontalAlignment = GridData.FILL;
        gd.grabExcessHorizontalSpace = true;
        TSNameText.setLayoutData(gd);
        TSNameText.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                setPageComplete(validatePage());
            }
        });

        Button browse = new Button(composite, SWT.PUSH);
        browse.setText(UiPluginResourceBundle.BROWSE); 
        browse.addSelectionListener(new SelectionListener() {
            public void widgetSelected(SelectionEvent e) {
                browseTS();
            }

            public void widgetDefaultSelected(SelectionEvent e) {
            }
        });
    }

    private class TestSuiteValidator implements ISelectionStatusValidator {
        private final IStatus errorStatus = new Status(IStatus.ERROR, UiPlugin.getID(), IStatus.ERROR, "", null); 
        private final IStatus errorMessageStatus = new Status(IStatus.ERROR, UiPlugin.getID(), IStatus.ERROR, UiPluginResourceBundle._ERROR_WIZ_SEL_INVALID, null); 
        private final IStatus okStatus = new Status(IStatus.OK, UiPlugin.getID(), IStatus.OK, "", null); //$NON-NLS-1$

        public IStatus validate(Object[] selection) {

        	if(selection.length == 0){
        		return errorStatus;
        	}
        	
        	if((selection.length == 1) && (selection[0] instanceof ITestSuiteProxyNode)){
                return okStatus;        		
        	}
            
            return errorMessageStatus;
        }
    }

    /**
     * Behaviors after clicking the Browse button
     */
    protected void browseTS() {
        TestProvider provider = new TestProvider(new TestOnlyFilter(), false);
        String title = UiPluginResourceBundle.WIZ_SEL_TST_SUITE_DLG_TTL; 
        String message = UiPluginResourceBundle.WIZ_SEL_TST_SUITE_DLG_MSG; 

        ElementTreeSelectionDialog dialog = new ElementTreeSelectionDialog(getShell(), provider, provider);
        dialog.setValidator(new TestSuiteValidator());
        //dialog.addFilter(new TestSuiteFilter());
        dialog.setTitle(title);
        dialog.setMessage(message);
        dialog.setInput(ResourcesPlugin.getWorkspace().getRoot());
        dialog.setSorter(new ResourceSorter(ResourceSorter.NAME));

        ITestSuite ts = null;
        if (dialog.open() == Window.OK) {
            Object[] objects = dialog.getResult();
            if (objects[0] instanceof ITestSuiteProxyNode) {
                ts = ((ITestSuiteProxyNode) objects[0]).getTestSuite();
                setTestSuite(ts);
            }
        }
        if (ts != null) {
            setText(ts.getName());
        }
    }

    private void setText(String name) {
        if (TSNameText != null) TSNameText.setText(name);
    }

    /**
     * @return true if the page is valid
     */
    protected boolean validatePage() {
        return (getTestSuite() != null);
    }

    /**
     * @see org.eclipse.jface.wizard.IWizardPage#canFlipToNextPage()
     */
    public boolean canFlipToNextPage() {
        //when the super is called, the getNextPage() is called...
        return isPageComplete();
    }

    /**
     * @see org.eclipse.jface.wizard.IWizardPage#getNextPage()
     */
    public IWizardPage getNextPage() {
        IWizardPage page = super.getNextPage();
        if (page instanceof TestCaseSelectionPage) {
            final TestCaseSelectionPage selPage = (TestCaseSelectionPage) page;
            if (isCurrentPage()) {
                HashMap testCaseTypesByTestSuiteType = new HashMap();
                ITestSuite testSuite = getTestSuite();
                if (testSuite == null) return page;

                String testSuiteType = testSuite.getType();
                if (testSuiteType == null) return page;

                Set testCaseTypes = new HashSet();
                testCaseTypesByTestSuiteType.put(testSuiteType, testCaseTypes);

                IAssociationMapping testSuiteTypeValidatorMapping = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_VALIDATORS);
                IAssociationDescriptor[] typeValidatorDescriptors = testSuiteTypeValidatorMapping.getAssociationDescriptors(testSuiteType);
                if (typeValidatorDescriptors.length == 0) super.getNextPage();

                IAssociationMapping testCaseWizardMapping = TestUIExtension.getTestCaseMappingRegistry().getAssociationMapping(UiPlugin.EP_TEST_CASE_WIZARD);
                String[] mappingTypes = testCaseWizardMapping.getTypes();

                for (int i = 0, maxi = mappingTypes.length; i < maxi; i++) {
                    for (int j = 0, maxj = typeValidatorDescriptors.length; j < maxj; j++) {
                        ITypeValidator typeValidator = (ITypeValidator) typeValidatorDescriptors[j].createImplementationClassInstance();
                        if (typeValidator.isValidType(mappingTypes[i])) {
                            testCaseTypes.add(mappingTypes[i]);
                            break;
                        }
                    }

                }
                List wizardElements = new ArrayList(testCaseTypes.size());
                for (Iterator i = testCaseTypes.iterator(); i.hasNext();) {
                    String testCaseType = (String) i.next();
                    IAssociationDescriptor[] wizardDescriptors = testCaseWizardMapping.getAssociationDescriptors(testCaseType);
                    for (int j = 0, maxj = wizardDescriptors.length; j < maxj; j++) {
                        Object wizard = wizardDescriptors[j].createImplementationClassInstance();
                        if (wizard instanceof ITestCaseWizard) {
                            WizardElement wizardElement = new WizardElement((ITestCaseWizard) wizard);
                            wizardElement.setId(wizardDescriptors[j].getId());
                            wizardElement.setLabel(wizardDescriptors[j].getName());
                            wizardElement.setDescription(wizardDescriptors[j].getDescription());
                            wizardElement.setImageDescriptor(wizardDescriptors[j].getImageDescriptor());

                            wizardElements.add(wizardElement);
                        }
                    }
                }
                selPage.setTestSuite(getTestSuite());
                selPage.setWizardElements((IWizardElement[]) wizardElements.toArray(new IWizardElement[wizardElements.size()]));
            }
        }
        return page;
    }

    /**
     * @return Returns the tSElem.
     */
    public ITestSuite getTestSuite() {
        return this.TSElem;
    }

    /**
     * @param elem
     *            The tSElem to set.
     */
    public void setTestSuite(ITestSuite elem) {
        this.TSElem = elem;
    }
}
