/*******************************************************************************
 * Copyright (c) 2007, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.HashMap;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.execution.harness.ITestExecutionHarnessOptions;
import org.eclipse.hyades.execution.harness.XMLExecutionDataProcessor;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;

/**
 * <p>Test log import wizard.</p>
 * 
 * 
 * @author  Joseph P. Toomey
 * @author  Paul E. Slauenwhite
 * @version May 21, 2010
 * @since   March 21, 2007
 */
public class TestLogImportWizard extends Wizard implements IImportWizard {
	
	TestLogImportWizardPage mainPage;

	public TestLogImportWizard() {
		super();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.Wizard#performFinish()
	 */
	public boolean performFinish() {
		final IFile modelTarget = mainPage.createNewFile();
		final File fragmentFile = mainPage.getFragmentFile();
		final boolean database = mainPage.isDatabase();
		final IFile testSuite = mainPage.getTestSuite();
		
		Job importJob = new Job(UiPluginResourceBundle.TestLogImportWizard_IMPORT_JOB_NAME) {

			protected IStatus run(IProgressMonitor monitor) {

				boolean result = importTestLog(fragmentFile, modelTarget, testSuite, database, monitor);
				try {
					modelTarget.delete(true, monitor);
				} catch (CoreException e) {
					// We create an unnecessary workspace copy of the fragment file
					// under the new name when we call mainPage.createNewFile().
					// If we can't delete the unnecessary file, there's nothing
					// we can do about it now.  Simply pass through.
				}
								
				if (result == true){
					
					IResource testResultResource = ResourcesPlugin.getWorkspace().getRoot().findMember((modelTarget.getFullPath().toString() + "." + ICommonConstants.EXECUTION_FILE_EXTENSION));

					if ((testResultResource != null) && (testResultResource.getType() == IResource.FILE)) {
						//Attempt to open the editor:
						TestUIUtil.openEditor(((IFile)(testResultResource)), TestUIExtension.EXECUTION_EDITOR_PART_ID);
					}
					
					return new Status(IStatus.OK, UiPlugin.PLUGIN_ID, 0, "", null); //$NON-NLS-1$
				}
				else{
					return new Status(IStatus.OK, UiPlugin.PLUGIN_ID, 0, "", null); //$NON-NLS-1$
				}
			}
			
		};
		importJob.setUser(true);
		importJob.schedule();
        return true;
	}
	 
	/* (non-Javadoc)
	 * @see org.eclipse.ui.IWorkbenchWizard#init(org.eclipse.ui.IWorkbench, org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		
		setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_IMPORT_XML_TEST_LOG));
		setWindowTitle(UiPluginResourceBundle.TestLogImportWizard_PAGE_TITLE); //NON-NLS-1
		setNeedsProgressMonitor(true);
		
		mainPage = new TestLogImportWizardPage(UiPluginResourceBundle.TestLogImportWizard_PAGE_TITLE,selection); //NON-NLS-1
	}
	
	/* (non-Javadoc)
     * @see org.eclipse.jface.wizard.IWizard#addPages()
     */
    public void addPages() {
        super.addPages(); 
        addPage(mainPage);        
    }
    
    protected boolean importTestLog(File fragmentFile, IFile modelTarget, IFile testSuite, boolean isDatabase, IProgressMonitor monitor) {

    	// TODO: Validate input data -- ensure it is valid XML -- report errors
    	// if loader fails.
		boolean isCancelled = false;
    	
		try {
			XMLExecutionDataProcessor processor = new XMLExecutionDataProcessor();
			
			EObject[] roots = EMFUtil.load(null, testSuite);
			ITest test = null;
			try {
				test = (ITest) roots[0];
			} catch (ClassCastException e1) {
				e1.printStackTrace();
				return false;
			} 

			HashMap optionsMap = new HashMap();
			optionsMap.put(ITestExecutionHarnessOptions.DATABASE, Boolean.toString(isDatabase));

			processor.setInitData(test, "localhost", modelTarget.getName(), modelTarget.getParent().getFullPath().toString(), false, null, null, optionsMap); //$NON-NLS-1$
			processor.init();
			
			try {
				long importSize = fragmentFile.length();
				long onePercentWork = importSize/100;
				long currentWork = 0;
				long totalBytes = 0;
				long totalEvents = 0;
				
				monitor.beginTask("", 100); //$NON-NLS-1$
				
				BufferedReader reader = new BufferedReader(new FileReader(fragmentFile));
				
				String event = reader.readLine();
				byte[] bytes;

				// TODO: inject execution start and end tags if not present
				
				while ( event != null )
				{
					if (monitor.isCanceled()) {
						isCancelled = true;
						break;
					}
					bytes = event.getBytes();
					currentWork += bytes.length;
					totalBytes += bytes.length;
					totalEvents++;
					if (currentWork > onePercentWork) {
						monitor.worked(1);
						currentWork -= onePercentWork;
					}
					processor.loadEvent(bytes, bytes.length);
					event = reader.readLine();
				}
				
				// Benchmarking code
//				long endTime = System.currentTimeMillis();
//				duration = endTime - startTime;
//				SimpleDateFormat format = new SimpleDateFormat("HH:mm:ss:SSS"); //$NON-NLS-1$
//				format.setTimeZone(TimeZone.getTimeZone("GMT")); //$NON-NLS-1$
//				System.out.println("Loaded " + totalEvents + " events, " + totalBytes + " bytes in " + format.format(new Date(duration))); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
				
			} catch (Exception e) {
				e.printStackTrace();
			} finally {
				processor.dataServerExited();
			}
		} catch (Throwable ex) {
			ex.printStackTrace();
		}
		finally {
			monitor.done();
		}
    	return !isCancelled;
    }
}
