/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: WizardResourceSelectionPage.java,v 1.11 2010/04/12 12:38:49 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.internal.wizard.selection;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.component.ProxyNodeSelectionDialog;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.jface.wizard.IWizardNode;
import org.eclipse.jface.wizard.WizardSelectionPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWizard;
import org.eclipse.ui.PlatformUI;

/**
 * <p>This is a {@link org.eclipse.jface.wizard.WizardSelectionPage} that selects 
 * one or more resources with a certain file extension in the workspace.</p>
 * 
 * 
 * @author      Paul E. Slauenwhite
 * @author      Bianca Xue Jiang
 * @author		Jerome Bozier
 * @version     April 12, 2010
 * @since       February 1, 2005
 */
public class WizardResourceSelectionPage extends WizardSelectionPage implements SelectionListener, ModifyListener
{
	private IWorkbench workbench;
	private IStructuredSelection structuredSelection;
	private IWizardNode newFileWizardNode;
	private String fileExtension;
	private Button useExistButton;
	private Button createNewButton;
	private Button browseButton;
	private Text existFileText;
	private boolean isValid;
	private List selectedIResources;
	
	/**
	 * @param pageName
	 */
	public WizardResourceSelectionPage(String pageName, IWorkbench workbench, IStructuredSelection structuredSelection, String fileExtension, IWizardNode newFileWizardNode)
	{
		super(pageName);
		this.workbench = workbench;
		this.structuredSelection = structuredSelection;
		this.fileExtension = fileExtension;
		this.newFileWizardNode = newFileWizardNode;
		isValid = false;
		setTitle(UiPluginResourceBundle.DLG_RESOURCE_SEL); 
		setMessage(UiPluginResourceBundle.DLG_RESOURCE_DSC); 
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent)
	{
		
		Group group = new Group(parent, SWT.SHADOW_NONE);
		group.setLayout( new GridLayout(1, false) );

		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, UiPlugin.getID() + ContextIds.RES_WIZ);			
		
		useExistButton = new Button(group, SWT.RADIO);
		useExistButton.setText(UiPluginResourceBundle.DLG_RESOURCE_USE_EXST); 
		useExistButton.setSelection(true);
		useExistButton.addSelectionListener(this);

		Composite browseContainer = new Composite(group, SWT.NONE);
		browseContainer.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		GridLayout browseLayout = new GridLayout();
		browseLayout.numColumns = 3;
		browseContainer.setLayout(browseLayout);
				
		Label label = new Label(browseContainer, SWT.NONE);
		label.setText(UiPluginResourceBundle.DLG_RESOURCE_EXST); 
		existFileText = new Text(browseContainer, SWT.BORDER);
		existFileText.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		//existFileText.setEditable(false);
		existFileText.addModifyListener(this);
		browseButton = new Button(browseContainer, SWT.PUSH);
		browseButton.setText(UiPluginResourceBundle.BTN_BROWSE); 
		browseButton.addSelectionListener(this);
		
		createNewButton = new Button(group, SWT.RADIO);
		createNewButton.setText(UiPluginResourceBundle.DLG_RESOURCE_NEW); 
		createNewButton.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL));
		createNewButton.addSelectionListener(this);
		
		setControl(group);
	}
	
	/**
	 * @see org.eclipse.jface.dialogs.DialogPage#setVisible(boolean)
	 */
	public void setVisible(boolean visible) {
		
		super.setVisible(visible);
		
		if (visible) {
			existFileText.setFocus();
		}
	}
	
	protected IWizardNode getNewFileWizardNode()
	{
		if((newFileWizardNode != null) && (!newFileWizardNode.isContentCreated()))
		{
			IWizard wizard = newFileWizardNode.getWizard();
			if(wizard instanceof IWorkbenchWizard)
			{
				((IWorkbenchWizard)wizard).init(workbench, structuredSelection);
				wizard.addPages();
			}
		}
		return newFileWizardNode;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e)
	{
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e)
	{
		if(e.widget == useExistButton)
		{
			if(isValid){
				setPageComplete(true);
			}
			existFileText.setEnabled(true);
			browseButton.setEnabled(true);
			browseButton.setFocus();
			super.setSelectedNode(null);
		}
		else if(e.widget == createNewButton)
		{
			setPageComplete(false);
			existFileText.setEnabled(false);
			browseButton.setEnabled(false);
			super.setSelectedNode(getNewFileWizardNode());
		}
		else if(e.widget == browseButton)
		{
		   
			ProxyNodeSelectionDialog proxyNodeSelectionDialog = new ProxyNodeSelectionDialog(Display.getCurrent().getActiveShell(), 
					UiPluginResourceBundle.DLG_RESOURCE_SEL, 
					new String[]{fileExtension}, 
					SWT.RESIZE); 

			if(proxyNodeSelectionDialog.open() == Window.OK){
			
				IProxyNode[] selectedProxyNodes = proxyNodeSelectionDialog.getSelectedProxyNodes();
				
				if(selectedProxyNodes.length > 0){

					StringBuffer selectedPaths = new StringBuffer();
					selectedIResources = new ArrayList();
					
					for (int counter = 0; counter < selectedProxyNodes.length; counter++) {
						
						//Resolve the selected file:
						IResource resource = selectedProxyNodes[counter].getUnderlyingResource();
						
						selectedIResources.add(resource);

						if(selectedPaths.length() > 0){
							selectedPaths.append(", "); //$NON-NLS-1$
						}

						selectedPaths.append(resource.getFullPath().toString());
					}

					existFileText.removeModifyListener(this);
					existFileText.setText(selectedPaths.toString());
					
					isValid = true;
					
					setPageComplete(true);
					
					existFileText.addModifyListener(this);
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e){
		
		if(e.widget == existFileText){

			//Validate:
			selectedIResources = new ArrayList();
			
			String fileText = existFileText.getText();			

			if((fileText != null) && (fileText.length() > 0)){
				
				IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
				StringTokenizer tokenizer = new StringTokenizer(fileText, ","); //$NON-NLS-1$
			
				while(tokenizer.hasMoreTokens()){
					
					String path = tokenizer.nextToken().trim();
					
					if(path.length() > 0){
					
						IResource resource = root.findMember(path);
						
						if((resource == null) || (!resource.exists()) || (!fileExtension.equals(resource.getFileExtension()))){
														
							isValid = false;
							
							setPageComplete(isValid);
							
							setErrorMessage(path.concat(UiPluginResourceBundle._ERROR_DLG_RESOURCE_INVLD)); 
							
							return;
						}
						else{
							selectedIResources.add(resource);
						}
					}
				}				
			}
			
			isValid = (selectedIResources.size() > 0);
			
			setPageComplete(isValid);
			
			setErrorMessage(null);	
		}
	}

	public EObject[] getSelectedEObjects(){
		
		if(selectedIResources == null){
			return (new EObject[0]);
		}
		
		//Note: Loading the EObjects is deferred to improve validation performance (see modifyText(ModifyEvent)).
		List selectedEObjects = new ArrayList();
		Iterator selectedIResourcesIterator = selectedIResources.iterator();
		
		while(selectedIResourcesIterator.hasNext()){

			EObject[] eObjects = EMFUtil.getEObjects(URI.createPlatformResourceURI(((IResource)(selectedIResourcesIterator.next())).getFullPath().toString(), false), true);

			if((eObjects != null) && (eObjects.length > 0)){

				for(int counter = 0; counter < eObjects.length; counter++){
					selectedEObjects.add(eObjects[counter]);
				}
			}		
		}
		
		return ((EObject[])(selectedEObjects.toArray(new EObject[selectedEObjects.size()])));
	}
}
