/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AbstractTabGroup.java,v 1.11 2007/06/04 13:58:50 jptoomey Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.launch.tabGroups;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.ui.AbstractLaunchConfigurationTabGroup;
import org.eclipse.debug.ui.CommonTab;
import org.eclipse.debug.ui.ILaunchConfigurationDialog;
import org.eclipse.debug.ui.ILaunchConfigurationTab;
import org.eclipse.debug.ui.sourcelookup.SourceLookupTab;
import org.eclipse.hyades.test.core.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.trace.ui.internal.launcher.DestinationTab;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileTab;

/**
 * <p>Provides an abstract tab group for TPTP launch configurations.</p>
 * <p>This class can be extended by external plugins to define their own launch
 * configuration type, based on the Test launch configuration type.</p>
 * <p>This tab group is organized into five parts:</p>
 * <ol>
 *   <li>A runnable item selection tab. This tab allows to select the item to launch.
 *       This class does not implement this section. To provide this section, subclasses
 *       are required to implement {@link #addRunnableItemSelectionTab(List, String)}.</li>
 *   <li>The Test Log tab. This tab enables the user to specify options about the Test Log
 *       that will produced during the Test execution. To override this section, subclasses
 *       needs to override {@link #addExecutionHistoryTab(List)}.</li>
 *   <li>Additional tabs providing options specifying how the runnable item
 *       will be run. In this implementation, this section is empty. To override this section,
 *       subclasses need to override {@link #addAdditionalTabs(List, String)}.</li>
 *   <li>Mode-specific tabs. This class allows to add tabs when the launch configuration
 *       displayed in the context of a specific mode. This implementation adds the "Profiling"
 *       tab when the launch configuration dialog is invoked using the "Profile..." command,
 *       and adds the Eclipse "Source Path Lookup" tab when the dialog is invoked using the
 *       "Debug..." command. To override this behavior, subclasses may override these methods:
 *       {@link #addProfilingTab(List, boolean)} and {@link #addDebuggingTab(List)}.</li>
 *   <li>The Eclipse Common Tab. To override this section, subclasses need to override
 *   {@link #addCommonTab(List)}.</li>
 * </ol>
 * @author jcanches
 */
/**
 * @author paul
 *
 */
public abstract class AbstractTabGroup extends AbstractLaunchConfigurationTabGroup {

	private boolean userActionsPerfomed;
	private TestLaunchConfigurationTabFactory tabFactory;

	/**
	 * Creates the tabs. This method method is not meant to be overrided.
	 * @see AbstractLaunchConfigurationTabGroup#createTabs
	 */
	public void createTabs(ILaunchConfigurationDialog dialog, String mode) {
		tabFactory = new TestLaunchConfigurationTabFactory(this);
		List tabs = new ArrayList(5);
		
		this.addRunnableItemSelectionTab(tabs, mode);
		this.addExecutionHistoryTab(tabs);
		this.addAdditionalTabs(tabs, mode);
		if (this.isProfilingMode(mode)) {
			this.addProfilingTab(tabs, getProfilingAttachOption());
		}
		if (this.isDebugMode(mode)) {
			addDebuggingTab(tabs);
		}
		this.addCommonTab(tabs);
		
		ILaunchConfigurationTab[] array = new ILaunchConfigurationTab[tabs.size()];
		tabs.toArray(array);
		setTabs(array);
	}
	
	/**
	 * Is the supplied mode the Profiling mode. This method is a conveniency method
	 * for the use of sub-classes and should not be overrided.
	 * @param mode
	 * @return
	 */
	protected boolean isProfilingMode(String mode) {
		return mode.equals(ILaunchManager.PROFILE_MODE);
	}
	
	/**
	 * Is the supplied mode the Debug mode. This method is a conveniency method
	 * for the use of sub-classes and should not be overrided.
	 * @param mode
	 * @return
	 */
	protected boolean isDebugMode(String mode) {
		return mode.equals(ILaunchManager.DEBUG_MODE);
	}
	
	/**
	 * Adds the Runnable Item selection tab (Runnable Item
	 * is typically what is to be run, e.g. Test(s) or Test Component(s)). This method
	 * must be implemented in the sub-class.
	 * @param tabList
	 */
	abstract protected void addRunnableItemSelectionTab(List tabList, String mode);
	
	/**
	 * Adds the Test Log related tabs.
	 * This implementation adds the "Test Log" tab to the list. This method may
	 * be overrided in sub-classes.
	 * @param tabList
	 */
	protected void addExecutionHistoryTab(List tabList) {
		tabList.add(tabFactory.newTestLogTab(_getLaunchConfigurationType()));
	}
	
	/**
	 * Adds any additional tabs that do not belong to any other sections.
	 * This implementation does nothing. This method may be overrided in sub-classes.
	 * @param tabList
	 */
	protected void addAdditionalTabs(List tabList, String mode) {
		// No additional tabs
	}
	
	/**
	 * Adds tabs that are specific to the "profile" mode.
	 * This implementation adds the TPTP Profiling tab to the list. This method may
	 * be overrided in sub-classes.
	 * @param tabList The list the tab is to be added to.
	 * @param attach Provides the auto-attach option of the Profiling tab.
	 * @see org.eclipse.hyades.trace.ui.internal.launcher.ProfileTab#ProfileTab(boolean)
	 */
	protected void addProfilingTab(List tabList, boolean attach) {
		tabList.add(new ProfileTab(false));
		tabList.add(new DestinationTab());
	}
	
	/**
	 * Adds tabs that are specific to the "debug" mode.
	 * This implementation adds the JDT Source Path Lookup tab to the list. This method may
	 * be overrided in sub-classes.
	 * @param tabList
	 */
	protected void addDebuggingTab(List tabList) {
		tabList.add(new SourceLookupTab());
	}
	
	/**
	 * Adds Eclise common tab. 
	 * This implementation adds the Eclipse "Common" tab. This method may be overrided,
	 * typically when the common tab should not be displayed.
	 * @param tabList
	 */
	protected void addCommonTab(List tabList) {
		tabList.add(new CommonTab());
	}
	
	/**
	 * Return the attach option for constructing the Profile tab (if applicable).
	 * By default, return true. This method may be redefined by subclasses.
	 * @param launchConfigurationType
	 * @return
	 */
	protected boolean getProfilingAttachOption() {
		return true;
	}
	
	/**
	 * Returns the Launch Configuration Type this tab group applies for.
	 * null is valid and means "Regular TPTP Test Launch Configuration Type".
	 * Subclasses should redefine this method (unless no particular type-based
	 * tab is used).
	 * @return
	 */
	protected String getLaunchConfigurationType() {
		return null;
	}
	
	/**
	 * @deprecated To instantiate a TestAndDeploymentTab, use
	 * {@link TestLaunchConfigurationTabFactory#createTestAndDeploymentTab(String, String)}
	 * instead. To define a tab group that extends the Test launch configuration
	 * tab group, extend the class {@link BasicTestTabGroup} so a call to this method
	 * will become unnecessary.
	 */
	protected void addTestAndDeploymentTab(List tabList, String mode) {
		tabList.add(tabFactory.newTestAndDeploymentTab(mode, _getLaunchConfigurationType()));
	}

	/**
	 * @deprecated To instantiate a TestComponentAndDeploymentTab, use
	 * {@link TestLaunchConfigurationTabFactory#createTestComponentAndDeploymentTab(String, String)}
	 * instead. To define a tab group that extends the Test Component launch configuration
	 * tab group, extend the class {@link BasicTestComponentTabGroup} so a call to this method
	 * will become unnecessary.
	 */
	protected void addTestComponentAndDeploymentTab(List tabList, String mode) {
		tabList.add(tabFactory.newTestComponentAndDeploymentTab(mode, _getLaunchConfigurationType()));
	}

	protected final String _getLaunchConfigurationType() {
		String lcType = getLaunchConfigurationType();
		return lcType == null ? TestLaunchConfigurationFacade.LAUNCH_CONFIGURATION_TYPE : lcType;
	}
	
	/**
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public final void setUserActionsPerformed(boolean state) {
		this.userActionsPerfomed = state;
	}
	
	/**
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public final boolean getUserActionsPerformed() {
		return userActionsPerfomed;
	}
	
	
	/**
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html). 
	 */
	protected TestLaunchConfigurationTabFactory getTabFactory() {
		return tabFactory;
	}
	
}
