/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CMNNamedElementProxyNode.java,v 1.18 2010/02/03 14:01:04 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.navigator;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.MoveModelChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer2;
import org.eclipse.hyades.test.ui.navigator.actions.IRefactoringContext;
import org.eclipse.hyades.test.ui.navigator.actions.RenamerUIInlineEditor;
import org.eclipse.hyades.test.ui.navigator.actions.RenamerUIStatus;
import org.eclipse.hyades.test.ui.navigator.refactoring.EMFRefactoringTransaction;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.osgi.util.NLS;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorReference;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

/**
 * <p>Proxy node for {@link CMNNamedElement}s.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Paul Slauenwhite
 * @version February 3, 2010
 * @since   February 1, 2005
 */
public abstract class CMNNamedElementProxyNode extends EObjectProxyNode implements IProxyNodeRenamer2, IProxyNodeRenamer  {
	
	private final static String TAG_NAME = "name"; //$NON-NLS-1$
	private final static String TAG_DESCRIPTION = "description"; //$NON-NLS-1$

	private String name;
	private String description;

	/**
	 * Constructor to create a {@link CMNNamedElementProxyNode} from a non-<code>null</code> textual name, URI, and parent
	 * and an optional description.
	 * 
	 * @param name The name of the {@link CMNNamedElementProxyNode}.
	 * @param description The optional description of the {@link CMNNamedElementProxyNode}, otherwise <code>null</code>.
	 * @param uri The {@link URI} of the {@link CMNNamedElementProxyNode}.
	 * @param parent An existing object (possibly another proxy node) which is the parent of the newly created {@link CMNNamedElementProxyNode} in the proxy node hierarchy.
	 */
	public CMNNamedElementProxyNode(String name, String description, URI uri, Object parent) {
		
		super(uri, parent);
		
		this.name = name;
		this.description = description;
	}

    /**
     * Constructor used to load an already built proxy.
     * @param memento the saved content of the proxy.
     * @param parent the parent node of this proxy.
     */
	public CMNNamedElementProxyNode(IMemento memento, Object parent) {
		super(memento, parent);
		this.name = memento.getString(TAG_NAME);
		if(this.name == null) {
			throw new IllegalArgumentException("Malformed saved proxy state: unable to retrieve name field"); //$NON-NLS-1$
		}
		this.description = memento.getString(TAG_DESCRIPTION);
	}
	
	/** 
	 * Constructor
	 * @param eObject the EMF named object to convert, the first argument should be a descendant of CMNNamedElement.
	 * @param parent its parent node.
	 */
	public CMNNamedElementProxyNode(EObject eObject, Object parent) {
		super(eObject, parent);
		if(eObject instanceof CMNNamedElement) {
			CMNNamedElement namedElement = (CMNNamedElement)eObject;
			this.name = namedElement.getName();
			this.description = namedElement.getDescription();
		} else {
			throw new IllegalArgumentException("CMNNamedElementProxyNode can only be built upon CMNNamedElement"); //$NON-NLS-1$
		}
	}

	/** Returns the name of the named element given in the constructor. This name is used as the text shown in the navigator.
	 *  @return the name of the proxy.
	 */
	public String getText() {
		return this.name;
	}
	
    /**
     * Setter for name.
     * Normally name is derived directly from the name of the CMNNamedElement, 
     * but in some cases it is useful to overload this behavior, for instance 
     * to decorate the given name with extra information.
     * @param name the new name of the proxy.
     */
	protected void setName(String name) {
		this.name = name;
	}
	
 	/**
 	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
 	 */
 	public Object getAdapter(Class adapter) {
 		if (adapter == IProxyNodeRenamer2.class) {
			return this;
		} if (adapter == IProxyNodeRenamer.class) {
			return this;
		} else {
			return super.getAdapter(adapter);
		}
	}

	/**
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public String getName() {
		return this.name;
	}
	
	/**
	 * Returns the description of the named element.
	 * 
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public String getDescription() {
		return this.description;
	}
	/** 
	 * @see org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer#isApplicableFor()
	 */
	public boolean isApplicableFor() {
		IWorkbenchWindow[] workbenchWindows = PlatformUI.getWorkbench().getWorkbenchWindows();
		for (int i = 0; i < workbenchWindows.length; i++) {
			IWorkbenchWindow window = workbenchWindows[i];
			IWorkbenchPage[] pages = window.getPages();
			for (int j = 0; j < pages.length; j++) {
				IWorkbenchPage page = pages[j];
				IEditorReference[] refs = page.getEditorReferences();
				for (int k = 0; k < refs.length; k++) {
					//- focus only on dirty editor
					if(refs[k].isDirty()) {
						IEditorPart editor = refs[k].getEditor(false);
						if(editor != null) {
							IEditorInput input = editor.getEditorInput();
							if (input instanceof IFileEditorInput) {
								IFile fileEdited = ((IFileEditorInput) input).getFile();
								if(fileEdited.equals(getUnderlyingResource())) {
									//- this element is under edition and has unsaved modification
									//- in this case the rename action is not allowed
									return false;
								}
							}
						}
					}
				}
			}
		}
		return true;
	}
	
    
	/**
	 * @see org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer#performRename(java.lang.String)
	 */
	public boolean performRename(String newName) {
		CMNNamedElement eObject = (CMNNamedElement)getEObject(); 
		Resource res = eObject.eResource();
		if(res == null) {
			res = EMFUtil.getResource(null, eObject);
		}
		if(!eObject.getName().equals(newName)) {
			eObject.setName(newName);
			if(res != null) {
				try {
					EMFUtil.save(res);
				} catch (Exception e) {
					UiPlugin.logError(e);
				}
			}
		}
		//- clear memory
		if(res != null) {
			res.unload();
		}
		return false;
	}
	
    
	/**
	 * @see org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer#performUserInteraction(java.lang.String)
	 * @deprecated
	 */
	public RenamerUIStatus performUserInteraction(String oldName) {
		return new RenamerUIInlineEditor();
	}
	
	public boolean saveState(IMemento memento) {
		if (this instanceof IPersistableProxyNode) {
			//- data from CMNNamedElementProxyNode
			memento.putString(TestUIConstants.TAG_NAME, getText());
			return super.saveState(memento);
		}	
		return false;
	}

	public Change createRenameChange(IRefactoringContext context, String newName) {
		if(getAdapter(IFile.class) != null) {
			IPath oldPath = getUnderlyingResource().getFullPath();
			IPath newPath = oldPath.removeLastSegments(1).append(new Path(newName+'.'+oldPath.getFileExtension()));
			return new RenameModelChange(this, context, newPath, newName);
		} else {
			return new RenameElementChange(this, context, newName);
		}
	}
	
	protected class RenameElementChange extends Change {
		private EMFRefactoringTransaction context;
		private String newName;
		private CMNNamedElementProxyNode proxy;
		
		public RenameElementChange(CMNNamedElementProxyNode node, IRefactoringContext context, String newName) {
			this.proxy = node;
			this.context = (EMFRefactoringTransaction) context.getRefactoringTransaction(EMFRefactoringTransaction.EMF_REFACTORING_TRANSACTION_ID);
			this.newName = newName;
		}
		
		public Change perform(IProgressMonitor pm) throws CoreException {
			if(context != null) {
				EObject object = context.getResourceSet().getEObject(getOriginatorURI(), true);
				((CMNNamedElement)object).setName(newName);
				context.addSavedResource(object.eResource());
			}
			return null;
		}

		public Object getModifiedElement() {
			return proxy;
		}

		public String getName() {
			return NLS.bind(RefactoringMessages.RENAME_PROXY, proxy.getText(), newName);		
		}

		public void initializeValidationData(IProgressMonitor pm) {
		}

		public RefactoringStatus isValid(IProgressMonitor pm) throws CoreException, OperationCanceledException {
			return new RefactoringStatus();
		}
	}
	
	protected class RenameModelChange extends MoveModelChange {
		private IRefactoringContext context;
		private String newName;
		public RenameModelChange(EObjectProxyNode node, IRefactoringContext context, IPath destinationPath, String newName) {
			super(node, context, destinationPath);
			this.context = context;
			this.newName = newName;
		}
		public Change perform(IProgressMonitor pm) throws CoreException {
			Change change = super.perform(pm);
			EMFRefactoringTransaction t = (EMFRefactoringTransaction) context.getRefactoringTransaction(EMFRefactoringTransaction.EMF_REFACTORING_TRANSACTION_ID);
			if(t != null) {
				EObject object = t.getResourceSet().getEObject(getOriginatorURI(), false);
				((CMNNamedElement)object).setName(newName);
			}
			return change;
		}
		
		protected void handleFileAlreadyExists(RefactoringStatus status) {
			//- this is actually a rename but there is a conflict name, this is an error.
			status.addError(NLS.bind(RefactoringMessages.ALREADY_EXISTS_PROXY_DURING_RENAME, destination.toPortableString()));
		}

		public String getName() {
			return NLS.bind(RefactoringMessages.RENAME_PROXY, proxy.getText(), destination.removeFileExtension().lastSegment());		
		}
	}
}
