/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id$
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.navigator;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.facades.behavioral.IBlock;
import org.eclipse.hyades.models.common.facades.behavioral.IDecision;
import org.eclipse.hyades.models.common.facades.behavioral.ILoop;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestInvocation;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.core.launch.configurations.TestLaunchConfigurationFacade;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.EMFResourceProxyFactory;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyNodeCache;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.TypedElementFactoryManager;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.DeleteTestInvocationChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.LaunchConfigurationDeleteChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.LaunchConfigurationUpdateChange;
import org.eclipse.hyades.test.ui.navigator.actions.IRefactoringContext;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.internal.extension.AssociationMappingRegistry;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.IMemento;

/**
 * <p>Default Test suite proxy node. The default behavior of this proxy node is the following:</p>
 * 
 * <ul>
 * <li> when the type of the test suite has been registered by any extension of <code>org.eclipse.hyades.ui.typeDescriptions</code>,
 *  the icon declared in that extension is used, otherwise it uses the default test suite icon, </li>
 * <li> test suite proxy node has children nodes which are proxy nodes of its test cases,</li>
 * <li> test suite proxy node is persisted using the proxy nodes persistency mechanism.</li>
 * </ul>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul E. Slauenwhite
 * @version April 12, 2010
 * @since   February 6, 2006
 */
public class DefaultTestSuiteProxyNode extends TypedElementProxyNode implements ITestSuiteProxyNode, IPersistableProxyNode {

	private IAssociationDescriptor descriptor;
	private CMNNamedElementProxyNode [] testCases;
	
	/**
     * Creates a test suite proxy node based on the EMF test suite element.
	 * @param ts the instance of the test suite.
	 * @param parent the parent element in the test navigator tree.
	 */
	public DefaultTestSuiteProxyNode(TPFTestSuite ts, Object parent) {
		super(ts, parent);
		AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
		IAssociationMapping associationMapping = registry.getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS);
		descriptor = associationMapping.getDefaultAssociationDescriptor(ts.getType());
		LinkedList tcProxies = new LinkedList();
		EList testcases = ts.getTestCases();
		for (Iterator iter = testcases.iterator(); iter.hasNext();) {
			TPFTestCase tc = (TPFTestCase) iter.next();
			IProxyNode proxy = HyadesProxyNodeFactory.getInstance().create(tc, this);
			if(proxy != null) {
				tcProxies.add(proxy);
			}
		}
		testCases = (CMNNamedElementProxyNode[]) tcProxies.toArray(new CMNNamedElementProxyNode[tcProxies.size()]);
		//- add references on associated datapools
		for (Iterator it = ts.getDatapools().iterator(); it.hasNext();) {
			DPLDatapool dp = (DPLDatapool) it.next();
			IProxyNode dpProxy = FileProxyNodeCache.getInstance().getCorrespondingProxy(dp);
			if(dpProxy != null) {
				addBidirectionalReference("Test2Datapool", (IReferencerProxyNode)dpProxy, "Datapool2Test");  //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
		//- add references on invocations
		addInvocationReferences(ts.getImplementor().getBlock());
	}

	private void addInvocationReferences(IBlock block) {
		if (block != null) {
			List actions = block.getActions();
			for (Iterator it = actions.iterator(); it.hasNext();) {
				Object action = it.next();
				if (action instanceof ILoop) {
					addInvocationReferences(((ILoop) action).getBlock());
				} else if (action instanceof IDecision) {
					IDecision ifAction = (IDecision) action;
					addInvocationReferences(ifAction.getFailureBlock());
					addInvocationReferences(ifAction.getSuccessBlock());
				} else if (action instanceof ITestInvocation) {
					ITestInvocation invocationAction = (ITestInvocation) action;
					ITest invokedTest = invocationAction.getInvokedTest();
					if ((invokedTest != null) && (((EObject) invocationAction).eResource() != ((EObject) invokedTest).eResource())) {
						IProxyNode invokedProxy = FileProxyNodeCache.getInstance().getCorrespondingProxy(invokedTest);
						if (invokedProxy != null) {
							addBidirectionalReference("TestInvoker2TestInvoked", (IReferencerProxyNode) invokedProxy, "TestInvoked2TestInvoker"); //$NON-NLS-1$ //$NON-NLS-2$
						}
					}
				}
			}
		}		
	}

    /**
     * Creates a test suite proxy node based on a previously saved proxy node. 
     * @param memento the content of the proxy node persisted.
     * @param parent the parent element in the test navigator tree.
     */
	public DefaultTestSuiteProxyNode(IMemento memento, Object parent) {
		super(memento, parent);
		//- set the descriptor according to the saved type. This is used in getImage() 
		AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
		IAssociationMapping associationMapping = registry.getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS);
		descriptor = associationMapping.getDefaultAssociationDescriptor(getType());

		IMemento [] tcStates = memento.getChildren(TestUIConstants.TAG_CHILD);
		LinkedList tcProxies = new LinkedList();

		for (int i = 0; i < tcStates.length; i++) {
			IMemento mementoChild = tcStates[i];
            //- inherit the TAG_URI_ROOT which is an extra data contained by the parent
            mementoChild.putString(TestUIConstants.TAG_URI_ROOT, memento.getString(TestUIConstants.TAG_URI_ROOT));
			String type = getType();
			if(type == null) {
				throw new IllegalArgumentException("Malformed saved proxy state: unable to retrieve type field"); //$NON-NLS-1$
			}
			ITypedElementProxyFactory factory = TypedElementFactoryManager.getInstance().getFactory(type);
			if(factory instanceof IPersistableTypedElementProxyFactory) {
				IProxyNode proxy = ((IPersistableTypedElementProxyFactory)factory).recreate(mementoChild, this);
				if(proxy != null) {
					tcProxies.add(proxy);
				}
			}
		}
		testCases = (CMNNamedElementProxyNode[]) tcProxies.toArray(new CMNNamedElementProxyNode[tcProxies.size()]);	
	}
	
    /**
     * Returns the icon used in the test navigator to display this test suite proxy node.
     * @return the default test suite node or the icon declared in the extension of <code>org.eclipse.hyades.ui.typeDescriptions</code>.
     */
	public Image getImage() {
		if(descriptor == null) {
			//- the type of this element wasn't registered by any extension (hyades.ui.typeDescriptions) 
			return TestUIImages.INSTANCE.getImage(TestUIImages.IMG_TEST_SUITE);
		} else {
			return descriptor.getImage();
		}
	}

    /**
     * Returns a java array of proxy node that represents children node of the test suite proxy node.
     * Each array element is an instance <code>CMNNamedElementProxyNode</code>.
     * @return test case proxy nodes children of the test suite.
     */
	public IProxyNode[] getChildren() {
		return testCases;
	}
	
    /**
     * Returns the EMF test suite element.
     * @return the test suite element associated to this proxy node. 
     */
	public TPFTestSuite getTestSuite() {
		EObject ts = getEObject();
		if (ts instanceof TPFTestSuite) {
			return (TPFTestSuite) ts;
		} else {
			return null;
		}
	}
	
    /**
     * Returns the ID of the factory that is used to recreate this proxy node.
     * @return the ID of the factory that is used to recreate this proxy node.
     */
	public String getFactoryID() {
		return EMFResourceProxyFactory.ID; 
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.navigator.EObjectProxyNode#getNodeKind()
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	protected String getNodeKind() {
		return TestUIConstants.TESTSUITE_NODE;
	}
	
    /**
     * Saves the content of this proxy node into the given memento.
     * <br>If clients override this method they should override getFactoryID method as well.
     * @param memento the saved state of the proxy node.
     * @return the returned value is no longer significant.
     */
	public boolean saveState(IMemento memento) {
		super.saveState(memento);
		//- children (testcases)
		for (int i = 0; i < testCases.length; i++) {
			CMNNamedElementProxyNode tcProxy = testCases[i];
			if(tcProxy instanceof IPersistableProxyNode) {
				IMemento childMemento = memento.createChild(TestUIConstants.TAG_CHILD);
				if(!((IPersistableProxyNode)tcProxy).saveState(childMemento)) {
					return false;
				}
			} else {
				return false;
			}
		}
		return true;
	}	

	private void launchConfigurationUpdateChanges(CompositeChange composite, URI newURI) {
		ILaunchConfigurationType configsType = TestLaunchConfigurationFacade.getLaunchConfigurationType();
		ILaunchConfiguration[] configs = new ILaunchConfiguration[0];
		try {
			configs = DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurations(configsType);
		} catch (CoreException e) {
			UiPlugin.logError("Unable to get Launch Configuraions", e); //$NON-NLS-1$
		}
		for (int i = 0; i < configs.length; i++) {
			try {
				URI originatorURI = getOriginatorURI();
				if (TestLaunchConfigurationFacade.isTestURI(configs[i], originatorURI)) {
					composite.add(new LaunchConfigurationUpdateChange(configs[i], newURI.appendFragment(originatorURI.fragment())));;
				}
			} catch (CoreException e) {
				UiPlugin.logError("Unable to update Launch Configuraions", e); //$NON-NLS-1$
			}		
		}
	}

	private void launchConfigurationDeleteChanges(CompositeChange composite) {
		ILaunchConfigurationType configsType = TestLaunchConfigurationFacade.getLaunchConfigurationType();
		ILaunchConfiguration[] configs = new ILaunchConfiguration[0];
		try {
			configs = DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurations(configsType);
		} catch (CoreException e) {
			UiPlugin.logError("Unable to get Launch Configuraions", e); //$NON-NLS-1$
		}
		for (int i = 0; i < configs.length; i++) {
			try {
				URI originatorURI = getOriginatorURI();
				if (TestLaunchConfigurationFacade.isTestURI(configs[i], originatorURI)) {
					composite.add(new LaunchConfigurationDeleteChange(configs[i]));
				}
			} catch (CoreException e) {
				UiPlugin.logError("Unable to delete Launch Configuraions", e); //$NON-NLS-1$
			}		
		}
	}	

	public Change createRenameChange(IRefactoringContext context, String newName) {
		CompositeChange composite = new CompositeChange(""); //$NON-NLS-1$
		Change superChange = super.createRenameChange(context, newName);
		if (superChange != null && superChange instanceof RenameModelChange) {
			launchConfigurationUpdateChanges(composite, ((RenameModelChange)superChange).getNewURI());
			composite.add(superChange);
		}
		composite.markAsSynthetic();
		return composite;
	}

	public Change createMoveChange(IRefactoringContext context, IPath destinationPath) {
		CompositeChange composite = new CompositeChange(""); //$NON-NLS-1$
		launchConfigurationUpdateChanges(composite, URI.createPlatformResourceURI(destinationPath.toPortableString(), false));
		Change superChange = super.createMoveChange(context, destinationPath);
		if (superChange != null) {
			composite.add(superChange);
		}
		composite.markAsSynthetic();
		return composite;
	}

	public Change createUpdateChange(IRefactoringContext context, IContainer container, IPath destinationPath) {
		CompositeChange composite = new CompositeChange(""); //$NON-NLS-1$
		launchConfigurationUpdateChanges(composite, URI.createPlatformResourceURI(destinationPath.toPortableString(), false));
		Change superChange = super.createUpdateChange(context, container, destinationPath);
		if (superChange != null) {
			composite.add(superChange);
		}
		composite.markAsSynthetic();
		return composite;
	}

	public Change createUpdateChange(IRefactoringContext context, IReferencerProxyNode referenced, String refType, IPath destinationPath) {
		if(destinationPath == null) {
			if("TestInvoked2TestInvoker".equals(refType)) { //$NON-NLS-1$
				//- remove the link to referenced in this (clean invocation)
				return new DeleteTestInvocationChange(context, this, referenced);
			}
		}
		return super.createUpdateChange(context, referenced, refType, destinationPath);
	}

	public Change createDeleteChange(IRefactoringContext context) {
		CompositeChange composite = new CompositeChange(""); //$NON-NLS-1$
		launchConfigurationDeleteChanges(composite);
		Change superChange = super.createDeleteChange(context);
		if(superChange != null) {
			composite.add(superChange);
		}
		composite.markAsSynthetic();
		return composite;
	}

}
