/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EObjectProxyNode.java,v 1.17 2010/04/12 12:38:50 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.navigator;

import java.util.Collection;
import java.util.Set;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.EMFResourceProxyFactory;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyManager.IUnboundedParent;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.IExtendedProxyNode;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceRegistry;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.DeleteModelElementChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.ModelPathChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.MoveModelChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.PasteModelChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.UpdateModelChange;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeDeleter;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeMover;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodePaster;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeUpdater;
import org.eclipse.hyades.test.ui.navigator.actions.IRefactoringContext;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ui.IMemento;

/**
 * <p>Proxy node for {@link EObject}s.</p>
 * 
 * <p>Subclasses should override the implementation of the {@link IProxyNode} methods.</p>
 * 
 * <p>EMF objects wrapped by subclasses will be stored in the Test Navigator's resource set.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Paul Slauenwhite
 * @author	Jerome Bozier
 * @version April 12, 2010
 * @since   February 4, 2005
 */
public abstract class EObjectProxyNode extends AbstractProxy implements IPersistableProxyNode, IExtendedProxyNode, 
	IReferencerProxyNode, IProxyNodeMover, IProxyNodeUpdater, IProxyNodeDeleter,IProxyNodePaster {
	
	private static final String TAG_URI_FRAGMENT = "uriFragment"; //$NON-NLS-1$
		
	private URI originatorURI;
	private Object parent;
	
	/**
	 * Constructor to create a {@link EObjectProxyNode} from a non-<code>null</code> URI and parent.
	 * 
	 * @param uri The {@link URI} of the {@link EObjectProxyNode}.
	 * @param parent An existing object (possibly another proxy node) which is the parent of the newly created {@link EObjectProxyNode} in the proxy node hierarchy.
	 */
	public EObjectProxyNode (URI uri, Object parent) {

		this.originatorURI = uri;
		this.parent = parent;
	}
	
	/**
     * Constructor used to load a persisted proxy. 
     * @param memento the memento containing the saved state of the proxy.
     * @param parent the parent node of the proxy node.
     */
	public EObjectProxyNode(IMemento memento,  Object parent) {
        String uriFragmentName;
        //- backward compatibility of the existing persisted proxy node states
        String uri = memento.getString(TestUIConstants.TAG_URI);
        if(uri != null) {
            URI oldURI = URI.createURI(uri);
            //- get the fragment part from the whole uri
            uriFragmentName = oldURI.fragment();
        } else {
            uriFragmentName = memento.getString(TAG_URI_FRAGMENT);
        }
		//- this data is not persisted but added by the proxy node loader (FileProxyNodeCache#buildProxyFromSavedState)
        String uriRootName = memento.getString(TestUIConstants.TAG_URI_ROOT);
        try {
            URI uriRoot = URI.createPlatformResourceURI(uriRootName, false);
            this.originatorURI = uriRoot.appendFragment(uriFragmentName);
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Malformed saved proxy state: unable to retrieve uri field"); //$NON-NLS-1$        }
        }
 		this.parent = parent;
	}

	/**
	 *  Creates a proxy node from a emf object.
	 *  CAUTION: this object should be stored in a EMF resource in order to get its URI
	 * @param eObject the originator object of the proxy
	 * @param parent the parent of this node
	 */
	public EObjectProxyNode (EObject eObject, Object parent) {
		originatorURI = EcoreUtil.getURI(eObject);
		this.parent = parent;
	}
	
	/**
	 * <p>Loads the EMF {@link EObject} that is associated with this proxy.</p>
	 * 
	 * <p><b>Note:</b> This method loads the EMF {@link EObject} using a new 
	 * {@link ResourceSet}, which <i>may</i> be a time consuming operation 
	 * for large resources (for example, datapools).</p>
	 * 
	 * @return EMF {@link EObject} that is associated with this proxy.
	 */
	public EObject getEObject() {
		return EMFUtil.getEObject(new ResourceSetImpl(), originatorURI, true);
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getParent()
	 */
	public Object getParent() {
		return parent;
	}
	
    /**
     * Returns the URI of the EObject that this proxy node is associated to. 
     * @return the URI of the EObject that this proxy node is associated to. 
     */
	public URI getOriginatorURI() {
		return originatorURI;
	}
	
	/** 
     * Identifier for an EMF object is its URI fragment part. 
     * If there is the fragment identifier is empty, this means that the object is a root object.  
	 * @return the identifier of the proxy node.
	 */
	public String getIdentifier() {
		String fragment = originatorURI.fragment();
		if(fragment != null) {
			return fragment;
		} else {
			return ""; //$NON-NLS-1$
		}
	}

    /**
     * Returns the physical resource associated to the given URI.
     * @param uri an URI
     * @return the physical resource associated to the given URI.
     */
    private IResource getWorkspaceResource(URI uri) {
        String uriPath = uri.trimFragment().toString();
        if(uriPath.startsWith("platform:/resource")) { //$NON-NLS-1$
            String path = uriPath.substring("platform:/resource".length()); //$NON-NLS-1$
            return ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(path));
        } else {
            return null;
        }
    }
    
    
	/**
	 * Underlying resource of an EMF object is the file containing this object.
     * @return the physical resource associated to this proxy node.
	 */
	public IResource getUnderlyingResource() {
		return getWorkspaceResource(originatorURI);
	}

	/** 
 	 * For performance reason, EObjectProxyNode instances are not, by default, adaptable in EObject.
     * Adaptation is delagated to the plateform adpater manager.
     * This method is not intended to be subclassed by clients.
     * @retrun an adapted object for the given class using the global plateform adapter manager.
	 */
	public Object getAdapter(Class adapter) {
		//- this proxy is adaptable in IFile if it has been placed instead of a file
		if(adapter.isAssignableFrom(IFile.class) && getParent() instanceof IUnboundedParent) {
		    return getUnderlyingResource();
 		} else if(adapter == IProxyNodeUpdater.class 
 			|| adapter == IProxyNodeMover.class 
 			|| adapter == IProxyNodeDeleter.class
 			|| adapter == IProxyNodePaster.class) {
 			return this;
	 	} else {
			return Platform.getAdapterManager().getAdapter(this, adapter);
		}
	}

	public void addReference(String refType, IProxy proxyRef) {
		IResource ur = proxyRef.getUnderlyingResource();
		if(ur instanceof IFile) {
			ReferenceRegistry.getInstance().addReference((IFile)getUnderlyingResource(), refType, (IFile)ur);
		}
	}

	public void addBidirectionalReference(String refType, IReferencerProxyNode proxyRef, String oppositeRefType) {
		IResource ur = proxyRef.getUnderlyingResource();
		if(ur instanceof IFile) {
			ReferenceRegistry.getInstance().addBidirectionalReference((IFile)getUnderlyingResource(), refType, (IFile)ur, oppositeRefType);
		}
	}
	
	public void removeReference(IProxyNode proxyRef) {
		IResource ur = proxyRef.getUnderlyingResource();
		if(ur instanceof IFile) {
			ReferenceRegistry.getInstance().removeReference((IFile)getUnderlyingResource(), (IFile)ur);
		}
	}

	public Collection getReferences(String refType) {
		return ReferenceRegistry.getInstance().getReferences((IFile)getUnderlyingResource(), refType);
	}
	
	public Set getReferenceTypes() {
		return ReferenceRegistry.getInstance().getReferenceTypes((IFile)getUnderlyingResource());
	}

	/**
	 * @see org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode#saveState(org.eclipse.ui.IMemento)
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public boolean saveState(IMemento memento) {
		memento.putString(TestUIConstants.TAG_NODE_KIND, getNodeKind());
		memento.putString(TAG_URI_FRAGMENT, getOriginatorURI().fragment());
		return true;
	}

	public Change createMoveChange(IRefactoringContext context, IPath destinationPath) {
		return this.getAdapter(IFile.class) != null ? new MoveModelChange(this, context, destinationPath) : null;
	}

	public Change createUpdateChange(IRefactoringContext context, IReferencerProxyNode referenced, String refType, IPath destinationPath) {
		return new UpdateModelChange(this, context);
	}

	public Change createUpdateChange(IRefactoringContext context, IContainer container, IPath destinationPath) {
		return new ModelPathChange(this, context, destinationPath);
	}

	public Change createDeleteChange(IRefactoringContext context) {
		return new DeleteModelElementChange(context, this);
	}
	
	public Change createPasteChange(IRefactoringContext context, IPath destinationPath,String [] targetName) {
		return new PasteModelChange(context,this,destinationPath,targetName);
	}
	
	//- overwrite AbstractProxy.equals to avoid to access to the underlying resource
	public boolean equals(Object arg0) {
		
		if(arg0 instanceof EObjectProxyNode) {
			
			URI argURI = ((EObjectProxyNode)(arg0)).originatorURI;
			
			return (((originatorURI == null) && (argURI == null)) || ((originatorURI != null) && (originatorURI.equals(argURI))));
		}
		
		return false;
	}

	//- overwrite AbstractProxy.hashCode to avoid to access to the underlying resource
	public int hashCode() {
		return originatorURI != null ? originatorURI.hashCode() : -2;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.navigator.IPersistableProxyNode#getFactoryID()
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	public String getFactoryID() {
		return EMFResourceProxyFactory.ID;
	}
	
	/**
	 * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
	protected abstract String getNodeKind();

	/**
	 * @provisional
	 */
	public IResource getCorrespondingResource() {
		return getUnderlyingResource();
	}
}
