/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FileProxyNode.java,v 1.10 2008/03/20 19:49:53 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.navigator;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.IExtendedProxyNode;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer;
import org.eclipse.hyades.test.ui.navigator.actions.RenamerUIInlineEditor;
import org.eclipse.hyades.test.ui.navigator.actions.RenamerUIStatus;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

/**
 * File proxy node.
 * This class is dedicated to wrap physical file into proxy 
 * 
 * @author jgout
 * @version February 26, 2008
 * @since 3.2
 */
public abstract class FileProxyNode extends AbstractProxy implements IProxyNodeRenamer, IExtendedProxyNode {
	
	private IFile file;

    /**
     * Constructor to create a proxy from a physical file.
     * @param file a physical file
     */
	public FileProxyNode(IFile file) {
		//- file without children default file node
		this.file = file;
	}
		
    /**
     * Retruns the name of the file as the name of the proxy node.
     * @return the text label of the proxy node.
     */
	public String getText() {
		return file.getName();
	}

    /**
     * Returns the icon of the proxy, by default, the standard eclipse file icon is used.
     * @return the icon of the proxy.
     */
	public Image getImage() {
		return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_FILE);
	}

	/**
	 * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getParent()
	 */
	public Object getParent() {
		return file.getParent();
	}
	
    /**
     * Return the associated file
     * @return the associated file
     */
	public IFile getFile() {
		return file;
	}
	
	/** 
     * A file proxy node is by definition a root level node, its identifier is empty.
	 * @return an empty string.
	 */
	public String getIdentifier() {
		return ""; //$NON-NLS-1$
	}
	
	/** 
     * The underlying resource of a file proxy node is the file itself.
	 * @return the file itself.
	 */
	public IResource getUnderlyingResource() {
		return file;
	}
	
    /**
     * Delegate to the platform adapter manager
     */
	public Object getAdapter(Class adapter) {
		return Platform.getAdapterManager().getAdapter(this, adapter);
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer#isApplicableFor()
	 */
	public boolean isApplicableFor() {
		return true;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer#performRename(java.lang.String)
	 */
	public boolean performRename(String newName) {
		//- do nothing if still the same name
		if (!newName.equals(file.getName())) {
			//- need to check if the new name is a valid one
			IWorkspace workspace = ResourcesPlugin.getWorkspace();
			IStatus status = workspace.validateName(newName, file.getType());
			if (!status.isOK()) {
				MessageDialog.openError(Display.getCurrent().getActiveShell(), UiPluginResourceBundle.W_ERROR, status.getMessage()); 
			}
			else {
				final IPath newPath = file.getFullPath().removeLastSegments(1).append(newName);
				WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
					public void execute(IProgressMonitor monitor) {
						try {
							file.move(newPath, IResource.KEEP_HISTORY | IResource.SHALLOW, new SubProgressMonitor(monitor, 50));
						} catch (CoreException e) {
							e.printStackTrace();
						}
					}
				};
				try {
					new ProgressMonitorDialog(Display.getCurrent().getActiveShell()).run(true, false, op);
				} catch (InvocationTargetException e) {
					e.printStackTrace();
				} catch (InterruptedException e) {
					//- can't be canceled
				}
			}
		}
		return false;
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer#performUserInteraction(java.lang.String)
	 */
	public RenamerUIStatus performUserInteraction(String oldName) {
		return new RenamerUIInlineEditor();
	}
	
	/**
	 * @provisional
	 */
	public IResource getCorrespondingResource() {
		return file;
	}

}
