/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesProxyNodeFactory.java,v 1.4 2008/10/21 18:19:57 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.navigator;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.configuration.CFGArtifact;
import org.eclipse.hyades.models.common.configuration.CFGLocation;
import org.eclipse.hyades.models.common.datapool.DPLDatapool;
import org.eclipse.hyades.models.common.testprofile.TPFDeployment;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestComponent;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.ArtifactProxyNode;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.DatapoolProxyNode;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.DeploymentProxyNode;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.LocationProxyNode;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.TypedElementFactoryManager;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.internal.extension.AssociationMappingRegistry;
import org.eclipse.ui.IMemento;

/** This singleton factory can be used to create default proxy from hyades elements such as datapool, test case or location objects.
 *  The accepted inputs are: <br>
 * <ul>
 * <li> TPFTestSuite </li>
 * <li> TPFTestCase </li>
 * <li> TPFExecutionResult </li>
 * <li> TPFTestComponent </li>
 * <li> DPLDatapool </li>
 * <li> TPFDeployment </li>
 * <li> CFGArtifact </li>
 * <li> CFGLocation </li>
 * </ul>
 * @author jgout
 * @author jbozier
 * @since 3.2
 * @version October 21, 2008
 */
public class HyadesProxyNodeFactory {

	private static HyadesProxyNodeFactory instance;
	
	/** 
     * Retrun the singleton instance.
     * This instance can be used by client to create instances of proxy derived 
     * from basic hyades test model elements such as test suite.
	 * @return the Hyades proxy node factory instance.
	 */
	public static HyadesProxyNodeFactory getInstance() {
		if(instance == null) {
			instance = new HyadesProxyNodeFactory();
		}
		return instance;
	}
	
    /**
     * Private constructor.
     */
	private HyadesProxyNodeFactory() {
	}
	
    /**
     * Check whether the given test suite has a type declared using 
     * the org.eclipse.hyades.ui.typeDecriptions extension point.
     * @param ts a given test suite.
     * @return <code>true</code> if the type of the given test suite has been properly declared and <code>false</code> otherwise.
     */
	private boolean isAKnownType(TPFTestSuite ts) {
		AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
		IAssociationMapping associationMapping = registry.getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS);
		if (associationMapping == null) {
			return false;
		}
		IAssociationDescriptor descriptor = associationMapping.getDefaultAssociationDescriptor(ts.getType());
		return descriptor != null;
	}
	
    /**
     * Check whether the given test case has a type declared using 
     * the org.eclipse.hyades.ui.typeDecriptions extension point.
     * @param tc a given test case.
     * @return <code>true</code> if the type of the given test case has been properly declared and <code>false</code> otherwise.
     */
	private boolean isAKnownType(TPFTestCase tc) {
		AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestCaseMappingRegistry();
		IAssociationMapping associationMapping = registry.getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS);
		if (associationMapping == null) {
			return false;
		}
		IAssociationDescriptor descriptor = associationMapping.getDefaultAssociationDescriptor(tc.getType());
		return descriptor != null;
	}

	/** 
     * Convert a given EMF object into a proxy node usuable in the test navigator.
	 * EMF object supported by this method are: <br>
	 * <ul>
	 * <li> TPFTestSuite </li>
	 * <li> TPFTestCase </li>
	 * <li> TPFExecutionResult </li>
	 * <li> TPFTestComponent </li>
	 * <li> DPLDatapool </li>
	 * <li> TPFDeployment </li>
	 * <li> CFGArtifact </li>
	 * <li> CFGLocation </li>
	 * </ul>
	 * 
	 * @param eObject object to be converted
	 * @param parent the parent of the converted proxy node
	 * @return a proxy node associated to the given EMF object or <code>null</code> if it is not supported by default
	 */
	public IProxyNode create(EObject eObject, Object parent) {
		if (eObject instanceof TPFTestSuite) {
			if(isAKnownType((TPFTestSuite)eObject)) {
				String type = ((TPFTestSuite)eObject).getType();
				ITypedElementProxyFactory factory = TypedElementFactoryManager.getInstance().getFactory(type);
				if(factory != null) {
					return factory.create((TPFTestSuite)eObject, parent);
				} else return null;
			} else {
				return null;
			}
		} else if (eObject instanceof TPFTestCase) {
			if(isAKnownType((TPFTestCase)eObject)) {
				String type = ((TPFTestCase)eObject).getType();
				ITypedElementProxyFactory factory = TypedElementFactoryManager.getInstance().getFactory(type);
				if(factory != null) {
					return factory.create((TPFTestCase)eObject, parent);
				} else return null;
			} else {
				return null;
			}
		} else if (eObject instanceof TPFExecutionResult) {
			String type = ((TPFExecutionResult)eObject).getType();
			ITypedElementProxyFactory factory = TypedElementFactoryManager.getInstance().getFactory(type);
			if(factory != null) {
				return factory.create((TPFExecutionResult)eObject, parent);
			} else return null;
		} else if (eObject instanceof TPFTestComponent) {
			String type = ((TPFTestComponent)eObject).getType();
			ITypedElementProxyFactory factory = TypedElementFactoryManager.getInstance().getFactory(type);
			if(factory != null) {
				return factory.create((TPFTestComponent)eObject, parent);
			} else return null;
		} else if (eObject instanceof DPLDatapool) {
			return new DatapoolProxyNode((DPLDatapool) eObject, parent);
		} else if (eObject instanceof TPFDeployment) {
			return new DeploymentProxyNode((TPFDeployment) eObject, parent);
		} else if (eObject instanceof CFGArtifact) {
			return new ArtifactProxyNode((CFGArtifact) eObject, parent);
		} else if (eObject instanceof CFGLocation) {
			return new LocationProxyNode((CFGLocation) eObject, parent);
		}
		return null;
	}
	
    /**
     * Create a proxy node starting to the persisted proxy node content.
     * 
     * @param memento the memento containing the pesisted proxy node state. 
     * @param parent the parent node of the new proxy.
     * @return a new instance of the persisted proxy node or <code>null</code> if it is an unknown proxy node state.
     */
	public IProxyNode recreate(IMemento memento, Object parent) {
		String nodeKind = memento.getString(TestUIConstants.TAG_NODE_KIND);
		if(nodeKind.equals(TestUIConstants.ARTIFACT_NODE)) {
			return new ArtifactProxyNode(memento, parent);
		} else if(nodeKind.equals(TestUIConstants.DATAPOOL_NODE)) {
			return new DatapoolProxyNode(memento, parent);
		} else if(nodeKind.equals(TestUIConstants.DEPLOY_NODE)) {
			return new DeploymentProxyNode(memento, parent);
		} else if(nodeKind.equals(TestUIConstants.LOCATION_NODE)) {
			return new LocationProxyNode(memento, parent);
		}  else if(nodeKind.equals(TestUIConstants.TESTSUITE_NODE) ||
					nodeKind.equals(TestUIConstants.EXECUTION_RESULT_NODE) || 
					nodeKind.equals(TestUIConstants.TESTCASE_NODE) || 
					nodeKind.equals(TestUIConstants.TESTCOMPONENT_NODE)) {
			ITypedElementProxyFactory factory = TypedElementFactoryManager.getInstance().getFactory(memento.getString(TestUIConstants.TAG_TYPE));
			if(factory instanceof IPersistableTypedElementProxyFactory) {
				return ((IPersistableTypedElementProxyFactory)factory).recreate(memento, parent); 
			}
		} 
		return null;
	}
	
}
