/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: IFileProxyManager.java,v 1.5 2008/03/20 19:49:53 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.ui.navigator;

import org.eclipse.core.resources.IFile;

/** Manager for proxies built from file.
 * This interface is not intended to be implemented by client.
 * 
 * @author jgout
 * @version February 26, 2008
 * @since 4.0
 */
public interface IFileProxyManager {
    
    /** 
     * Returns the proxy node corresponding to the given file.
     * The returned proxy is not automatically cached. If the caller intends to call
     * this method with the same arguments, it should put the returned proxy in cache
     * using cacheProxy().
     * @param file the file to convert
     * @param parent the parent of the proxy node in case of creation. This parameter may be
     * <code>null</code>. In this case, this method will only return a proxy node already
     * cached. If this parameter is not null, this method may instantiate the proxy it is not
     * already cached and ensures that the returned proxy has <code>parent</code> as a parent.
     * @return the IProxyNode matching with the given file or null if no factory can handle this file
     */
    public IProxyNode getProxy(IFile file, Object parent);

    /**
     * Put the specified proxy in the cache so further calls to getProxy(IFile, Object) return
     * in an efficient way the same cached proxy.
     * @param file
     * @param newProxy
     */
    public void cacheProxy(IFile file, IProxyNode proxy);

    /** 
     * Returns whether the given file is already in the local proxy data base. 
     * @param file
     * @return true if the given file exists as proxy and false otherwise
     */
    public boolean isaCachedFile(IFile file);

    /** 
     * Removes the proxy entry associated to the given file.
     * @param file
     */
    public IProxyNode uncacheProxy(IFile file);
    
    /** 
     * Retrieves the node associated to the given object. 
     * This parameter is adapted to a IProxy in order to retrieve the node
     * representing it.
     * 
     * @param object An object adaptable to IProxy.
     * @return the proxy associated to the given object or <code>null</code> if such proxy node does not exist.
     */
    public IProxyNode getCorrespondingProxy(Object object);

    /** 
     * Retrieves the node using its underlying resource and its identifier. 
     * 
     * @param fileName name of the underlying resource of the searched proxy
     * @param identifier identifier of the searched proxy
     * @return a proxy node or <code>null</code> if not found.
     * @deprecated use {@link #getCorrespondingProxy(new Proxy(fileName, String))} instead.
     */
    public IProxyNode getCorrespondingProxy(String fileName, String identifier);

    /** Returns the proxy which has the given uid. This search starts from the given proxy node and cross its complete sub tree of proxies.
     * 
     * @param proxy root of the proxy tree where the search is done. 
     * @param uid the uid of the searched proxy.
     * @return a proxy node which has the given uid or <code>null</code> if there is no such proxy found in the complete given tree of proxy.
     */
    public IProxyNode findProxyByID(IProxyNode proxy, String uid);

    /** 
     * Computes a new proxy using registered file proxy factories.
     * This method updates the cached proxy for the given file 
     * @param file the file from which the proxy is derived.
     * @param parent the parent node to pass to the factory.
     * @return the proxy associated to the given file.
     */
    public IProxyNode updateProxy(IFile file, Object parent);

    /**
     * Returns the parent of the given proxy node.
     * @param proxy a proxy node
     * @return the parent of the given proxy node or <code>null</code> if this proxy node has not been cached.
     * @since 4.2
     */
    public Object getParent(IProxyNode proxy);
}
