/**********************************************************************
 * Copyright (c) 2009, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProxyRequestManager.java,v 1.5 2010/05/27 15:07:40 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.navigator;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.hyades.test.ui.internal.navigator.proxy.async.ProxiesRequest;

/**
 * <p>Manager for proxy refresh requests that invokes {@link IProxyRefreshListener#start()} 
 * and {@link IProxyRefreshListener#end()} for of all registered {@link IProxyRefreshListener}s 
 * when a refresh job is started and ended on a proxy inside the Test Navigator.</p>
 * 
 *  
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version May 27, 2010
 * @since   April 16, 2009
 * @provisional As of TPTP V4.6.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class ProxyRequestManager {

	private List<IProxyRefreshListener> allListeners = new ArrayList<IProxyRefreshListener>();
	private List<ProxiesRequest> allRequest = new ArrayList<ProxiesRequest>();
	private boolean started = false;

	private static ProxyRequestManager instance = null;

	private static final Object LOCK = new Object();
	
	private ProxyRequestManager() {
		//No-operation.
	}
	
	/**
	 * return true if request is started
	 * return false if request is not started
	 */
	public boolean isStarted() {
		boolean ret;
		synchronized(LOCK) {
			ret = started;
		}
		return ret;
	}
	
	/**
	 * called by request to register them when they starts
	 * @param request
	 */
	public void startRequest(ProxiesRequest request) {
		
		if (request != null) {
			boolean empty = false;
			IProxyRefreshListener[] copyAllListeners = null;
			synchronized(LOCK) {
				empty = allRequest.isEmpty();
				allRequest.add(request);			
				started = true;
				if (empty) { // we make a copy list inside synchronize block to avoid concurrentAccess problems when iterating on it
					copyAllListeners = ((IProxyRefreshListener[])(allListeners.toArray(new IProxyRefreshListener[allListeners.size()]))); 					
				}
			}
			if (empty) {
				if (copyAllListeners != null) { // test should not be needed but safer
					for (int i=0; i<copyAllListeners.length; i++) {
						copyAllListeners[i].start(); // start should be called outside synchronized block to avoid deadlock
					}
				}
			}
		}
	}
	
	/**
	 * called by request to unregister them when they ends
	 * @param request
	 */
	public void endRequest(ProxiesRequest request) {
		
		if (request != null) {
			boolean empty = false;
			IProxyRefreshListener[] copyAllListeners = null;
			synchronized(LOCK) {
				if (allRequest.remove(request)) {
					empty = allRequest.isEmpty();
					started = !empty;
					if (empty) { // we make a copy list inside synchronize block to avoid concurrentAccess problems when iterating on it
						copyAllListeners = ((IProxyRefreshListener[])(allListeners.toArray(new IProxyRefreshListener[allListeners.size()]))); 					
					}
				}
			}
			if (empty) {
				if (copyAllListeners != null) { // test should not be needed but safer
					for (int i=0; i<copyAllListeners.length; i++) {
						copyAllListeners[i].end(); // end should be called outside synchronized block to avoid deadlock
					}
				}		
			}
		}
	}
	
	/**
	 * return instance of the manager
	 */
	public static ProxyRequestManager instance() {
		
		if (instance == null) {			
			instance = new ProxyRequestManager();
		}
		
		return instance;
	}
	
	/**
	 * register a listener for refresh start/end notification
	 * @param listener
	 */
	public void register(IProxyRefreshListener listener) {
		
		if (listener != null) {

			synchronized(LOCK) {
				allListeners.add(listener);
			}
		}
	}
	
	/**
	 * unregister a registered listener 
	 * @param listener
	 * @return true if found, false else
	 */
	public boolean unRegister(IProxyRefreshListener listener) {
		
		if (listener != null) {
			
			synchronized(LOCK) {
				return (allListeners.remove(listener));
			}
		}
		
		return false;
	}	
}
