/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestFolderProxyNode.java,v 1.17 2007/06/04 13:58:47 jptoomey Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.navigator;

import java.lang.reflect.InvocationTargetException;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.IExtendedProxyNode;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer;
import org.eclipse.hyades.test.ui.navigator.actions.RenamerUIInlineEditor;
import org.eclipse.hyades.test.ui.navigator.actions.RenamerUIStatus;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.ProgressMonitorDialog;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

/**
 * This class is test folder proxy node. A test folder is associated to a actual
 * folder but it filters all test elements which are not ok according into the
 * ITestFolderContentValidator interface. <br>
 * This class is used inside the type provider proxies trees.
 * 
 * @author jgout
 * @since 4.0
 */
public class TestFolderProxyNode implements IProxyNodeRenamer, IResourceChangeListenerProxyNode, IExtendedProxyNode {

    private IContainer container;
    private Object parent;
    private IProxyNode[] children;
    private ITestFolderContentValidator validator;
    private boolean flat;
    private IFileProxyManager fileProxyManager;
    
    /**
     * Public API of creation. It returns an instance of test folder proxy node or <code>null</code> if the new test folder is empty.
     * 
     * @param container the actual folder (or project) the proxy is associated to.
     * @param validator a validator used to test whether the members of the given 
     * container should be convert into test folder children.
     * @param fileProxyManager the file proxy manager used to cache file proxy node.
     * @param flat a flag that control the content hierarchy. If flag is <code>true</code> this means that the physical hierarchy is flattened. 
     * Otherwise, the original folder hierarchy is respected and converted into the proxy node hierarchy. 
     * @param parent the parent node of the test folder proxy node.
     * @return an instance of test folder proxy node or <code>null</code> if the new test folder is empty.
     */
    public static TestFolderProxyNode create(IContainer container, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, boolean flat, Object parent) {
        TestFolderProxyNode proxy = new TestFolderProxyNode(container, validator, fileProxyManager, flat, parent);
        if(proxy.getChildren().length > 0) {
            return proxy;
        } else return null;
    }

    /**
     * Constructor. Equivalent to {@link #TestFolderProxyNode(IContainer, ITestFolderContentValidator, IFileProxyManager, boolean, Object, boolean)}
     * with computeChildren=true.
     * @param container the actual folder (or project) the proxy is associated to.
     * @param validator a validator used to test whether the members of the given container should be convert into test folder children.
     * @param fileProxyManager the file proxy manager used to cache file proxy node.
     * @param flat flat a flag that control the content hierarchy. If flag is <code>true</code> this means that the physical hierarchy is flattened. 
     * Otherwise, the original folder hierarchy is respected and converted into the proxy node hierarchy. 
     * @param parent the parent node of the test folder proxy node.
     */
    protected TestFolderProxyNode(IContainer container, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, boolean flat, Object parent) {
        this(container, validator, fileProxyManager, flat, parent, true);
    }
    
    /**
     * Constructor.
     * @param container the actual folder (or project) the proxy is associated to.
     * @param validator a validator used to test whether the members of the given container should be convert into test folder children.
     * @param fileProxyManager the file proxy manager used to cache file proxy node.
     * @param flat flat a flag that control the content hierarchy. If flag is <code>true</code> this means that the physical hierarchy is flattened. 
     * Otherwise, the original folder hierarchy is respected and converted into the proxy node hierarchy. 
     * @param parent the parent node of the test folder proxy node.
     * @param computeChildren Whether the children computation should occur immediately. Otherwise,
     * {@link #computeChildren()} must explicitely invoked by the sub-class's constructor.
     * 
     * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
     */
    protected TestFolderProxyNode(IContainer container, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, boolean flat, Object parent, boolean computeChildren) {
    	this.container = container;
        this.validator = validator;
        this.parent = parent;
        this.flat = flat;
        this.fileProxyManager = fileProxyManager;
        if (computeChildren) {
	        computeChildren();
        }
    }
    
    /**
     * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
     */
    protected final void computeChildren() {
    	List _children = new LinkedList();
        buildChildren(_children, this.container);
        children = (IProxyNode[]) _children.toArray(new IProxyNode[_children.size()]);
    }

    /** 
     * Builds the list of proxy node children of a given physical folder.
     * @param _children the list of proxy node children
     * @param cont an actual folder (or project).
     */
    protected void buildChildren(List _children, IContainer cont) {
        IResource[] resources = null;
        if (cont != null && cont.isAccessible()) {
            try {
                resources = cont.members();
            } catch (CoreException e) {
                UiPlugin.logError(e);
                _children = new LinkedList();
                return;
            }
            for (int i = 0; i < resources.length; i++) {
                if (!TestNavigator.getFiltersManager().filter(resources[i])) {
                    if (resources[i].getType() != IResource.FILE) {
                        if (flat) {
                            buildChildren(_children, (IContainer) resources[i]);
                        } else {
                            IProxyNode proxyFolder = createChildFolderProxyNode((IContainer) resources[i]);
                            if (proxyFolder != null) {
                                _children.add(proxyFolder);
                            }
                        }
                    } else {
                        //- convert the file in something else (proxy)
                        if (validator.isFileOk((IFile) resources[i])) {
                            IProxyNode node = createChildFileProxyNode((IFile) resources[i]);
                            if (validator.isProxyOk(node) && !TestNavigator.getFiltersManager().filter(node)) {
                                _children.add(node);
                                fileProxyManager.cacheProxy((IFile)resources[i], node);
                            }
                        }
                    }
                }
            }
        }
    }
    
    /**
     * Returns a TestFolderProxyNode that should represent the given container.
     * This implementation instantiates a TestFolderProxyNode representing the given container.
     * Sub-classes may override this method to instantiate sub-types of TestFolderProxyNode.
     * @param container A container that is a child of the resource represented by this
     * proxy node.
     * @return A TestFolderProxyNode (or a sub-type), or null if the container should not
     * be represented.
     */
    protected IResourceChangeListenerProxyNode createChildFolderProxyNode(IContainer container) {
    	return create(container, validator, fileProxyManager, flat, this);
    }
        
    /**
     * Returns an IProxyNode that represents the given file.
     * This implementation is using the Test Navigator extensions point to instantiate
     * the proper proxy node representing the file.
     * @param file A file that is a child of the resource represented by this proxy node. 
     * @return An IProxyNode. This method is not expected to return null (since this method
     * is invoked only if the validator returns <code>true</code> to <code>isFileOk()</code>).
     * 
     * @provisional As of TPTP V4.4.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
     */
    protected IProxyNode createChildFileProxyNode(IFile file) {
    	return getFileProxyManager().getProxy(file, this);
    }
 
    /**
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getText()
     */
    public String getText() {
        return container.getName();
    }

    /**
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getImage()
     */
    public Image getImage() {
        return PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_OBJ_FOLDER);
    }

    /**
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getParent()
     */
    public Object getParent() {
        return parent;
    }

    /**
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getChildren()
     */
    public IProxyNode[] getChildren() {
        return children;
    }

    /**
     * @see org.eclipse.hyades.test.ui.navigator.IProxy#getUnderlyingResource()
     */
    public IResource getUnderlyingResource() {
        return container;
    }

    /**
     * @see org.eclipse.hyades.test.ui.navigator.IProxy#getIdentifier()
     */
    public String getIdentifier() {
        return getText();
    }

    /**
     * @see org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer#isApplicableFor()
     */
    public boolean isApplicableFor() {
        return true;
    }

    /**
     * @see org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer#performUserInteraction(java.lang.String)
     */
    public RenamerUIStatus performUserInteraction(String oldName) {
        return new RenamerUIInlineEditor();
    }

    /**
     * @see org.eclipse.hyades.test.ui.navigator.actions.IProxyNodeRenamer#performRename(java.lang.String)
     */
    public boolean performRename(String newName) {
        //- do nothing if still the same name
        if (!newName.equals(container.getName())) {
            //- need to check if the new name is a valid one
            IWorkspace workspace = ResourcesPlugin.getWorkspace();
            IStatus status = workspace.validateName(newName, container.getType());
            if (!status.isOK()) {
                MessageDialog.openError(Display.getCurrent().getActiveShell(), UiPluginResourceBundle.W_ERROR, status.getMessage()); 
            } else {
                final IPath newPath = container.getFullPath().removeLastSegments(1).append(newName);
                WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
                    public void execute(IProgressMonitor monitor) {
                        try {
                            container.move(newPath, IResource.KEEP_HISTORY | IResource.SHALLOW, new SubProgressMonitor(monitor, 50));
                        } catch (CoreException e) {
                            e.printStackTrace();
                        }
                    }
                };
                try {
                    new ProgressMonitorDialog(Display.getCurrent().getActiveShell()).run(true, false, op);
                } catch (InvocationTargetException e) {
                    e.printStackTrace();
                } catch (InterruptedException e) {
                    //- can't be canceled
                }
            }
        }
        return false;
    }

    /**
     * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
     */
    public Object getAdapter(Class adapter) {
        //- this proxy is adaptable in IResource
        if (adapter == IResource.class || adapter == IContainer.class) {
            return container;
        } else {
            return Platform.getAdapterManager().getAdapter(this, adapter);
        }
    }
    
    /**
     * Returns the test folder child proxy that has been built from the given resource.
     * @param resource a physical resource.
     * @return the test folder child proxy that has been built from the given resource.
     */
    private IProxyNode getProxy(IResource resource) {
        for (int i = 0; i < children.length; i++) {
            if (resource.equals(children[i].getUnderlyingResource())) {
                return children[i];
            }
        }
        return null;
    }

    /**
     * Add a new proxy as a child of the test folder proxy node.
     * @param proxy the new child proxy.
     * @return <code>true</code> the addition is successful.
     */
    private boolean addChild(IProxyNode proxy) {
        if(proxy == null) return false;
        IProxyNode[] newChildren = new IProxyNode[children.length + 1];
        for (int i = 0; i < children.length; i++) {
            //- if the proxy to add is already a child do not add it
            if(children[i].equals(proxy)) {
                return false;
            }
            newChildren[i] = children[i];
        }
        newChildren[children.length] = proxy;
        //- change the children of the current proxy node
        children = newChildren;
        return true;
    }

    /**
     * Add a set of children in one time as children of the current test folder.
     * @param children2 an array of proxy nodes. 
     */
    private void addChildren(IProxyNode[] children2) {
        IProxyNode[] newChildren = new IProxyNode[children.length + children2.length];
        int i = 0;
        for (; i < children.length; i++) {
            newChildren[i] = children[i];
        }
        for (int j = 0; j < children2.length; j++) {
            newChildren[i++] = children2[j];
        }
        //- change the children of the current proxy node
        children = newChildren;
    }

    /**
     * Replaces the given old proxy by a new one.
     * @param oldProxy an proxy to be changed.
     * @param newProxy an new proxy node.
     */
    private void replaceChild(IProxyNode oldProxy, IProxyNode newProxy) {
        for (int i = 0; i < children.length; i++) {
            if(children[i] == oldProxy) {
                children[i] = newProxy;
                break;
            }
        }
    }
 
    /**
     * Remove a given proxy node from the list of the test folder children.
     * @param proxy a proxy node.
     */
    private void removeChild(IProxyNode proxy) {
        IProxyNode[] newChildren = new IProxyNode[children.length - 1];
        int j = 0;
        for (int i = 0; i < children.length; i++) {
            if(children[i] != proxy) {
                newChildren[j++] = children[i];
            }
        }
        //- change the children of the current proxy node
        children = newChildren;
    }

    /**
     * @see org.eclipse.hyades.test.ui.navigator.IResourceChangeListenerProxyNode#resourceChanged(org.eclipse.core.resources.IResourceDelta)
     */
    public IProxyNode resourceChanged(IResourceDelta rootDelta) {
        IProxyNode lowestChanged = null;
        IResourceDelta[] res = rootDelta.getAffectedChildren();
        for (int i = 0; i < res.length; i++) {
            int kind = res[i].getKind();
            IProxyNode lowestChild = null;
            if (kind == IResourceDelta.ADDED) {
                lowestChild = updateADDED(res[i]);
            } else if (kind == IResourceDelta.REMOVED) {
                lowestChild = updateREMOVED(res[i]);
            } else if (kind == IResourceDelta.CHANGED) {
                lowestChild = updateCHANGED(res[i]);
            }
            if(lowestChild != null) {
                //- the current child has reported a change in its content
                if(lowestChanged == null) {
                    //- not a child modified yet, so the lowest is this child
                    lowestChanged = lowestChild;
                } else {
                    //- this child is not the first child to change, so the lowest is the parent
                    lowestChanged = this;
                }
            }
        }
        return lowestChanged;
    }

    /**
     * Resource changed method focused upon new files 
     * @param delta the resource changed delta.
     * @return the inner most proxy that has changed.
     */
    private IProxyNode updateADDED(IResourceDelta delta) {
        IResource resource = delta.getResource();
        int resType = resource.getType();
        //- a resource (folder or file) has been added to the file system
        if (resType == IResource.FILE) {
            //- new file check if this file should kept
            if (validator.isFileOk((IFile) resource)) {
                IProxyNode newProxy = fileProxyManager.getProxy((IFile)resource, this);
                //- check if new proxy should be added as a child of current
                if (validator.isProxyOk(newProxy)) {
                    boolean wasAdded = addChild(newProxy);
                    if(wasAdded) {
                        fileProxyManager.cacheProxy((IFile)resource, newProxy);
                        return this;
                    }
                }
            }
        } else if (resType == IResource.FOLDER) {
            return updateADDEDFolder(resource);
        }
        return null;
    }

    /**
     * Resource changed method focused upon new folder
     * @param resource the new added folder.
     * @return the inner most proxy that has changed.
     */
    private IProxyNode updateADDEDFolder(IResource resource) {
        IProxyNode newFolderProxy = createChildFolderProxyNode((IFolder)resource);
        if (newFolderProxy != null) {
            if (flat) {
                //- add all children of the new node into the current one.
                addChildren(newFolderProxy.getChildren());
            } else {
                addChild(newFolderProxy);
            }
            return this;
        }
        return null;
    }

    /**
     * Resource changed method focused upon changes.
     * @param delta the resource changed delta.
     * @return the inner most proxy that has changed.
     */
    private IProxyNode updateCHANGED(IResourceDelta delta) {
        IResource resource = delta.getResource();
        int resType = resource.getType();
        int flags = delta.getFlags();
         //- a resource (folder or file) has changed in the file system
        if (resType == IResource.FILE) {
            return updateCHANGEDFile(resource, flags);
        } else if (resType == IResource.FOLDER) {
            //- a folder has changed
            return updateCHANGEDFolder(delta, resource);
        }
        return null;
    }

    /**
     * Resource changed method focused upon changes of files
     * @param resource the file that as changed.
     * @param flags changed flag of the delta.
     * @return the inner most proxy that has changed.
     */
    private IProxyNode updateCHANGEDFile(IResource resource, int flags) {
        //- find the matching proxy in the children array of the current proxy.
        IProxyNode proxy = getProxy(resource);
        if ((flags & IResourceDelta.CONTENT) != 0) {
            if(proxy != null) {
                //- Since proxy exists, we need to update it
                if(validator.isFileOk((IFile)resource)) {
                    fileProxyManager.uncacheProxy((IFile)resource);
                    IProxyNode newProxy = fileProxyManager.getProxy((IFile)resource, this);
                    if(validator.isProxyOk(newProxy)) {
                        replaceChild(proxy, newProxy);
                        fileProxyManager.cacheProxy((IFile)resource, newProxy);
                    } else {
                        removeChild(proxy);
                    }
                    return this;
                }
            } else {
                //- even if the proxy is not yet built, we need to check if it can be converted.
                if(validator.isFileOk((IFile)resource)) {
                    IProxyNode newProxy = fileProxyManager.getProxy((IFile)resource, this);
                    if(validator.isProxyOk(newProxy)) {
                        boolean wasAdded = addChild(newProxy);
                        if(wasAdded) {
                            fileProxyManager.cacheProxy((IFile)resource, newProxy);
                            return this;
                        }
                    }
                }
                return null;
            }
        }
        return null;
    }

    /**
     * Resource changed method focused upon changes of files
     * @param delta the resource changed delta.
     * @param resource the folder that as changed.
     * @return the inner most proxy that has changed.
     */
    private IProxyNode updateCHANGEDFolder(IResourceDelta delta, IResource resource) {
        //- find the matching proxy in the children array of the current proxy.
        IProxyNode proxy = getProxy(resource);
        if (proxy != null) {
            //- since no change at this level, go ahead !
            if (proxy instanceof IResourceChangeListenerProxyNode) {
                IProxyNode lowestChanged = ((IResourceChangeListenerProxyNode) proxy).resourceChanged(delta);
                if (proxy.getChildren().length == 0) {
                    //- proxy once updated should be removed
                    removeChild(proxy);
                    lowestChanged = this;
                }
                return lowestChanged;
            } else {
                //- this should never occur, because the proxy of a folder
                // should be a TestFolderProxyNode
                UiPlugin.logError("The proxy node derived from a folder should be a TestFolderProxyNode");//$NON-NLS-1$
            }
        } else if(!flat) {
            //- the test folder proxy node is not yet created.
            //- we need to build it since there would have a change
            // beneath.
            //- if a new file (that the validator accepts) has been created
            // we will catch it,
            //- since the proxy is not here, the situation can't be a
            // remove of a file (that the validator accepts)
            return updateADDEDFolder(resource);
        } else {
            //- flat, so go head in delta
            return resourceChanged(delta);
        }
        return null;
    }

    /**
     * Resource changed method focused upon removed elements
     * @param delta the resource changed delta.
     * @return the inner most proxy that has changed.
     */
    private IProxyNode updateREMOVED(IResourceDelta delta) {
        IResource resource = delta.getResource();
        int resType = resource.getType();
        //- a resource (folder or file) has been removed to the file system
        if (resType == IResource.FILE) {
            IProxyNode proxy2Remove = getProxy(resource);
            if(proxy2Remove != null) {
                //- the proxy to remove is one of the current node children, we need to remove it form the cache
                fileProxyManager.uncacheProxy((IFile)resource);
                //- and remove it from the children list of the current folder
                removeChild(proxy2Remove);
                return this;
            } else {
                //- proxy not found among children so this file removal has no impacts in this current folder
                return null;
            }
        } else if (resType == IResource.FOLDER) {
            IProxyNode proxy2Remove = getProxy(resource);
            if(proxy2Remove != null) {
                //- recursive call to remove children
                if (proxy2Remove instanceof IResourceChangeListenerProxyNode) {
                    ((IResourceChangeListenerProxyNode)proxy2Remove).resourceChanged(delta);
                    removeChild(proxy2Remove);
                    return this;
                } else {
                    //- this should never arrive, because the proxy of a folder
                    //- should be a IResourceChangeListenerProxyNode
                    UiPlugin.logError("The proxy node derived from a folder should be a IResourceChangeListenerProxyNode"); //$NON-NLS-1$
                }
            } else if(!flat) {
                //- not a proxy at this level, this file removal has no impacts in the current folder
                return null;
            } else {
                //- if flat we need to recurse on resource delta
                return resourceChanged(delta);
            }
        }
        return null;
    }

	/**
     * Returns the flat attribute.
	 * @return the flat attribute.
	 */
	protected final boolean isFlat() {
		return flat;
	}
	
	/**
     * Returns the validator used by this test folder.
	 * @return the validator used by this test folder.
	 */
	protected final ITestFolderContentValidator getValidator() {
		return validator;
	}
    /**
     * Returns the file proxy manager instance used by this folder proxy node.
     * @return the file proxy manager instance used by this folder proxy node.
     */
    protected final IFileProxyManager getFileProxyManager() {
        return fileProxyManager;
    }

	/**
	 * @provisional
	 */
	public IResource getCorrespondingResource() {
		//- if the test folder is flat (hierarchy flatten in its content),
		//- there are no corresponding resrouces
		return isFlat() ? null : container;
	}
}
