/**********************************************************************
 * Copyright (c) 2008, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: 
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.navigator.refactoring;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.IPath;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.DeleteContainerChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.DeleteContainerCompositeChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.DeleteFileChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.MoveFileChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.MoveFolderChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.MoveFolderCompositeChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.PasteFileChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.PasteFolderChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.PasteFolderCompositeChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.RenameContainerChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.RenameFileChange;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.CompositeChange;

/** 
 * <p>Various utilities for to allow user to use basic changes (paste, delete, move, rename) 
 * on resource elements (file/container).</p>
 * 
 * 
 * @author  Jerome Bozier
 * @version November 3, 2009
 * @since   November 19, 2008
 * @provisional As of TPTP 4.5.2, this is stable provisional API.
 */
public class ActionRefactoringFactory {

	private static ActionRefactoringFactory instance = null;
	
	private ActionRefactoringFactory() {
	}
	
	/**
	 * return an instance of the factory
	 * @return factory instance
	 * @provisional As of TPTP 4.5.2, this is stable provisional API	 
	 */
	public static ActionRefactoringFactory getInstance() {
		if (instance == null) {
			instance = new ActionRefactoringFactory();
		}
		return instance;
	}
	
	/**
	 * return a composite change relative to container deletion 
	 * The resulted change has a root change (delete of the folder itself) and children
	 * whose are delete of each container member.
	 * @param container : container to delete
	 * @param deleteContent : if container is a project, force to delete its contents
	 * @return change that will perform the delete
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public CompositeChange createDeleteContainerCompositeChange(IContainer container,boolean deleteContent) {
		return new DeleteContainerCompositeChange(container,deleteContent,RefactoringMessages.DELETE_FOLDER,null);
	}
	
	/**
	 * return a change relative to file deletion
	 * @param file : file to delete
	 * @return change that will perform the delete
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public Change createDeleteFileChange(IFile file) {
		return new DeleteFileChange(file);
	}
	
	/**
	 * return a change relative to container deletion
	 * @param cont : container to delete
	 * @param deleteContent : if container is a project, force to delete its contents
	 * @return change that will perform the delete
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public Change createDeleteContainerChange(IContainer cont,boolean deleteContent) {
		return new DeleteContainerChange(cont,deleteContent,null);
	}
	
	/**
	 * return a change relative to classic file pasting
	 * @param file : file to paste
	 * @param destination : container where the file will be pasted
	 * @return change that will perform the paste
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public AbstractPasteChange createPasteFileChange(IFile file, IContainer destination) {
		return new PasteFileChange(file,destination);
	}
	
	/**
	 * return a change relative to file pasting when no IFile is available (extern copy then pasting into eclipse)
	 * @param externalFileName : name of file to paste
	 * @param destination : container where the file will be pasted
	 * @return change that will perform the paste
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public AbstractPasteChange createPasteFileChange(String externalFileName, IContainer destination) {
		return new PasteFileChange(externalFileName,destination);
	}
	
	/**
	 * return a composite change relative to container pasting
	 * @param folder : folder to paste
	 * @param newPath : place where folder will be pasted
	 * @param targetPath : array of one element, will contain name on which the resource have been pasted after the call 
	 * @return change that will perform the paste
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public CompositeChange createPasteFolderCompositeChange(IFolder folder, IPath newPath,IPath [] targetPath) {
		return new PasteFolderCompositeChange(folder,newPath,targetPath);	
	}
	
	/**
	 * return a change relative to container pasting
	 * @param folder : folder to paste
	 * @param destPath : place where folder will be pasted
	 * @param finalPath : array of one element, will contain name on which the resource have been pasted after the call
	 * @return change that will perform the paste
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public Change createPasteFolderChange(IFolder folder, IPath destPath,IPath [] finalPath) {
		return new PasteFolderChange(folder,destPath,finalPath);
	}
	
	/**
	 * return a change relative to file moving
	 * @param file : file to move
	 * @param destination : container where file will be moved
	 * @return change that will perform the move
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public Change createMoveFileChange(IFile file, IContainer destination) {
		return new MoveFileChange(file,destination);
	}
	
	/**
	 * return a change relative to folder moving
	 * @param folder : folder to move
	 * @param destPath : path where the folder will be moved
	 * @return change that will perform the move
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public Change createMoveFolderChange(IFolder folder, IPath destPath) {
		return new MoveFolderChange(folder,destPath);
	}
	
	/**
	 * return a composite change relative to folder moving 
	 * @param folder : folder to move
	 * @param newPath : path where the folder will be moved
	 * @return change that will perform the move
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public CompositeChange createMoveFolderCompositeChange(IFolder folder, IPath newPath) {
		return new MoveFolderCompositeChange(folder,newPath);
	}
	
	/**
	 * return a change relative to container renaming
	 * @param container : container to rename
	 * @param newName : new name
	 * @return change that will perform the rename
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public Change createRenameContainerChange(IContainer container, String newName) {
		return new RenameContainerChange(container,newName);
	}
	
	/**
	 * return a change relative to file renaming
	 * @param file : file to rename
	 * @param name : new name
	 * @return change that will perform the rename
	 * @provisional As of TPTP 4.5.2, this is stable provisional API
	 */
	public Change createRenameFileChange(IFile file, String name) {
		return new RenameFileChange(file,name);
	}
}
