/********************************************************************** 
 * Copyright (c) 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: EMFRefactoringTransaction.java,v 1.2 2010/10/26 13:18:58 bjerome Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.test.ui.navigator.refactoring;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.navigator.IRefactoringTransaction;
import org.eclipse.ltk.core.refactoring.Change;

/**
 * <p>Refactoring transaction used for EMF resources.</p>
 * 
 * <p>For more information, see the <b>Test Navigator Refactoring Transactions</b> 
 * (<code>org.eclipse.hyades.test.ui.testNavigatorRefactoringTransactions</code>) extension 
 * point.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version September 3, 2010
 * @since   February 3, 2010
 * @see 	IRefactoringTransaction
 * @provisional As of TPTP V4.7.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class EMFRefactoringTransaction implements IRefactoringTransaction {
	
	private Set<Resource> resources = null;
	private Map<Resource, URI> resourceURIMap = null;
	private ResourceSet resourceSet = null;
	private List<Change> customChange = null;

	/**
	 * The ID of the EMF refactoring transaction (<code>org.eclipse.hyades.test.ui.navigator.refactoring.EMFRefactoringTransaction</code>).
	 */
	public static final String EMF_REFACTORING_TRANSACTION_ID = "org.eclipse.hyades.test.ui.navigator.refactoring.EMFRefactoringTransaction"; //$NON-NLS-1$

	/**
	 * No-argument constructor.
	 */
	public EMFRefactoringTransaction() {
		
		this.resources = new HashSet<Resource>();
		this.resourceURIMap = new HashMap<Resource, URI>();
		this.resourceSet = new ResourceSetImpl();
		this.customChange = new ArrayList<Change>();
	}
	
	/**
	 * Resolves the resource set for the transaction.
	 * 
	 * @return The resource set for the transaction.
	 */
	public ResourceSet getResourceSet() {
		return resourceSet;
	}
	
	/**
	 * Resolves the set of resource(s) to be changed.
	 * 
	 * @return The set of resource(s) to be changed.
	 */
	public Set<Resource> getChangedResources() {
		return (resourceURIMap.keySet());
	}
	
	/**
	 * Return the list of custom changes
	 * 
	 * @return The list of custom changes
	 */
	public List<Change> getCustomChanges() {
		return customChange;
	}
	
	/**
	 * Resolves the new URI of the resource to be changed.
	 * 
	 * @param resource The resource to be changed.
	 * @return The new URI of the resource to be changed.
	 */
	public URI getChangedResourceURI(Resource resource) {
		return resourceURIMap.get(resource);
	}
	
	/**
	 * Adds the new URI for the resource to be changed.
	 * 
	 * @param resource The resource to be changed.
	 * @param newURI The new URI for the resource to be changed.
	 */
	public void addChangedResourceURI(Resource resource, URI newURI) {
		resourceURIMap.put(resource, newURI);
	}

	/**
	 * Adds a custom change to be performed during commit time
	 * 
	 * @param change Change to be performed
	 */
	public void addCustomChange(Change change) {
		customChange.add(change);
	}
	
	/**
	 * Resolves the set of resource(s) to be saved.
	 * 
	 * @return The set of resource(s) to be saved.
	 */
	public Set<Resource> getSavedResources() {
		return resources;
	}
	
	/**
	 * Adds the non-<code>null</code> resource to the set of 
	 * resource(s) to be saved.
	 * 
	 * @param resource The non-<code>null</code> resource to be saved.
	 */
	public void addSavedResource(Resource resource) {
		
		Assert.isNotNull(resource);
		
		if(!resources.contains(resource)) {
			resources.add(resource);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.navigator.IRefactoringTransaction#beginChanges(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void beginChanges(IProgressMonitor monitor) { 
		//No operation (extended by subclasses).
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.navigator.IRefactoringTransaction#cancelChanges(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void cancelChanges(IProgressMonitor monitor) { 
		//No operation (extended by subclasses).
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.navigator.IRefactoringTransaction#commitChanges(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public Change commitChanges(IProgressMonitor pm) throws CoreException {
		
		pm.beginTask("", (getChangedResources().size() + getSavedResources().size() + getCustomChanges().size() + 1)); //$NON-NLS-1$
		
		try {
			
			List<IFile> files = new LinkedList<IFile>();
			Map<Resource, IFile> undoMap = new HashMap<Resource, IFile>();
			
			//Change the URI of resource(s) to be moved:
			Iterator<Resource> resourcesToMoveIterator = getChangedResources().iterator();
			
			while(resourcesToMoveIterator.hasNext()) {
				
				Resource resource = resourcesToMoveIterator.next();
				
				if (resource != null) {
					
					//Change the URI: 
					IFile file = EMFUtil.getWorkspaceFile(resource);
					
					resource.setURI(getChangedResourceURI(resource));
					
					if (file != null) {
						
						files.add(file);
						
						undoMap.put(resource, file);
					}
				}
				
				pm.worked(1);
			}
			
			//Save the resource(s) to be saved:
			Iterator<Resource> resourcesToSaveIterator = getSavedResources().iterator();
			
			while(resourcesToSaveIterator.hasNext()) {
				
				Resource resource = resourcesToSaveIterator.next();
				
				try {
					
					EMFUtil.save(resource);
					EMFUtil.getWorkspaceFile(resource).refreshLocal(0, pm);
				} 
				catch (Exception e) {
					
					UiPlugin.logError(e);
					
					//In the event of an error during saving, remove the original resource from list of resource to be deleted:
					Object fileToKeep = undoMap.get(resource);
					
					if (fileToKeep != null) {
						files.remove(fileToKeep); 
					}
				}
				
				pm.worked(1);
			}
			
			//Delete the file(s) to be moved:
			Iterator<IFile> filesIterator = files.iterator();
			
			while(filesIterator.hasNext()) {
				
				IFile localFile = filesIterator.next();
				
				if ((localFile.isAccessible()) && (!localFile.isReadOnly())) {
					localFile.delete(false, new SubProgressMonitor(pm, 1));
				}
			}
			
			undoMap.clear();

			// perform the custom changes
			Iterator<Change> changesIterator = customChange.iterator();
			
			while(changesIterator.hasNext()) {
				try {
					Change change = changesIterator.next();
					change.perform(new SubProgressMonitor(pm, 0));
				} catch (Throwable e) {
					UiPlugin.logError(e);
				}
				pm.worked(1);
			}

		} 
		finally {
			pm.done();
		}		
		
		return null;
	}
}
