/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id$
 * 
 * Contributors: 
 * IBM Corporation - initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.ui.testgen.util;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.hyades.internal.execution.testgen.ui.TestgenUIUtility;
import org.eclipse.hyades.test.core.testgen.ITestgenListener;
import org.eclipse.hyades.test.core.testgen.TestGenerator;
import org.eclipse.hyades.test.core.testgen.TestGenerator2;
import org.eclipse.hyades.test.core.testgen.TestGeneratorFactory;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IWorkbenchWindow;

/**
 * <p>Utility methods for test generation from a recording.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Jeff Nevicosi
 * @version October 23, 2009
 * @since   March 31, 2006
 */
public class TestgenUtil {

	/**
	 * Launches a specific test generator, while initializing default status listeners.
	 * @param IFile recFile - the recording file used in testgen (the generator's input)
	 * @param String testgenFile - the testsuite file (the generator's output)
	 * @param testgenID - the id of the defined TestGenerator extension
	 * @param listeners - additional ITestgenListeners to notify when testgen is complete
	 * @param name - the test generator's name (used as the Job name) 
	 */
	public static void launchGenerator(final IFile recFile, final String testgenFile, final String testgenID, final ITestgenListener[] listeners, String name) {
		Job testGenJob = new Job(name) {
		
		public IStatus run(IProgressMonitor monitor) {
			boolean bSuccess = false;
			IConfigurationElement element = TestGeneratorFactory.getInstance().getGeneratorConfigElement(testgenID);
			if (element == null)
			{
				UiPlugin.logError(NLS.bind(UiPluginResourceBundle.TESTGEN_LOAD_ERR_, testgenID));
				return finish(Status.CANCEL_STATUS);
			}
			final TestGenerator generator;
			try {
				generator = (TestGenerator) element.createExecutableExtension("class");
				generator.initialize(recFile, testgenFile, monitor);
				bSuccess = generator.runTestGen(monitor); 
			}			
			catch (Throwable e)
			{
				e.printStackTrace();
				bSuccess = false;
				return finish(Status.CANCEL_STATUS);
			}
			if (generator != null && bSuccess) {

				//Open the test suite(s):
				if (Display.getDefault() != null){
					
					Display.getDefault().asyncExec( new Runnable() {

						/* (non-Javadoc)
						 * @see java.lang.Runnable#run()
						 */
						public void run(){

							String[] outputFiles = null;

							if (generator instanceof TestGenerator2){
								outputFiles =((TestGenerator2)(generator)).getOutputFiles();						
							}

							else{
								outputFiles = new String[]{generator.getOutputFileName()};
							}

							IWorkbenchWindow activeWorkbenchWindow = UiPlugin.getDefault().getWorkbench().getActiveWorkbenchWindow();
							IWorkspaceRoot workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();

							for (int counter = 0; counter < outputFiles.length; counter++) {

								//Resolve the test suite file:
								IFile outputFile = workspaceRoot.getFile(new Path(outputFiles[counter]));

								//Attempt to select and reveal the test suite in the workbench window's active page:
								TestgenUIUtility.selectAndReveal(outputFile, activeWorkbenchWindow);

								//Attempt to open the test suite in the editor:
								TestUIUtil.openEditor(outputFile, TestUIExtension.GENERIC_TEST_SUITE_EDITOR_PART_ID);
							}								
						}
					});
				}
				return finish(Status.OK_STATUS);
			}
			return finish(Status.CANCEL_STATUS);

		}
		/**
		 * notify listeners of completion
		 * @param status
		 * @return
		 */
		private IStatus finish(IStatus status)
		{
			boolean bSuccess = false;
			if (status.equals(Status.OK_STATUS))
				bSuccess = true;
			if (listeners == null || listeners.length == 0)
				return status;
			synchronized (listeners)
			{
				for (int i = 0; i < listeners.length; i++ )
				{
					ITestgenListener li = listeners[i];
					li.notifyEnd(bSuccess);
				}
			}
			return status;
		}
		};
		testGenJob.setUser(true);
		testGenJob.schedule();
	}

}
