/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestUIUtil.java,v 1.34 2010/05/28 16:05:29 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.util;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.models.common.facades.behavioral.INamedElement;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUI;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.FileProxyManager;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceRegistry;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.reference.ReferenceTypeRegistry;
import org.eclipse.hyades.test.ui.internal.navigator.refactoring.resources.RefactoringMessages;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.EObjectEditorInput;
import org.eclipse.hyades.test.ui.internal.util.TestUIUtilities;
import org.eclipse.hyades.test.ui.navigator.CMNNamedElementProxyNode;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.hyades.ui.util.LocationValidator;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.editors.text.EditorsUI;
import org.eclipse.ui.model.IWorkbenchAdapter;
import org.eclipse.ui.part.FileEditorInput;

/**
 * <p>Test UI utilities.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version May 28, 2010
 * @since   February 1, 2005
 * @provisional As of TPTP V4.5.6, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class TestUIUtil{
	
	/**
	 * Valid (valid resource path) return code for 
	 * {@link #validateResourceName(String, IContainer)}.
	 */
	public static final int VALID_RESOURCE_PATH = 0;
	
	/**
	 * Invalid (invalid resource path length) return code for 
	 * {@link #validateResourceName(String, IContainer)}.
	 * 
	 * @see #isResourcePathLengthValid(String, IContainer)
	 */
	public static final int INVALID_RESOURCE_PATH_LENGTH = 1;
	
	/**
	 * Invalid (invalid resource path) return code for 
	 * {@link #validateResourceName(String, IContainer)}.
	 * 
	 * @see #isResourcePathValid(String, IContainer)
	 */
	public static final int INVALID_RESOURCE_PATH = 2;
	
	/**
	 * Invalid (unavailable resource path) return code for 
	 * {@link #validateResourceName(String, IContainer)}.
	 * 
	 * @see #isResourcePathAvailable(String, IContainer)
	 */
	public static final int UNAVAILABLE_RESOURCE_PATH = 3;
	
	/**
	 * Invalid (invalid logical name length) return code for 
	 * {@link #validateResourceName(String, IContainer)}.
	 * 
	 * @see #isLogicalNameLengthValid(String)
	 */
	public static final int INVALID_LOGICAL_NAME_LENGTH = 4;
	
	/**
	 * Invalid (invalid logical name) return code for 
	 * {@link #validateResourceName(String, IContainer)}.
	 * 
	 * @see #isLogicalNameValid(String)
	 */
	public static final int INVALID_LOGICAL_NAME = 5;
	
	/**
	 * Invalid (unavailable logical name) return code for 
	 * {@link #validateResourceName(String, IContainer)}.
	 * 
	 * @see #isLogicalNameAvailable(String, IContainer)
	 */
	public static final int UNAVAILABLE_LOGICAL_NAME = 6;
	
	/**
	 * The maximum file name length on Windows.
	 */
	public static final int WINDOWS_FILE_NAME_MAX_LENGTH = 250;
	
	/**
	 * If the current platform is Windows.
	 */
	public static final boolean IS_WINDOWS_OS = ("win32".equals(Platform.getWS())); //$NON-NLS-1$
	
	/**
	 * <p>List of reserved/restricted characters for file paths including:</p>
	 * 
	 * <ul>
	 * 	 <li>;</li>
	 *   <li>?</li>
	 *   <li>@</li>
	 *   <li>&</li>
	 *   <li>=</li>
	 *   <li>+</li>
	 *   <li>$</li>
	 *   <li>,</li>
	 *   <li><</li>
	 *   <li>></li>
	 *   <li>#</li>
	 *   <li>%</li>
	 *   <li>"</li>
	 *   <li>{</li>
	 *   <li>}</li>
	 *   <li>|</li>
	 *   <li>^</li>
	 *   <li>[</li>
	 *   <li>]</li>
	 *   <li>'</li>
	 *   <li>!</li>
	 *   <li>*</li>
	 *   <li>(</li>
	 *   <li>)</li>
	 * </ul>
	 */
	private static final char[] RESERVED_FILE_PATH_CHARS = {';', '?', '@', '&' , '=', '+', '$', ',', '<', '>', '#', '%', '"', '{', '}', '|', '^', '[', ']', '\'', '!', '*', '(', ')'};
		
	/**
	 * <p>Validates the resource name for the container.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The resource name in the container is valid according to {@link #isResourcePathLengthValid(String, IContainer)} ({@link #INVALID_RESOURCE_PATH_LENGTH} return code).</li>
	 * <li>The resource name in the container is valid according to {@link #isResourcePathValid(String, IContainer)} ({@link #INVALID_RESOURCE_PATH} return code).</li>
	 * <li>The resource name in the container is valid according to {@link #isResourcePathAvailable(String, IContainer)} ({@link #UNAVAILABLE_RESOURCE_PATH} return code).</li>
	 * <li>The resource name (without the file extension (<code>.&ltfile extension&gt;</code>), if exists) is valid according to {@link #isLogicalNameLengthValid(String)} ({@link #INVALID_LOGICAL_NAME_LENGTH} return code).</li>
	 * <li>The resource name (without the file extension (<code>.&ltfile extension&gt;</code>), if exists) is valid according to {@link #isLogicalNameValid(String)} ({@link #INVALID_LOGICAL_NAME} return code).</li>
	 * <li>The resource name (without the file extension (<code>.&ltfile extension&gt;</code>), if exists) in the container is valid according to {@link #isLogicalNameAvailable(String, IContainer)} ({@link #UNAVAILABLE_LOGICAL_NAME} return code).</li>
	 * </ul>
	 * 
	 * <p>To validate a project name, the project name is the <code>resourceName</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
	 * 
	 * @param  resourceName The resource name.
	 * @param  container The container.
	 * @return {@link #VALID_RESOURCE_PATH} if all of the validity conditions are satisfied (see above), otherwise {@link #INVALID_RESOURCE_PATH_LENGTH}, {@link #INVALID_RESOURCE_PATH}, {@link #UNAVAILABLE_RESOURCE_PATH}, {@link #INVALID_LOGICAL_NAME_LENGTH}, {@link #INVALID_LOGICAL_NAME}, or {@link #UNAVAILABLE_LOGICAL_NAME}.
	 * @see    #isResourcePathLengthValid(String, IContainer) 
	 * @see    #isResourcePathValid(String, IContainer) 
	 * @see    #isResourcePathAvailable(String, IContainer) 
	 * @see    #isLogicalNameLengthValid(String) 
	 * @see    #isLogicalNameValid(String) 
	 * @see    #isLogicalNameAvailable(String, IContainer) 
	 */
	public static int validateResourcePath(String resourceName, IContainer container){
		
		if (!isResourcePathLengthValid(resourceName, container)) {
			return INVALID_RESOURCE_PATH_LENGTH;
		}

		if(!isResourcePathValid(resourceName, container)){
			return INVALID_RESOURCE_PATH;
		}

		if (!isResourcePathAvailable(resourceName, container)) {
			return UNAVAILABLE_RESOURCE_PATH;
		}
		
		String logicalName = resourceName;
		
		//Remove the .<file extension>, if exists:
		if ((resourceName != null) && (resourceName.length() > 0)){
			
			int dotIndex = resourceName.lastIndexOf('.');
			
			if (dotIndex != -1) {
				logicalName = resourceName.substring(0, dotIndex);
			}
		}			
			
		if (!isLogicalNameLengthValid(logicalName)) {
			return INVALID_LOGICAL_NAME_LENGTH;
		}

		if(!isLogicalNameValid(logicalName)){
			return INVALID_LOGICAL_NAME;
		}

		if (!isLogicalNameAvailable(logicalName, container)) {
			return UNAVAILABLE_LOGICAL_NAME;
		}
				
		return VALID_RESOURCE_PATH;
	}
	
	/**
	 * <p>Determines if the length of the path to the resource name for the container is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>The length of the path to the resource name in the container is valid according to {@link #isResourcePathLengthValid(String)}.</li>
	 * </ul>
	 * 
	 * <p>To validate a project name, the project name is the <code>resourceName</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
	 * 
	 * @param  resourceName The resource name.
	 * @param  container The container.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #isResourcePathLengthValid(String) 
	 */
	public static boolean isResourcePathLengthValid(String resourceName, IContainer container) {

		//Validate parameters:
		if ((resourceName != null) && (resourceName.trim().length() > 0) && (container != null)){			
			return (isResourcePathLengthValid(container.getFullPath().append(resourceName).toOSString()));
		}
		
		return false;		
	}
	
	/**
	 * <p>Determines if the length of the resource path is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The resource path is not <code>null</code>, empty, or whitespace.</li>
	 * <li>On Windows, the resource path length is less or equal to 250 characters (see {@link #WINDOWS_FILE_NAME_MAX_LENGTH}).</li>
	 * </ul>
	 * 
	 * @param  resourcePath The resource path.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #WINDOWS_FILE_NAME_MAX_LENGTH
	 */
	public static boolean isResourcePathLengthValid(String resourcePath) {

		//Validate parameters:
		if ((resourcePath != null) && (resourcePath.trim().length() > 0)){
			
			//Windows path length check:
			if (IS_WINDOWS_OS){
				return (resourcePath.length() <= WINDOWS_FILE_NAME_MAX_LENGTH);
			}

			return true;
		}
		
		return false;		
	}
	
	/**
	 * <p>Determines if the path to the resource name for the container is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>The path to the resource name in the container is valid according to {@link #isFilePathValid(String)}.</li>
	 * <li>The path to the resource name in the container is valid according to {@link Path#isValidPath(String)}.</li>
	 * <li>The path to the resource name in the container is valid according to {@link IWorkspace#validatePath(String, int)}.  
	 * The type is {@link IResource#FOLDER} <code>|</code> {@link IResource#FILE} if the container path has more than one segment,
	 * otherwise {@link IResource#PROJECT}.</li>
	 * </ul>
	 * 
	 * <p>To validate a project name, the project name is the <code>resourceName</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
	 * 
	 * @param  resourceName The resource name.
	 * @param  container The container.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #isFilePathValid(String) 
	 * @see    Path#isValidPath(String)
	 * @see    IWorkspace#validatePath(String, int)
	 */
	public static boolean isResourcePathValid(String resourceName, IContainer container) {
		
		//Validate parameters:
		if ((resourceName != null) && (resourceName.trim().length() > 0) && (container != null)){
			
			IPath resourceFullPath = container.getFullPath();
			int resourceType = IResource.PROJECT;

			if(resourceFullPath.segmentCount() > 0){
				resourceType = (IResource.FOLDER | IResource.FILE);
			}
			
			String resourcePath = resourceFullPath.append(resourceName).toOSString();

			return ((isFilePathValid(resourcePath)) && (Path.ROOT.isValidPath(resourcePath)) && (ResourcesPlugin.getWorkspace().validatePath(resourcePath, resourceType).isOK()));
		}
		
		return false;
	}
	
	/**
	 * <p>Determines if the path to the resource name for the container is available.</p>
	 *  
	 * <p>Availability is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>There are no other resources (with and without the file extension) in the container with the same resource name (case insensitive on Windows).</li>
	 * </ul>
	 * 
	 * <p>To validate a project name, the project name is the <code>resourceName</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
	 * 
	 * @param  resourceName The resource name.
	 * @param  container The container.
	 * @return <code>true</code> if all of the availability conditions are satisfied (see above), otherwise <code>false</code>.
	 */
	public static boolean isResourcePathAvailable(String resourceName, IContainer container) {
		
		//Validate parameters:
		if ((resourceName != null) && (resourceName.trim().length() > 0) && (container != null)){
		
			try {
				
				IResource[] children = container.members();
				
				for (int counter = 0; counter < children.length; counter++) {
					
						//Note: The child resource could be a project, folder, or file.
						String name = children[counter].getName();					
						String nameWithoutExtension = name;
						
						if(children[counter] instanceof IFile){

							String extension = children[counter].getFileExtension();

							if(extension != null){
								nameWithoutExtension = name.substring(0, (name.length() - extension.length() - 1));
							}
						}
						
						if (IS_WINDOWS_OS){

							if ((name.equalsIgnoreCase(resourceName)) || (nameWithoutExtension.equalsIgnoreCase(resourceName))) {
								return false;
							}					
						} 
						else if ((name.equals(resourceName)) || (nameWithoutExtension.equals(resourceName))) {
							return false;
						}
				}
				
				return true;
			} 
			catch (CoreException e) {
				//Ignore and return false.
			}
		}
		
		return false;
	}
	
	/**
	 * <p>Determines if the length of the logical name is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The logical name is not <code>null</code>, empty, or whitespace.</li>
	 * </ul>
	 * 
	 * <p>Note: The logical does not contain a file extension (<code>.&ltfile extension&gt;</code>).
	 * 
	 * @param  logicalName The logical name.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 */
	public static boolean isLogicalNameLengthValid(String logicalName) {

		//Validate parameters:
		if ((logicalName != null) && (logicalName.trim().length() > 0)){
			return true;
		}
		
		return false;
	}

	/**
	 * <p>Determines if the logical name is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The logical name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The logical name is valid according to {@link LocationValidator#validateEMFSegment(String)}.</li>
	 * <li>The logical name does not start or end with any of the following characters:
	 * <ul>
	 *   <li>-</li>
	 *   <li>[space]</li>
	 *   <li>.</li>
	 * </ul>
	 * <li>The logical name does not contain any of the following characters:
	 * <ul>
	 *   <li>:</li>
	 *   <li>\</li>
	 *   <li>/</li>
	 * </ul>
	 * </ul>
	 * 
	 * <p>Note: The logical does not contain a file extension (<code>.&ltfile extension&gt;</code>).
	 * 
	 * @param  logicalName The logical name.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    LocationValidator#validateEMFSegment(String)
	 */
	public static boolean isLogicalNameValid(String logicalName){
		
		//Validate parameters:
		if ((logicalName != null) && (logicalName.trim().length() > 0)){
			return ((LocationValidator.validateEMFSegment(logicalName) == null) &&
					(!logicalName.startsWith("-")) &&
					(!logicalName.endsWith("-")) &&
					(!logicalName.startsWith(" ")) &&
					(!logicalName.endsWith(" ")) &&
					(!logicalName.startsWith(".")) &&
					(!logicalName.endsWith(".")) &&				
					(logicalName.indexOf(':') == -1) && 
					(logicalName.indexOf('\\') == -1) && 
					(logicalName.indexOf('/') == -1)); 				
			}
		
		return false;
	}
	
	/**
	 * <p>Determines if the logical name is available.</p>
	 *  
	 * <p>Availability is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The logical name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>There are no other resources in the container that resolve to proxies with the same logical name (case insensitive on Windows).</li>
	 * </ul>
	 * 
	 * <p>Note: The logical does not contain a file extension (<code>.&ltfile extension&gt;</code>).
	 * 
	 * @param  logicalName The logical name.
	 * @param  container The container.
	 * @return <code>true</code> if all of the availability conditions are satisfied (see above), otherwise <code>false</code>.
	 */
	public static boolean isLogicalNameAvailable(String logicalName, IContainer container) {
		
		//Validate parameters:
		if ((logicalName != null) && (logicalName.trim().length() > 0) && (container != null)){

			try {
				
				//Resolve the file proxy manager:
				IFileProxyManager fileProxyManager = TestNavigator.getFileProxyManager();

				if (fileProxyManager == null) {					
					fileProxyManager = new FileProxyManager();
				}
				
				IResource[] children = container.members();

				for (int counter = 0; counter < children.length; counter++) {

					if(children[counter] instanceof IFile){
						
						IProxyNode proxy = fileProxyManager.getProxy(((IFile)(children[counter])), null); 
						
						if ((proxy != null) && (proxy instanceof CMNNamedElementProxyNode)) {

							if (IS_WINDOWS_OS){

								if (((CMNNamedElementProxyNode)(proxy)).getName().equalsIgnoreCase(logicalName)) {
									return false;
								}					
							} 
							else if (((CMNNamedElementProxyNode)(proxy)).getName().equals(logicalName)) {
								return false;
							}
						}			
					}
				}

				return true;
			} 
			catch (CoreException e) {
				//Ignore and return false.
			}
		}

		return false;
	}

	/**
	 * <p>Determines if the file path is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The file path is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The file path does not contain any of the following characters (see {@link #RESERVED_FILE_PATH_CHARS}):
	 * <ul>
	 * 	 <li>;</li>
	 *   <li>?</li>
	 *   <li>@</li>
	 *   <li>&</li>
	 *   <li>=</li>
	 *   <li>+</li>
	 *   <li>$</li>
	 *   <li>,</li>
	 *   <li><</li>
	 *   <li>></li>
	 *   <li>#</li>
	 *   <li>%</li>
	 *   <li>"</li>
	 *   <li>{</li>
	 *   <li>}</li>
	 *   <li>|</li>
	 *   <li>^</li>
	 *   <li>[</li>
	 *   <li>]</li>
	 *   <li>'</li>
	 *   <li>!</li>
	 *   <li>*</li>
	 *   <li>(</li>
	 *   <li>)</li>
	 * </ul>
	 * </li>
	 * </ul>
	 * 
	 * @param  filePath The file path.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.	 
	 * @see    #RESERVED_FILE_PATH_CHARS
	 * */
	public static boolean isFilePathValid(String filePath){
		
		if ((filePath != null) && (filePath.trim().length() > 0)){
		
			//Check for reserved/restricted characters for file paths:
			for (int i = 0; i < RESERVED_FILE_PATH_CHARS.length; i++){
				
				if (filePath.indexOf(RESERVED_FILE_PATH_CHARS[i]) !=  -1){
					return false;
				}
			}

			return true;
		}
		
		return false;
	}

	/**
	 * <p>Resolves the label for the {@link Object}.</p>
	 * 
	 * <p>The label is resolved using the following rules:</p>
	 * 
	 * <ul>
	 * <li>If the {@link Object} is adaptable to a {@link IWorkbenchAdapter}, the {@link IWorkbenchAdapter#getLabel(Object)} is returned.</li>
	 * <li>Otherwise, if the {@link Object} is an instance of {@link INamedElement}, the {@link INamedElement#getName()} is returned.</li>
	 * <li>Otherwise, {@link String#valueOf(Object)} is returned.</li>
	 * </ul>
	 * 
	 * @param  object The {@link Object} to resolve to a label.
	 * @return The label for the {@link Object}.
	 * @see    IWorkbenchAdapter#getLabel(Object)
	 * @see    INamedElement#getName()
	 * @see    String#valueOf(Object)
	 */
	public static String getLabel(Object object){
		
		if (object instanceof IAdaptable){
			
			IWorkbenchAdapter workbenchAdapter = ((IWorkbenchAdapter)(((IAdaptable)(object)).getAdapter(IWorkbenchAdapter.class)));
			
			if (workbenchAdapter != null){
				return (workbenchAdapter.getLabel(object));
			}
		}

		if (object instanceof INamedElement){
			return (((INamedElement)(object)).getName());
		}

		return (String.valueOf(object));
	}
	
	/**
	 * <p>Opens, potentially activates, and returns the editor for the specified {@link Resource} 
	 * based on the specified editor identifier.</p>
	 * 
	 * <p>If the resource exists in the Eclipse workspace, the editor is opened by 
	 * invoking {@link UIUtil#openEditor(IFile, String, boolean)}.  Otherwise, if the 
	 * resource exists outside the Eclipse workspace, the editor is opened by invoking 
	 * {@link IWorkbenchPage#openEditor(IEditorInput, String, boolean)} </p>
	 * 
	 * <p>If editor identifier is <code>null</code>, empty, or all whitespace, the identifier for the 
	 * default editor identifier is used.  If the default editor identifier can not be resolved, 
	 * the default text editor identifier is used ({@link EditorsUI#DEFAULT_TEXT_EDITOR_ID}).</p>
	 * 
	 * <p>If the editor cannot be opened, this method logs the exception in the Eclipse 
	 * <code>.log</code> file and returns <code>null</code>.</p>  
	 * 
	 * <p>Note, this method does NOT use the {@link TestUI#OPEN_EDITOR} user preference.
	 * 
	 * <p>Note: This method MUST be called from the UI thread.</p>
	 * 
	 * @param resource The {@link Resource} to be opened in the editor.
	 * @param editorId The editor identifier.
	 * @param activate <code>true</code> if the editor will be activated, otherwise <code>false</code>.
	 * @return The opened and potentially activated {@link IEditorPart}, otherwise <code>null</code>.
	 * @see   UIUtil#openEditor(IFile, String, boolean)
	 * @see   IWorkbenchPage#openEditor(IEditorInput, String, boolean)
	 * @see   EditorsUI#DEFAULT_TEXT_EDITOR_ID
	 */
	public static IEditorPart openEditor(Resource resource, String editorId, boolean activate){
		
		IWorkbench workbench = PlatformUI.getWorkbench();
		IFile file = EMFUtil.getWorkspaceFile(resource);
		
		if ((file != null) && (file.exists())){
				
			if ((editorId == null) || (editorId.trim().length() == 0)){

				IEditorDescriptor descriptor = workbench.getEditorRegistry().getDefaultEditor(file.getName());

				if (descriptor != null){
					editorId = descriptor.getId();
				}
				else{
					editorId = EditorsUI.DEFAULT_TEXT_EDITOR_ID;
				}
			}
			
			return (UIUtil.openEditor(file, editorId, false));
		}
		else{

			File ioFile = new File(EMFUtil.getFilePath(resource));
			
			if(ioFile.exists()){
				
				if ((editorId == null) || (editorId.trim().length() == 0)){
	
					IEditorDescriptor descriptor = workbench.getEditorRegistry().getDefaultEditor(ioFile.getName());
	
					if (descriptor != null){
						editorId = descriptor.getId();
					}
					else{
						editorId = EditorsUI.DEFAULT_TEXT_EDITOR_ID;
					}
				}
				
				try {
					
					if(workbench.getActiveWorkbenchWindow() != null){
						return (workbench.getActiveWorkbenchWindow().getActivePage().openEditor(new RCPFileEditorInput(ioFile), editorId, activate));
					}
					else if(workbench.getWorkbenchWindows().length > 0){
						return (workbench.getWorkbenchWindows()[0].getActivePage().openEditor(new RCPFileEditorInput(ioFile), editorId, activate));
					}
				} 
				catch (PartInitException e) {
					UiPlugin.logError(e);
				}
			}
		}
		
		return null;
	}
	
	/**
	 * <p>Opens the editor for the specified {@link EObject} based on the specified 
	 * editor identifier and the {@link TestUI#OPEN_EDITOR} user preference.</p>
	 * 
	 * <p>If the underlying workspace resource for the specified {@link EObject} 
	 * is <code>null</code> or does not exist, the {@link TestUI#OPEN_EDITOR} 
	 * user preference is set to {@link MessageDialogWithToggle#NEVER}, or the active editor in the active 
	 * workbench page is open on the specified {@link IEditorInput}, this method gracefully exits.  If the 
	 * {@link TestUI#OPEN_EDITOR} user preference is set to {@link MessageDialogWithToggle#PROMPT}, the user 
	 * to prompted to open the editor.  If the user elects to open the editor (and possibly save the preference) 
	 * or the {@link TestUI#OPEN_EDITOR} user preference is set to {@link MessageDialogWithToggle#ALWAYS}, the 
	 * editor is opened and brought to front.</p>
	 * 
	 * <p>If the editor cannot be opened, this method logs the exception in the Eclipse 
	 * <code>.log</code> file and gracefully exits.</p>  
	 * 
	 * <p>Note: This method MUST be called from the UI thread.</p>
	 * 
	 * <p>Note: Since the editor input created from the {@link EObject} is not persisted 
	 * due to the life cycle of the underlying {@link EObject}, it is only available in 
	 * the navigation history while the Eclipse workbench is open.</p>
	 * 
	 * @param eObject The {@link EObject} to be opened in the editor.
	 * @param editorId The non-<code>null</code> editor identifier.
	 * @see   #openEditor(IEditorInput, String)
	 * @see   EObjectEditorInput
	 * @see   TestUI#OPEN_EDITOR
	 */
	public static void openEditor(EObject eObject, String editorId) {
				
		try {				
			openEditor(new EObjectEditorInput(eObject), editorId);				
		} 
		catch (IllegalArgumentException i) {
			UiPlugin.logError(i);
		}
	}

	/**
	 * <p>Opens the editor for the specified {@link Resource} based on the specified 
	 * editor identifier and the {@link TestUI#OPEN_EDITOR} user preference.</p>
	 * 
	 * <p>If the underlying workspace resource for the specified {@link Resource} 
	 * is <code>null</code> or does not exist, the {@link TestUI#OPEN_EDITOR} 
	 * user preference is set to {@link MessageDialogWithToggle#NEVER}, or the active editor in the active 
	 * workbench page is open on the specified {@link IEditorInput}, this method gracefully exits.  If the 
	 * {@link TestUI#OPEN_EDITOR} user preference is set to {@link MessageDialogWithToggle#PROMPT}, the user 
	 * to prompted to open the editor.  If the user elects to open the editor (and possibly save the preference) 
	 * or the {@link TestUI#OPEN_EDITOR} user preference is set to {@link MessageDialogWithToggle#ALWAYS}, the 
	 * editor is opened and brought to front.</p>
	 * 
	 * <p>If the editor cannot be opened, this method logs the exception in the Eclipse 
	 * <code>.log</code> file and gracefully exits.</p>  
	 * 
	 * <p>Note: This method MUST be called from the UI thread.</p>
	 * 
	 * @param resource The {@link Resource} to be opened in the editor.
	 * @param editorId The non-<code>null</code> editor identifier.
	 * @see   #openEditor(IFile, String)
	 * @see   TestUI#OPEN_EDITOR
	 */
	public static void openEditor(Resource resource, String editorId) {
		openEditor(EMFUtil.getWorkspaceFile(resource), editorId);		
	}
	
	/**
	 * <p>Opens the editor for the specified {@link IFile} based on the specified 
	 * editor identifier and the {@link TestUI#OPEN_EDITOR} user preference.</p>
	 * 
	 * <p>If the specified {@link IFile} is <code>null</code> or does not exist, the {@link TestUI#OPEN_EDITOR} 
	 * user preference is set to {@link MessageDialogWithToggle#NEVER}, or the active editor in the active 
	 * workbench page is open on the specified {@link IEditorInput}, this method gracefully exits.  If the 
	 * {@link TestUI#OPEN_EDITOR} user preference is set to {@link MessageDialogWithToggle#PROMPT}, the user 
	 * to prompted to open the editor.  If the user elects to open the editor (and possibly save the preference) 
	 * or the {@link TestUI#OPEN_EDITOR} user preference is set to {@link MessageDialogWithToggle#ALWAYS}, the 
	 * editor is opened and brought to front.</p>

	 * <p>If the editor cannot be opened, this method logs the exception in the Eclipse 
	 * <code>.log</code> file and gracefully exits.</p>  
	 * 
	 * <p>Note: This method MUST be called from the UI thread.</p>
	 * 
	 * @param file The {@link IFile} to be opened in the editor.
	 * @param editorId The non-<code>null</code> editor identifier.
	 * @see   TestUI#OPEN_EDITOR
	 * @see   TestUI#OPEN_EDITOR
	 */
	public static void openEditor(IFile file, String editorId) {
		
		if ((file != null) && (file.exists())){
			openEditor(new FileEditorInput(file), editorId);
		}
	}

	/**
	 * <p>Opens the editor for the specified {@link IEditorInput} based on the specified 
	 * editor identifier and the {@link TestUI#OPEN_EDITOR} user preference.</p>
	 * 
	 * <p>If the {@link TestUI#OPEN_EDITOR} user preference is set to {@link MessageDialogWithToggle#NEVER}
	 * or the active editor in the active workbench page is open on the specified {@link IEditorInput},
	 * this method gracefully exits.  If the {@link TestUI#OPEN_EDITOR} user preference is set to 
	 * {@link MessageDialogWithToggle#PROMPT}, the user to prompted to open the editor.  If the user
	 * elects to open the editor (and possibly save the preference) or the {@link TestUI#OPEN_EDITOR} 
	 * user preference is set to {@link MessageDialogWithToggle#ALWAYS}, the editor is opened and brought 
	 * to front.</p>
	 * 
	 * <p>If the editor cannot be opened, this method logs the exception in the Eclipse 
	 * <code>.log</code> file and gracefully exits.</p>  
	 * 
	 * <p>Note: This method MUST be called from the UI thread.</p>
	 * 
	 * @param editorInput The {@link IEditorInput} to be opened in the editor.
	 * @param editorId The non-<code>null</code> editor identifier.
	 * @see   UIUtil#openEditor(IEditorInput, String)
	 * @see   TestUI#OPEN_EDITOR
	 */
	public static void openEditor(final IEditorInput editorInput, final String editorId) {

		final IPreferenceStore preferenceStore = UiPlugin.getInstance().getPreferenceStore();
		
		final String openEditorPreferenceValue =  preferenceStore.getString(TestUI.OPEN_EDITOR);

		if (!openEditorPreferenceValue.equals(MessageDialogWithToggle.NEVER)){
		
			Display.getDefault().syncExec(new Runnable() {
	        	
				public void run() {
	        	
					try{
	    		 		
						IWorkbenchPage activeWorkbenchPage = UIUtil.getActiveWorkbenchPage();

						//Only prompt the user for the open editor preference if the active editor was not opened with the editor input:
						if((activeWorkbenchPage != null) && (activeWorkbenchPage.getActiveEditor() != null) && (activeWorkbenchPage.getActiveEditor().getEditorInput().equals(editorInput))){
							return;
						}
						
						//Default to true since the preference value is either 'prompt' or 'always':
						boolean openEditor = true;
						
						if (openEditorPreferenceValue.equals(MessageDialogWithToggle.PROMPT)){
							
							MessageDialogWithToggle messageDialog = MessageDialogWithToggle.openYesNoQuestion(PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(), 
													UiPluginResourceBundle.OPEN_EDITOR_CONFIRM_DIALOG_TITLE, 
													UiPluginResourceBundle.OPEN_EDITOR_CONFIRM_DIALOG_MESSAGE,
													null, false, preferenceStore, TestUI.OPEN_EDITOR);
							
							openEditor = (messageDialog.getReturnCode() == IDialogConstants.YES_ID);
						}

						if(openEditor){
							UIUtil.openEditor(editorInput, editorId);
						}
	        		} 
					catch (Exception e) {
						UiPlugin.logError(e);
	        		}
	        	}
	        });
		}		
	}

	/**
	 * <p>Resolves the map of parameter test asset {@link IFile}s (keys) mapped to lists of  
	 * test asset {@link IFile} references (values).</p>
	 *
	 * <p>References can be implicit or explicit (see the <code>org.eclipse.hyades.test.ui.testNavigatorReferenceTypes</code> extension
	 * point).</p>
	 * 
	 * <p>If parameter test asset {@link IFile} list is <code>null</code> or empty, <code>null</code> is returned.</p>
	 * 
	 * <p>Note: This method creates proxies for all test assets below the 
	 * workspace root.  As such, this method <i>may</i> be long running and should be 
	 * controlled via a {@link IProgressMonitor}.</a>  
	 * 
	 * @param files The list of test asset {@link IFile}s.
	 * @param isExplicit <code>true</code> to resolve explicit references, otherwise <code>false</code>.
	 * @param progressMonitor The progress monitor, otherwise {@link NullProgressMonitor} or <code>null</code>.
	 * @return The map of parameter test asset {@link IFile}s (keys) mapped to lists of test asset {@link IFile} references (values).
	 * @throws OperationCanceledException If the non-<code>null</code> progress monitor is canceled.
	 */
	public static Map<IFile, List<IFile>> getReferences(List<IFile> files, boolean isExplicit, IProgressMonitor progressMonitor) throws OperationCanceledException {
		
		if ((files != null) && (files.size() > 0)) {

			if(progressMonitor == null){
				progressMonitor = new NullProgressMonitor();
			}
			
			try{

				progressMonitor.beginTask(RefactoringMessages.TestUIUtil_SEARCHING_REFERENCES_TASK, (files.size() + 1));

				//Create proxies for all test assets below the workspace root:
				TestUIUtilities.createProxyNodes(ResourcesPlugin.getWorkspace().getRoot(), new SubProgressMonitor(progressMonitor, 1));

				Map<IFile, List<IFile>> references = new HashMap<IFile, List<IFile>>(files.size());						
				Iterator<IFile> filesIterator = files.iterator();

				while (filesIterator.hasNext()) {

					if(progressMonitor.isCanceled()){
						throw new OperationCanceledException();
					}

					IFile file = filesIterator.next();

					if(file != null){

						List<IFile> fileReferences = new ArrayList<IFile>();
						Iterator<String> referenceTypesIterator = ReferenceRegistry.getInstance().getReferenceTypes(file).iterator();

						while(referenceTypesIterator.hasNext()) {			

							String referenceType = referenceTypesIterator.next();

							if (ReferenceTypeRegistry.getInstance().isExplicit(referenceType) == isExplicit) {

								Iterator<IProxyNode> referencesIterator = ReferenceRegistry.getInstance().getReferences(file, referenceType).iterator();

								while(referencesIterator.hasNext()) {	

									IFile childFile = ((IFile)(referencesIterator.next().getUnderlyingResource()));

									if (childFile != null) {
										fileReferences.add(childFile);
									}
								}
							}
						}

						references.put(file, fileReferences);
					}

					progressMonitor.worked(1);
				}

				return references;
			} 
			finally {
				progressMonitor.done();
			}
		}
		
		return null;
	}
	
	/**
	 * <p>Resolves the list of test asset {@link IFile}s that are explicitly referenced by the 
	 * parameter test asset {@link IFile}.</p>
	 * 
	 * <p>If parameter test asset {@link IFile} is <code>null</code>, <code>null</code> is returned.</p>
	 * 
	 * <p>Note: This method creates proxies for all test assets below the 
	 * workspace root.  As such, this method <i>may</i> be long running.</a>  
	 * 
	 * @param file The test asset {@link IFile}.
	 * @return The list of test asset {@link IFile}s that are explicitly referenced by the parameter test asset {@link IFile}, otherwise <code>null</code>.
	 * @deprecated As of TPTP V4.7.0, use {@link #getReferences(List, boolan, IProgressMonitor)}.
	 */
	public static List<IFile> getExplicitDependencies(IFile file) {
		
		if (file != null) {

			List<IFile> files = new ArrayList<IFile>(1);
			files.add(file);
			
			try {
				
				Map<IFile, List<IFile>> explicitFileReferences = getReferences(files, true, new NullProgressMonitor());
				
				if(explicitFileReferences != null){
					return (explicitFileReferences.get(file));
				}
			} 
			catch (OperationCanceledException o) {
				//Ignore and return null.
			}
		}
		
		return null;
	}
	
	/**
	 * <p>Determines if the length of the path to the resource name for the container is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>The length of the path to the resource name in the container is valid according to {@link #isResourcePathLengthValid(String)}.</li>
	 * </ul>
	 * 
	 * <p>To validate a project name, the project name is the <code>resourceName</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
	 * 
	 * @param  resourceName The resource name.
	 * @param  container The container.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #isResourcePathLengthValid(String, IContainer)
	 * @see    #isResourcePathLengthValid(String) 
	 * @deprecated As of TPTP V4.6.2, use {@link #isResourcePathLengthValid(String, IContainer)}.
	 */
	public static boolean isResourceNameLengthValid(String resourceName, IContainer container) {
		return (isResourcePathLengthValid(resourceName, container));
	}
	
	/**
	 * <p>Determines if the length of the path to the new resource name associated with the resource is valid.</p>
	 *  
	 * <p>If the resource is a {@link IFile}, the new new resource name is considered to be a sibling.  That is, its
	 * container is the same the resource's container.  If the resource is a {@link IContainer}, the new new resource 
	 * name is considered to be a child.  That is, its container is the resource.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>The length of the path to the resource name in the container is valid according to {@link #isResourcePathLengthValid(String)}.</li>
	 * </ul>
	 * 
	 * @param  resource The resource.
	 * @param  newResourceName The new resource name.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #isResourcePathLengthValid(String, IContainer)
	 * @see    #isResourcePathLengthValid(String) 
	 * @deprecated As of TPTP V4.6.2, use {@link #isResourcePathLengthValid(String, IContainer)}.
	 */
	public static boolean isResourceNameLengthValid(IResource resource, String newResourceName) {	

		//Validate parameters:
		if ((resource != null) && (newResourceName != null) && (newResourceName.trim().length() > 0)){

			if (resource instanceof IFile) {				
				return (isResourcePathLengthValid(newResourceName, ((IFile)(resource)).getParent()));
			}
			else if (resource instanceof IContainer) {
				return (isResourcePathLengthValid(newResourceName, ((IContainer)(resource))));
			}
		}

		return false;		
	 }
	
	/**
	 * <p>Determines if the length of the paths to the list of resource names for the container is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>Each resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>The length of the path to each of resource names in the container is valid according to {@link #isResourcePathLengthValid(String)}.</li>
	 * </ul>
	 * 
	 * <p>To validate a list of project names, the project names are the <code>resourceNameList</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
	 * 
	 * @param  resourceNameList The list of resource names.
	 * @param  container The container.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #isResourcePathLengthValid(String, IContainer)
	 * @see    #isResourcePathLengthValid(String) 
	 * @deprecated As of TPTP V4.6.2, use {@link #isResourcePathLengthValid(String, IContainer)}.
	 */
	public static boolean isResourceNameListLengthValid(List<String> resourceNameList, IContainer container) {

		//Validate parameters:
		if ((resourceNameList != null) && (container != null)) {

			//Resource name length check:
			Iterator<String> resourceNameListIterator = resourceNameList.iterator();

			while((resourceNameListIterator.hasNext())) {

				if (!isResourcePathLengthValid(resourceNameListIterator.next(), container)) {
					return false;			
				}
			}
			
			return true;
		}

		return false;
	}
	
	/**
	 * <p>Determines if the path to the resource name for the container is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>The path to the resource name in the container is valid according to {@link #isFilePathValid(String)}.</li>
	 * <li>The path to the resource name in the container is valid according to {@link Path#isValidPath(String)}.</li>
	 * <li>The path to the resource name in the container is valid according to {@link IWorkspace#validatePath(String, int)}.</li>
	 * </ul>
	 * 
	 * <p>To validate a project name, the project name is the <code>resourceName</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
	 * 
	 * @param  resourceName The resource name.
	 * @param  container The container.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #isResourcePathValid(String, IContainer)
	 * @see    #isFilePathValid(String) 
	 * @see    Path#isValidPath(String)
	 * @see    IWorkspace#validatePath(String, int)
	 * @deprecated As of TPTP V4.6.2, use {@link #isResourcePathValid(String, IContainer)}.
	 */
	public static boolean isResourceNameValid(String resourceName, IContainer container){	
		return (isResourcePathValid(resourceName, container));
	}
	
	/**
	 * <p>Determines if the path to the list of resource names for the container is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>Each resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>The path to each of the resource names in the container is valid according to {@link #isFilePathValid(String)}.</li>
	 * <li>The path to each of the resource names in the container is valid according to {@link Path#isValidPath(String)}.</li>
	 * <li>The path to each of the resource names in the container is valid according to {@link IWorkspace#validatePath(String, int)}.</li>
	 * </ul>
	 * 
	 * <p>To validate a list of project names, the project names are the <code>resourceNameList</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
     *
	 * @param  resourceNameList The list of resource names.
	 * @param  container The container.
	 * @return <code>true</code> if all of the validity conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #isResourcePathValid(String, IContainer)
	 * @see    #isFilePathValid(String) 
	 * @see    Path#isValidPath(String)
	 * @see    IWorkspace#validatePath(String, int)
	 * @deprecated As of TPTP V4.6.2, use {@link #isResourcePathValid(String, IContainer)}.
	 */
	public static boolean isResourceNameListValid(List<String> resourceNameList, IContainer container) {
		
		//Validate parameters:
		if ((resourceNameList != null) && (container != null)) {

			//Resource name length check:
			Iterator<String> resourceNameListIterator = resourceNameList.iterator();

			while((resourceNameListIterator.hasNext())) {

				if (!isResourcePathValid(resourceNameListIterator.next(), container)) {
					return false;			
				}
			}
			
			return true;
		}

		return false;		
	}
	
	/**
	 * <p>Determines if the path to the resource name for the container is available.</p>
	 *  
	 * <p>Availability is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>There are no other resources (with and without the file extension) in the container with the same resource name (case insensitive on Windows).</li>
	 * </ul>
	 * 
	 * <p>To validate a project name, the project name is the <code>resourceName</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
	 * 
	 * @param  resourceName The resource name.
	 * @param  container The container.
	 * @return <code>true</code> if all of the availability conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #isResourcePathAvailable(String, IContainer)
	 * @deprecated As of TPTP V4.6.2, use {@link #isResourcePathAvailable(String, IContainer)}.
	 */
	public static boolean isResourceNameAvailable(String resourceName, IContainer container) {
		return (isResourcePathAvailable(resourceName, container));
	}
	
	/**
	 * <p>Determines if the path to the list of resource names for the container is available.</p>
	 *  
	 * <p>Availability is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>Each resource name is not <code>null</code>, empty, or whitespace.</li>
	 * <li>The container is not <code>null</code>.</li>
	 * <li>There are no other resources (with and without the file extension) in the container with the same resource name (case insensitive on Windows).</li>
	 * </ul>
	 * 
	 * <p>To validate a list of project names, the project names are the <code>resourceNameList</code> and the 
	 * workspace root ({@link ResourcesPlugin.getWorkspace().getRoot()}) is the <code>container</code>.</p>
	 * 
	 * @param  resourceNameList The list of resource names.
	 * @param  container The container.
	 * @return <code>true</code> if all of the availability conditions are satisfied (see above), otherwise <code>false</code>.
	 * @see    #isResourcePathAvailable(String, IContainer)
	 * @deprecated As of TPTP V4.6.2, use {@link #isResourcePathAvailable(String, IContainer)}.
	 */
	public static boolean isResourceNameListAvailable(List<String> resourceNameList, IContainer container) {

		//Validate parameters:
		if ((resourceNameList != null) && (container != null)) {

			//Resource name length check:
			Iterator<String> resourceNameListIterator = resourceNameList.iterator();

			while((resourceNameListIterator.hasNext())) {

				if (!isResourcePathAvailable(resourceNameListIterator.next(), container)) {
					return false;			
				}
			}
			
			return true;
		}

		return false;
	}
	
	/**
	 * Resolves a unique name across the list of {@link CMNNamedElement}s based on a base name.
	 * 
	 * @param baseName The base name of the unique name.
	 * @param namedElementList The list of {@link CMNNamedElement}s to ensure uniqueness.
	 * @return The unique name across the list of CMNNamedElement based on a base name.
	 * @deprecated As of TPTP V4.6.2, use <code>org.eclipse.hyades.test.tools.core.common.util.TestCommonUtil#getUniqueName(String, List)</code>.
	 */
	public static String getUniqueName(String baseName, List<CMNNamedElement> namedElementList){
		
		if((namedElementList != null) && (namedElementList.size() > 0)){

			int initialNumber = (namedElementList.size() + 1);
			String name = baseName.concat(" ").concat(Integer.toString(initialNumber));
			Iterator<CMNNamedElement> namedElementListIterator = namedElementList.iterator();
			
			while (namedElementListIterator.hasNext()) {
				
				CMNNamedElement cmnNamedElement = namedElementListIterator.next();
				
				if(name.equals(cmnNamedElement.getName())){
					
					name = baseName.concat(" ").concat(Integer.toString(++initialNumber));
					
					namedElementListIterator = namedElementList.iterator();
				}	
			}
			
			return name;
		}

		return baseName;
	}
	
	/**
	 * Resolves the file with the file extension.</p>
	 * 
	 * <p>If the file name already contains the file extension, the file name will be returned.</p> 
	 * 
	 * @param fileName The file name.
	 * @param fileExtension The file extension.
	 * @return correct The file name with the file extension.
	 * @deprecated As of TPTP V4.6.2, use {@link String#endsWith(String)}, that is <br/><code>if(fileName.endsWith("." + fileExtension)){<br/>&nbsp;fileName += ("." + fileExtension);<br/>}</code>.
	 */
	public static String validateExtension(String fileName, String fileExtension) {

		if(!fileName.endsWith("." + fileExtension)){
			return (fileName + "." + fileExtension);
		}
		
		return fileName;
	}
}