/*******************************************************************************
 * Copyright (c) 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestLocationPage.java,v 1.2 2010/05/05 20:42:47 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.wizard;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.test.ui.TestUIConstants;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.navigator.TestNavigatorFilter;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.tptp.platform.common.ui.wizard.LocationPage;

/**
 * <p>Extension of the {@link LocationPage} wizard location page for specifying test files.</p>
 * 
 * <p>In addition:</p>
 * 
 * <ul>
 * <li>Applies the {@link TestNavigatorFilter} to the tree of workspace resources (see {@link #addFilter(org.eclipse.jface.viewers.IFilter)}).</li>
 * <li>Displays the {@link TestUIConstants#NEW_TEST_WIZARD_CATEGORY} category of new project wizards (see {@link #setNewProjectWizardCategory(String)}).</li>
 * </ul>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @version May 5, 2010
 * @since   May 4, 2010
 * @see     LocationPage
 * @provisional As of TPTP V4.7.0, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
 */
public class TestLocationPage extends LocationPage {
	
	public TestLocationPage(String pageName, IStructuredSelection selection) {	      

		super(pageName, selection);
		
		addFilter(new TestNavigatorFilter());
		
		setNewProjectWizardCategory(TestUIConstants.NEW_TEST_WIZARD_CATEGORY);
	}

	/**
	 * <p>Determines if the page's controls contain valid values.</p>
	 * 
	 * <p>Valid values require all of the following conditions (in order):</p>
	 * 
	 * <ol>
	 * <li>{@link LocationPage#validatePage(boolean)}</li>
	 * <li>The container is valid:
	 * <ol>
	 * <li>{@link TestUIUtil#isFilePathValid(String)}</li>
	 * </ol>
	 * </li>
	 * <li>The file path is valid:
	 * <ol>
	 * <li>{@link TestUIUtil#validateResourcePath(String, IContainer)}</li>
	 * </ol>
	 * </li>
	 * </ol>
	 * 
	 * <p>If the values are not valid, an error message is displayed if the page's message is to be 
	 * updated and the method returns false.</p>
	 * 
	 * @param updateMessage <code>true</code> if the page's message is to be updated, otherwise <code>false</code>.
	 * @return <code>true</code> if the page's controls contain valid values, otherwise <code>false</code>.
	 * @see LocationPage#validatePage(boolean)
	 */
	protected boolean validatePage(boolean updateMessage){
		
		//Case 1: Container and/or file name are missing or invalid:
		if(!super.validatePage(updateMessage)){
			return false;
		}

		//Case 2: Container is invalid:
		if(!TestUIUtil.isFilePathValid(getContainerFullPath().toString())){
			
			if(updateMessage){

				setMessage(null);
				setErrorMessage(UiPluginResourceBundle.TestLocationPage_INVALID_CONTAINER);
			}
			
			return false;
		}

		//Case 3: File name is invalid (invalid resource and/or logical names/lengths):
		//Note: The file name is valid if the resource and/or logical names are unavailable to
		//support selecting and overwriting existing files.
		IFile file = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(getFilePath()));

		switch(TestUIUtil.validateResourcePath(file.getName(), file.getParent())) {

		case TestUIUtil.INVALID_RESOURCE_PATH:
		case TestUIUtil.INVALID_LOGICAL_NAME:

			if(updateMessage){

				setMessage(null);
				setErrorMessage(UiPluginResourceBundle.TestLocationPage_INVALID_NAME);
			}

			return false;

		case TestUIUtil.INVALID_RESOURCE_PATH_LENGTH:
		case TestUIUtil.INVALID_LOGICAL_NAME_LENGTH:

			if(updateMessage){

				setMessage(null);
				setErrorMessage(UiPluginResourceBundle.TestLocationPage_TOO_LONG_NAME);
			}

			return false;
		}
		
		//Case 5 (default): Valid container and file name:
		if(updateMessage){

			setMessage(null);
			setErrorMessage(null);
		}
		
		return true;
	}
}