/*******************************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id$
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.ui.wizard;

import java.util.Iterator;

import org.eclipse.core.resources.IFile;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.facades.behavioral.impl.HyadesFactory;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.core.util.JavaUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.tptp.platform.common.ui.wizard.LocationPage;
import org.eclipse.ui.PartInitException;

/**
 * <p>Extension of the {@link HyadesNewWizard} wizard for creating test suites.</p>
 * 
 * <p>This class is intended to be subclassed by clients.</p>
 * 
 * <p>By default the wizard contains only two pages:</p>
 * 
 * <ul>
 * <li>a location page to specify the location of the new test suite,</li>
 * <li>a name and description page.</li>
 * </ul>
 * 
 * <p>At least, clients need to provide the type of the test suite by implementing the {@link #getType()} method.</p> 
 * 
 * 
 * @author  Patrick Nedelic
 * @author  Jerome Gout
 * @author  Tony Wang
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version May 4, 2010
 * @since   March 9, 2006
 */
public abstract class TestSuiteNewWizard2 extends HyadesNewWizard {
	
	private TestLocationPage testLocationPage = null;
	
	/**
	 * Constructor for TestSuiteNewWizard
	 */
	public TestSuiteNewWizard2()
	{
		super();
		setWindowTitle(UiPluginResourceBundle.WIZ_TST_SUITE_TTL); 
		setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_NEWTESTSUITE));

		IDialogSettings dialogSettings = UiPlugin.getInstance().getDialogSettings();
		IDialogSettings settings = dialogSettings.getSection("TestSuiteNewWizard"); //$NON-NLS-1$
		if(settings == null) {
			settings = dialogSettings.addNewSection("TestSuiteNewWizard"); //$NON-NLS-1$
        }
		setDialogSettings(settings);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard#getLocationPage()
	 */
	protected LocationPage getLocationPage() {
		
		if(testLocationPage == null){

			testLocationPage = new TestLocationPage("testLocation", selection); //$NON-NLS-1$

			String fileExtension = getFileExtension();
			
			if (fileExtension != null) {
				
				testLocationPage.setFileExtension(fileExtension);
				testLocationPage.setFileName(""); //$NON-NLS-1$
			}
		}

		return testLocationPage;
	}
	
	/**
	 * Returns the type for the test suite created by this wizard.
	 * @return the type for the test suite created by this wizard.
	 */
    abstract protected String getType();
	
	/**
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard#getFileExtension()
	 */
	protected String getFileExtension()
	{
		return ICommonConstants.TEST_SUITE_FILE_EXTENSION;
	}

	/**
	 * @see org.eclipse.jface.wizard.IWizard#addPages()
	 */
	public void addPages()
	{
		getLocationPage().setTitle(UiPluginResourceBundle.WIZ_TST_SUITE_GEN_PG_TTL); 
		getLocationPage().setDescription(UiPluginResourceBundle.WIZ_TST_SUITE_GEN_PG_LOC_DSC); 
		addPage(getLocationPage());
		
		getAttributeWizardPage().setTitle(UiPluginResourceBundle.WIZ_TST_SUITE_GEN_PG_TTL); 
		getAttributeWizardPage().setDescription(UiPluginResourceBundle.WIZ_TST_SUITE_GEN_PG_ATT_DSC); 
		addPage(getAttributeWizardPage());
	}
	
	/**
	 * @see org.eclipse.hyades.ui.test.internal.wizard.HyadesNewWizard#createObject(org.eclipse.core.resources.IFile)
	 */
	protected boolean createObject(IFile file) throws Exception
	{
        Resource resource = createResource(file);
        createObject(resource);
        for (Iterator i = resource.getResourceSet().getResources().iterator(); i.hasNext();)
            EMFUtil.save((Resource)i.next());
        return true;
	}
	
    /**
     * Creates the Object setting the attributes based on the information
     * the user has defined in this wizard's pages.
     * <br> In case of overriding this method, extenders need to call <code>ajustTetsSuite()</code> 
     * in order to restore the ability to modify the content of the test suite just before saving it.
     * @param resource
     */
    protected void createObject(Resource resource)
    {
        ITestSuite testSuite = HyadesFactory.INSTANCE.createTestSuite(resource);
        testSuite.setType(getType());
        testSuite.setName(getLocationPage().getFileNameWithoutExtension());
        
        String description = getAttributeWizardPage().getItemDescription();
        if((description != null) && (description.trim().length() > 0))
            testSuite.setDescription(description);
            
        if(testSuite.getImplementor() == null)
            HyadesFactory.INSTANCE.createImplementor(testSuite, false);
        
        adjustTestSuite(testSuite);    
    }
    
    /**
     * Clients can use this hook to do some initializations of the newly created test suite.<br>
     * This method is called just before saving the test suite on the file system.
     * @param testsuite the new test suite created by the wizard.
     */
	protected void adjustTestSuite(ITestSuite testSuite) {
        String clsName = JavaUtil.getValidClassName(testSuite.getName(), true);
        if(clsName == null) {
            clsName = "Test";
        }
        testSuite.getImplementor().setResource("test." + clsName);
    }

    /**
	 * Creates the EMF resource in which the object is added to.
	 * @return a not null Resource
	 */
	protected Resource createResource(IFile file)
	{
        ResourceSet resourceSet = new ResourceSetImpl();
        URI uri = URI.createPlatformResourceURI(file.getFullPath().toString(), false);
        return resourceSet.createResource(uri);
	}

		
	/**
	 * @see org.eclipse.hyades.ui.internal.wizard.HyadesNewWizard#openEditor(org.eclipse.core.resources.IFile)
	 */
	protected void openEditor(IFile file) 
	throws PartInitException
	{
		//Attempt to open the editor:
		TestUIUtil.openEditor(file, TestUIExtension.GENERIC_TEST_SUITE_EDITOR_PART_ID);		
	}	
}
