/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NewRecordingWizardPage.java,v 1.20 2009/08/01 00:50:54 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.tptp.test.internal.recorder.ui.wizards;

import java.net.URL;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.hyades.execution.recorder.RecorderFactory;
import org.eclipse.hyades.test.core.testgen.TestGeneratorFactory;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.ContextIds;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseAdapter;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.tptp.test.provisional.recorder.ui.wizards.IGenericRecorderWizard;
import org.eclipse.tptp.test.provisional.recorder.ui.wizards.IRecorderPageProvider;
import org.eclipse.ui.PlatformUI;

/**
 * <p>The first page of the Generic Recorder Framework (GRF) wizard.</p>
 *
 * <p>Allows users to create a new test from a new recording (e.g. "Launch" a
 * Recorder) or from an existing recording (aka Generate Test).</p>
 * 
 * 
 * @author  Jeffery Nevicosi
 * @author  Paul E. Slauenwhite
 * @version July 31, 2009
 * @since   September 23, 2005
 */
public class NewRecordingWizardPage extends WizardPage implements
		IPrimaryRecorderPage {

	private IGenericRecorderWizard wizard;

	private Button btnNewRec = null;

	private Button btnExistRec = null;

	private Button btnAdvanced = null;

	private Label lblRec = null;

	private Label lblTestgen = null;

	private Combo cmbTestgen1 = null;

	private Table listRecorders = null;

	private String[] registeredRecorders;

	private List<IConfigurationElement> registeredTestGeneratorConfigurationElements = null;

	private boolean bShowAdvanced = false;

	private Label lblDesc = null;

	private Set imagesToDispose = new HashSet();

	private static final String KEY_IS_NEW_REC = "GenRec_IsNewRec"; //$NON-NLS-1$

	private static final String KEY_RECORDER_ID = "GenRec_RecID"; //$NON-NLS-1$

	private static final String KEY_TESTGEN_ID = "GenRec_TestgenID"; //$NON-NLS-1$

	public NewRecordingWizardPage(IGenericRecorderWizard wizard) {
		super(NewRecordingWizardPage.class.getName(), UiPluginResourceBundle.NewRecordingWizardPage_TITLE, null); // TODO:
																			// image
																			// descriptor
		this.wizard = wizard;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		
		//Set the context sensitive help:
		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, UiPlugin.getID() + ContextIds.NEW_TEST_FROM_RECORDING_WIZARD);		
		
		setPageComplete(false);
		setDescription(UiPluginResourceBundle.NewRecordingWizardPage_DESCRIPTION);
		Composite composite = new Composite(parent, SWT.NULL);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(new GridData(GridData.VERTICAL_ALIGN_FILL
				| GridData.HORIZONTAL_ALIGN_FILL));
		setControl(composite);

		btnNewRec = new Button(composite, SWT.RADIO);
		btnNewRec.setText(UiPluginResourceBundle.NewRecordingWizardPage_BTN_NEW_REC);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		btnNewRec.setLayoutData(gd);

		btnExistRec = new Button(composite, SWT.RADIO);
		btnExistRec.setText(UiPluginResourceBundle.NewRecordingWizardPage_BTN_EXIST_REC);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		btnExistRec.setLayoutData(gd);

		btnNewRec.addSelectionListener(new SelectionAdapter() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				wizard.notifyRecorderChanged();
				setPageComplete(validatePage());
				super.widgetSelected(e);

			}
		});

		btnExistRec.addSelectionListener(new SelectionAdapter() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				wizard.notifyRecorderChanged();
				setPageComplete(validatePage());
				super.widgetSelected(e);

			}
		});

		layoutRecorders(composite);

		layoutTestgen(composite);
		redrawControl();

		doDefaultEvents();
	}

	private void layoutRecorders(Composite composite)
	{
		lblRec = new Label(composite, SWT.NULL);
		lblRec.setText(UiPluginResourceBundle.NewRecordingWizardPage_SELECT_RECORDER);
		listRecorders = new Table(composite, SWT.BORDER | SWT.FULL_SELECTION);
		GridData gd = new GridData(GridData.FILL_BOTH);
		listRecorders.setLayoutData(gd);
		listRecorders.addSelectionListener(new SelectionAdapter() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */			
			public void widgetSelected(SelectionEvent e) {
				recorderChanged();
				
			}			
		});

		listRecorders.addMouseListener(new MouseAdapter() {			
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.swt.events.MouseAdapter#mouseDoubleClick(org.eclipse.swt.events.MouseEvent)
			 */
			public void mouseDoubleClick(MouseEvent event) {
				// TODO: next
		        if (getSelectedRecorder()!= null) {
		    		if (canFlipToNextPage())
		    			getContainer().showPage(getNextPage());
		        }
			}
		});
		addRegisteredRecorders();
		
		lblDesc = new Label(composite, SWT.WRAP);		
		lblDesc.setVisible(false);
		lblDesc.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
	}	

	private void recorderChanged() {
		int nSelectedRec = listRecorders.getSelectionIndex();
		wizard.notifyRecorderChanged();
		populateRegisteredTestgenNew();
		updateDescriptionText();
		if (nSelectedRec > -1 && isTestgenValid()
				&& (btnExistRec.getSelection() || btnNewRec.getSelection())) {
			setPageComplete(true);
		} else
			setPageComplete(false);
	}

	private void layoutTestgen(Composite composite) {
		btnAdvanced = new Button(composite, SWT.NULL);
		btnAdvanced.setText(UiPluginResourceBundle.showAdvanced);

		btnAdvanced.addSelectionListener(new SelectionAdapter() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent e) {
				bShowAdvanced = !bShowAdvanced;
				redrawControl();
				super.widgetSelected(e);
			}
		});

		lblTestgen = new Label(composite, SWT.NULL);
		lblTestgen.setText(UiPluginResourceBundle.NewRecordingWizardPage_SELECT_TESTGEN);
		cmbTestgen1 = new Combo(composite, SWT.READ_ONLY);
		cmbTestgen1.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

	}

	public String getSelectedRecorder() {
		if (listRecorders.getSelectionIndex() > -1)
			return registeredRecorders[listRecorders.getSelectionIndex()];
		else
			return "";

	}

	public boolean isNewRecording() {
		return btnNewRec.getSelection();
	}

	private void doDefaultEvents() {
		IPreferenceStore store = UiPlugin.getDefault().getPreferenceStore();
		store.setDefault(KEY_IS_NEW_REC, true);

		boolean bNew = store.getBoolean(KEY_IS_NEW_REC);
		btnNewRec.setSelection(bNew);
		btnExistRec.setSelection(!bNew);

		if (selectRecorder(store.getString(KEY_RECORDER_ID))) { // listeners not
																// enabled until
																// after
																// createControl(),
																// so do it
																// manually
			recorderChanged();
		}

		selectGenerator(store.getString(KEY_TESTGEN_ID));

	}

	private boolean selectRecorder(String id) {
		if (id == null || id.length() < 1)
			return false;

		for (int i = 0; i < registeredRecorders.length; i++) {
			if (id.equals(registeredRecorders[i])) {
				listRecorders.deselectAll();
				listRecorders.select(i);
				return true;
			}
		}
		return false;
	}

	private boolean selectGenerator(String id) {
		
		if ((id != null) && (id.trim().length() > 0)){

			for (int counter = 0; counter < registeredTestGeneratorConfigurationElements.size(); counter++) {
				
				if (id.equals(registeredTestGeneratorConfigurationElements.get(counter).getAttribute("id"))) { //$NON-NLS-1$
					
					cmbTestgen1.select(counter);
					
					return true;
				}
			}
		}
		
		return false;
	}

	private void addRegisteredRecorders() {
		listRecorders.removeAll();
		imagesToDispose.clear();
		RecorderFactory factory = RecorderFactory.getInstance();
		String[] ids = factory.getRegisteredRecorderID();
		//load default image
		Image defaultImage = TestUIImages.INSTANCE.getImage(TestUIImages.IMG_GEN_RECORD);
		
		registeredRecorders = new String[ids.length];
		for (int i = 0; i < ids.length; i++) {
			// listRecorders.add(RecorderFactory.getInstance().getRecorderName(ids[i]));
			TableItem item = new TableItem(listRecorders, SWT.NULL);
			item.setText(factory.getRecorderName(ids[i]));
			// check for image
			URL url = factory.getRecorderIconURL(ids[i]);
			Image image = null;
			if (url != null)
				image = ImageDescriptor.createFromURL(url).createImage();
			if (image == null)
				image = defaultImage;
			item.setImage(image);
			registeredRecorders[i] = ids[i];
			if (image != defaultImage)  //bugzilla_137554
				imagesToDispose.add(image);
		}
		if (listRecorders.getSelectionIndex() > -1)
			listRecorders.select(listRecorders.getSelectionIndex());
		if (ids.length == 0) {
			listRecorders.setEnabled(false);
			lblRec.setEnabled(false);
			setMessage(
					UiPluginResourceBundle.NewRecordingWizardPage_NO_RECORDERS_FOUND,
					WizardPage.ERROR);
			setPageComplete(false);
		}

	}

	private void populateRegisteredTestgenNew() {
		
		//Remove the registered test generators from the combo box:
		cmbTestgen1.removeAll();

		//Resolve the registered test generators for the selected recorder:
		registeredTestGeneratorConfigurationElements = TestGeneratorFactory.getInstance().getGeneratorConfigurationElementsByRecorderID(getSelectedRecorder());
		
		if (registeredTestGeneratorConfigurationElements.size() > 0) {

			//Add the registered test generators to the combo box:
			for (int counter = 0; counter < registeredTestGeneratorConfigurationElements.size(); counter++) {
				cmbTestgen1.add(((IConfigurationElement)(registeredTestGeneratorConfigurationElements.get(counter))).getAttribute("name"));
			}
			
			cmbTestgen1.select(0);
		}
	}

	protected void updateDescriptionText() {
		String desc = RecorderFactory.getInstance().getRecorderDescription(
				getSelectedRecorder());
		if (desc == null || desc.equals("")) {
			lblDesc.setVisible(false);
			lblDesc.setText("");
			return;
		}
		
		lblDesc.setText(NLS.bind(UiPluginResourceBundle.NewRecordingWizardPage_DESCRIPTION_LBL, desc));
		lblDesc.setVisible(true);
		
		//Layout the control to handle the case where the new description spans a different number of lines:
		((Composite)(getControl())).layout(true);
	}

	protected void redrawControl() {
		if (bShowAdvanced) {
			btnAdvanced.setText(UiPluginResourceBundle.hideAdvanced);
			lblTestgen.setVisible(true);
			cmbTestgen1.setVisible(true);
		} else {
			btnAdvanced.setText(UiPluginResourceBundle.showAdvanced);
			cmbTestgen1.setVisible(false);
			lblTestgen.setVisible(false);
		}

		populateRegisteredTestgenNew();

	}

	/**
	 * validate whether proper testgen options are selected
	 * 
	 * @return true if all options are valid
	 */
	protected boolean isTestgenValid() {
		return (cmbTestgen1.getSelectionIndex() > -1);
	}

	protected boolean validatePage() {
		return ((btnExistRec.getSelection() || btnNewRec.getSelection())
				&& isTestgenValid() && listRecorders.getSelectionIndex() > -1);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.tptp.test.provisional.recorder.ui.wizards.IGenericRecorderPage#getRecorderData(java.lang.String)
	 */
	public String getRecorderData(String key) {
		
		if (key.equals(IRecorderPageProvider.WIZARD_KEY_TESTGEN_ID)) {
			
			if (cmbTestgen1.getSelectionIndex() == -1){
				return "-1"; //$NON-NLS-1$
			}

			return (registeredTestGeneratorConfigurationElements.get(cmbTestgen1.getSelectionIndex()).getAttribute("id")); //$NON-NLS-1$
		}
		
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.tptp.test.provisional.recorder.ui.wizards.IGenericRecorderPage#getRecorderObjectData(java.lang.String)
	 */
	public Object getRecorderObjectData(String key) {
		// not supported
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.tptp.test.internal.recorder.ui.wizards.IGenericRecorderPage#saveSettings()
	 */
	public void saveSettings() {
		IPreferenceStore store = UiPlugin.getDefault().getPreferenceStore();
		store.setValue(KEY_IS_NEW_REC, btnNewRec.getSelection());
		store.setValue(KEY_RECORDER_ID, getSelectedRecorder());
		store.setValue(KEY_TESTGEN_ID,
				getRecorderData(IRecorderPageProvider.WIZARD_KEY_TESTGEN_ID));
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.jface.dialogs.DialogPage#dispose()
	 */
	public void dispose() {
		for (Iterator it = imagesToDispose.iterator(); it.hasNext();) {
			((Image) it.next()).dispose();
		}
		super.dispose();
	}
}
