/*******************************************************************************
 * Copyright (c) 2003, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IDatapool.java,v 1.3 2006/02/08 19:59:53 slavescu Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.execution.runtime.datapool;

/**
 * This class defines the primary interface to datapools.  A datapool
 * is an ordered collection of data.  The structure of data is in the
 * form of uniformly sized records that are collected into logical groups.
 * <p>
 * <code>datapool --> (0..n) EquivalenceClasses --> (0..n) Records</code>
 * <p>
 * Each record contains a uniform set of {@link IDatapoolVariable variables}
 * (logically columns) with appropriate name and id properties associated 
 * with each.
 *
 * @see	IDatapoolEquivalenceClass
 * @see	IDatapoolRecord
 * @see	IDatapoolVariable
 * @see DatapoolException
 */
public interface IDatapool extends INamedElement
{
	
	//************************************************************
	// These methods provide support for "variables" (logically 
	// columns) in the datapool.
	//************************************************************
	
	/**
	 * The number of variables available in each record.
	 * 
	 * @return	The number of data variables available in each record.
	 */
	public int getVariableCount();
	
	/**
	 * The variable at a specified zero based column index.
	 * A {@link DatapoolException DatapoolException} is thrown if the
	 * specified index is not valid.
	 * 
	 * @param	variableIndex		A zero based variable index.
	 * @return	The variable for the specified zero based index.
	 * 
	 * @see #getVariableIndex(String)
	 */
	public IDatapoolVariable getVariable(int variableIndex);
	
	/**
	 * Locates the appropriate index associated with a specific
	 * {@link IDatapoolVariable variable} name.  Users will depend
	 * on this method to determine the appropriate cell to access
	 * within a record given a current record and a variable name.
	 * The <i>current record</i> is typically made available from
	 * an {@link IDatapoolIterator iterator} that is performing an
	 * ordered traversal of the records in a datapool.
	 * 
	 * @param	variableName	The name of the variable associated 
	 *							with the desired cell.
	 * @return	The variable index associated with the specified 
	 *			variableName.
	 * 
	 * @see	#getVariableIndexById(String)
	 */
	public int getVariableIndex(String variableName);
	
	/**
	 * Locates the appropriate index associated with a specific
	 * {@link IDatapoolVariable variable} ID.  Users may use
	 * this method to determine the appropriate cell to access
	 * within a record given a current record and a variable ID.
	 * 
	 * @param	variableId	The ID of the variable associated 
	 *						with the desired cell.
	 * @return	The variable index associated with the specified 
	 *			variableId.
	 * 
	 * @see	#getVariableIndex(String)
	 */
	public int getVariableIndexById(String variableId);
	
	
	//************************************************************
	// These methods provide support for the logical equivalence classes
	// associated with an instance of a datapool.
	//************************************************************
	
	/**
	 * Returns the number of {@link IDatapoolEquivalenceClass equivalence classes}
	 * associated with a datapool.
	 * 
	 * @return	The number of equivalence classes associated with a datapool.
	 */
	public int getEquivalenceClassCount();
	
	/**
	 * Accessor for the {@link IDatapoolEquivalenceClass equivalence class} at the specified
	 * zero based index in the datapool.  If an invalid index is supplied a 
	 * {@link DatapoolException} with an appropriate message is thrown.
	 * 
	 * @return The zero-based index for the desired equivalence class.
	 */
	public IDatapoolEquivalenceClass getEquivalenceClass(int index);
	
	/**
	 * Accessor for the default {@link IDatapoolEquivalenceClass equivalence class}
	 * in the datapool.  If the default equivalence class has not been defined then
	 * a value of <b>-1</b> is returned.
	 * 
	 * @return The zero-based index for the default equivalence class.
	 */
	public int getDefaultEquivalenceClassIndex();
	
	/**
	 * Locate an {@link IDatapoolEquivalenceClass equivalence class} by name 
	 * within an instance of a datapool.
	 * 
	 * @return The zero-based index of the equivalence class with the supplied
	 *			equivalenceClassName.
	 * 
	 * @see	#getEquivalenceClassIndexById(String)
	 */
	public int getEquivalenceClassIndex(String equivalenceClassName);
	
	/**
	 * Locate an {@link IDatapoolEquivalenceClass equivalence class} by ID 
	 * within an instance of a datapool.
	 * 
	 * @return The zero-based index of the equivalence class with the supplied
	 *			equivalenceClassId.
	 * 
	 * @see	#getEquivalenceClassIndex(String)
	 */
	public int getEquivalenceClassIndexById(String equivalenceClassId);
	
	
}
