/**********************************************************************
 * Copyright (c) 2003, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ExecutionResultData.java,v 1.7 2010/01/15 15:11:33 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.loaders.common;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.InternalEObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.fragments.BVRCombinedFragment;
import org.eclipse.hyades.models.common.fragments.Common_Behavior_FragmentsFactory;
import org.eclipse.hyades.models.common.interactions.BVRExecutionOccurrence;
import org.eclipse.hyades.models.common.interactions.Common_Behavior_InteractionsFactory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionEvent;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionHistory;
import org.eclipse.hyades.models.common.testprofile.TPFExecutionResult;
import org.eclipse.hyades.models.common.testprofile.TPFInvocationEvent;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.models.common.testprofile.impl.Common_TestprofileFactoryImpl;

/**
 * <p>Maintains the information about a given execution result.</p>
 * 
 * 
 * @author  Ashish Mathur
 * @author  Joseph P. Toomey
 * @version January 15, 2010
 * @since   January 25, 2005
 */
public class ExecutionResultData {

	private TPFExecutionResult result = null;
	private TPFInvocationEvent invocationEvent = null;
	private String id = null;

	private TPFTest test = null;
	private ResourceSet			resourceSet;
	private URI					testURI;
	
	/**
	 * Returns the test to which this execution result belongs.
     *
	 * @return TPFTest
	 */
	public TPFTest getTest() {
		
		if (test.eIsProxy()){
			return test;
		}
		
		if(test.eResource().getResourceSet() == null){
		
			EObject eObject = resourceSet.getEObject(testURI, true);
			
			if(eObject instanceof TPFTest){
				test = (TPFTest)eObject;
			}
		}
			
		return test;
	}
	
	/**
	 * Get the ID of this execution result
	 * @return
	 */
	public String getId() {
		return id;
	}

	/**
	 * Set the ID of this execution result
	 * @param id
	 */
	public void setId(String id) {
		this.id = id;
	}

	/**
	 * Get the invocation event that created this execution result
	 * @return
	 */
	public TPFInvocationEvent getInvocationEvent() {
		return invocationEvent;
	}

	/**
	 * Set the invocation event that is responsble for creating this execution result
	 * @param invocationEvent
	 */
	public void setInvocationEvent(TPFInvocationEvent invocationEvent) {
		this.invocationEvent = invocationEvent;
		if (result != null) {
			TPFExecutionHistory history = result.getExecutionHistory();
			if (history != null) {
				this.invocationEvent.setExecutionHistory(history);
			}
		}
	}

	/**
	 * Get the TPFExecutionResult.
	 * @return
	 */
	public TPFExecutionResult getResult() {
		if (result == null) {
			
			// Create a result set for the given Id
			result =
				Common_TestprofileFactoryImpl
					.eINSTANCE
					.createTPFExecutionResult();
			result.setId(id);
			result.setTest(getTest());
			if (invocationEvent != null) {
				// This execution result was created by an invocation event
				result.setInvocationExecutionEvent(invocationEvent);
			}
		}

		// Now create an execution history for the result, if it doesnt
		// already exist
		if (result.getExecutionHistory() == null) {
			TPFExecutionHistory history =
				Common_TestprofileFactoryImpl
					.eINSTANCE
					.createTPFExecutionHistory();
			result.setExecutionHistory(history);
		}

		return result;
	}
	
	/**
	 * Set the Execution Result.
	 * @param result
	 */
	public void setResult(TPFExecutionResult result) {
		this.result = result;
	}
	
	
	/**
	 * Get the Execution History
	 * @return
	 */
	public TPFExecutionHistory getHistory() {
		return getResult().getExecutionHistory();
	}

	/**
	 * Create an ExecutionResultData object for the given suite and ID
	 * @param suite
	 * @param id
	 */
	public ExecutionResultData(TPFTest test, String id) {
		this.test = test;
		this.id = id;
		this.testURI = EcoreUtil.getURI(test);

		if ((test != null) && (!test.eIsProxy())) {
			resourceSet = test.eResource().getResourceSet();
		}
	}
	
	/**
	 * Find the EObject corresponding to the ID within the test.
	 * @param id
	 * @return
	 */
	public EObject getObjectInTest(String id) {
		
		if (test != null) {
			
			Resource resource = test.eResource();
			
			if(resource != null){
				return (resource.getEObject(id));
			}
		}

		return null;
	}
	
	public static TPFExecutionResult getContainingResult( TPFExecutionEvent event )
	{
		TPFExecutionResult result = null;
		
		for ( EObject container = event.eContainer(); container != null;
			  container = container.eContainer() ) {
			if ( container instanceof TPFExecutionResult ) {
				result = (TPFExecutionResult) container;
				break;
			}
		}
		
		return result;
	}
	
	/**
	 * Returns the model element matching an id within a test model.
	 * 
	 * @param test A Test.
	 * @param id An id. This value may be <code>null</code>.
	 * @return An object, or <code>null</code> if id was <code>null</code>.
	 */
	public static EObject getObjectFromTest(TPFTest test, String id){

		if ((id != null) && (test != null)) { 

			Resource resource = test.eResource();

			if(resource != null){
				return (resource.getEObject(id));
			}
		}
		
		return null;
	}

	/**
	 * Returns the model element matching an id within a test model.
	 * 
	 * @param test A Test.
	 * @param id An id. This value may be <code>null</code>.
	 * @return An object, or <code>null</code> if id was <code>null</code>.
	 */
	public static EObject getObjectFromTest(TPFTest test, String id, String type){
		
		EObject testElement = null;
		
		if(type.trim().equals(BVRExecutionOccurrence.class.getSimpleName())) {
			
			testElement = Common_Behavior_InteractionsFactory.eINSTANCE.createBVRExecutionOccurrence();
		
			((InternalEObject)(testElement)).eSetProxyURI(EcoreUtil.getURI(test).trimFragment().appendFragment(id));
		} 
		else if (type.trim().equals(BVRCombinedFragment.class.getSimpleName())) {
			
			testElement = Common_Behavior_FragmentsFactory.eINSTANCE.createBVRCombinedFragment();
			
			((InternalEObject)(testElement)).eSetProxyURI(EcoreUtil.getURI(test).trimFragment().appendFragment(id));
		}
		
		return testElement;
	}
}
